<?php

namespace App\Notifications;

use App\Models\WriterInfraction;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterInfractionReported extends Notification implements ShouldQueue
{
    use Queueable;

    protected $infraction;

    /**
     * Create a new notification instance.
     */
    public function __construct(WriterInfraction $infraction)
    {
        $this->infraction = $infraction;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $severityColors = [
            'warning' => '#fbbf24',
            'minor' => '#f59e0b',
            'major' => '#dc2626',
            'critical' => '#7c2d12'
        ];

        $severityColor = $severityColors[$this->infraction->severity_level] ?? '#6b7280';

        return (new MailMessage)
            ->subject('Writer Infraction Reported - ' . ucfirst($this->infraction->severity_level) . ' Level')
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('A writer infraction has been reported against your account.')
            ->line('**Infraction Details:**')
            ->line('• **Type:** ' . ucwords(str_replace('_', ' ', $this->infraction->infraction_type)))
            ->line('• **Severity:** ' . ucfirst($this->infraction->severity_level))
            ->line('• **Description:** ' . $this->infraction->description)
            ->line('• **Reported:** ' . $this->infraction->reported_at->format('M d, Y \a\t g:i A'))
            ->when($this->infraction->order_id, function ($message) {
                return $message->line('• **Related Order:** #' . $this->infraction->order_id);
            })
            ->line('**Next Steps:**')
            ->line('1. Review the infraction details carefully')
            ->line('2. Submit your response within 7 days if you wish to appeal')
            ->line('3. Contact support if you have questions')
            ->action('View Infraction Details', route('writer.infractions.show', $this->infraction->id))
            ->line('**Appeal Deadline:** ' . $this->infraction->appeal_deadline->format('M d, Y \a\t g:i A'))
            ->line('Please note that this is a serious matter and may affect your account status and future assignments.')
            ->salutation('Best regards, AcademicScribe Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'infraction_id' => $this->infraction->id,
            'infraction_type' => $this->infraction->infraction_type,
            'severity_level' => $this->infraction->severity_level,
            'description' => $this->infraction->description,
            'reported_at' => $this->infraction->reported_at,
            'appeal_deadline' => $this->infraction->appeal_deadline,
            'order_id' => $this->infraction->order_id,
            'type' => 'writer_infraction_reported',
            'message' => 'A ' . $this->infraction->severity_level . ' level infraction has been reported against your account.',
        ];
    }

    /**
     * Get the notification's database type.
     */
    public function toDatabase(object $notifiable): array
    {
        return $this->toArray($notifiable);
    }
} 