<?php

namespace App\Notifications;

use App\Models\User;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterRegistered extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    /**
     * Create a new notification instance.
     *
     * @param User $user
     */
    public function __construct(protected User $user)
    {
        //
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $platformName = config('app.name', 'Our Platform');

        return (new MailMessage)
            ->subject('Welcome to Our Writer Team!')
            ->greeting('Hello ' . $this->user->name . '!')
            ->line('Thank you for registering as a writer on ' . $platformName . '.')
            ->line('Please verify your email address to continue with the writer testing process.')
            ->line('Writer Qualification Process:')
            ->line('1. Email verification (required to access your account)')
            ->line('2. Multiple-choice test (tests your knowledge of writing standards)')
            ->line('3. Essay test (demonstrates your writing abilities)')
            ->line('Both tests must be passed to qualify as a writer on our platform.')
            ->action('Verify Email', url('/verify-email'))
            ->line('If you have any questions, our support team is here to help.')
            ->line('Thank you for choosing ' . $platformName . '!');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Create verify email action
        $verifyAction = $this->createAction(
            'Verify Email',
            url('/verify-email'),
            'GET',
            'primary'
        );

        // Create view writer guidelines action
        $guidelinesAction = $this->createAction(
            'Writer Guidelines',
            url('/writer/resources/guides'),
            'GET',
            'secondary'
        );

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'writer_registered',
            "Welcome to Our Writer Team!",
            "Thank you for registering, {$this->user->name}! Please verify your email to begin the qualification process.",
            'pen-tool',
            'info',
            'high',
            'account',
            [$verifyAction, $guidelinesAction],
            now()->addDays(30), // Expires after 30 days
            [
                'user_id' => $this->user->id,
                'user_name' => $this->user->name,
                'user_email' => $this->user->email,
                'registered_at' => now()->toISOString(),
                'needs_verification' => true,
                'qualification_steps' => [
                    'email_verification' => false,
                    'multiple_choice_test' => false,
                    'essay_test' => false
                ],
            ]
        );
    }
}
