<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterStartedWorkingAdminNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     */
    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('writer_started_working_admin', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $writerName = $this->order->writer ? $this->order->writer->name : 'Unknown Writer';
        $clientName = $this->order->user ? $this->order->user->name : 'Unknown Client';
        $startTime = $this->order->work_started_at ? $this->order->work_started_at->format('M j, Y g:i A') : 'Just now';
        $deadline = $this->order->writer_deadline ? $this->order->writer_deadline->format('M j, Y g:i A') : 'Not set';

        $mailMessage = (new MailMessage)
            ->subject("Writer Started Working - Order #{$this->order->order_number}")
            ->greeting("Hello Admin,")
            ->line("A writer has started working on an assigned order.")
            ->line("Order Details:")
            ->line("- Order #: {$this->order->order_number}")
            ->line("- Title: {$this->order->title}")
            ->line("- Writer: {$writerName}")
            ->line("- Client: {$clientName}")
            ->line("- Work Started: {$startTime}")
            ->line("- Writer Deadline: {$deadline}")
            ->line("- Pages: {$this->order->pages}")
            ->line("- Academic Level: {$this->order->academic_level}")
            ->line("- Subject: {$this->order->subject}");

        $mailMessage->line("The writer is now actively working on this assignment.")
            ->action('View Order Details', url("/admin/orders/{$this->order->id}"))
            ->line('Thank you for your attention to this matter.');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $writerName = $this->order->writer ? $this->order->writer->name : 'Unknown Writer';
        $clientName = $this->order->user ? $this->order->user->name : 'Unknown Client';
        $startTime = $this->order->work_started_at ? $this->order->work_started_at->format('M j, Y g:i A') : 'Just now';

        // Create view order action
        $action = $this->createAction(
            'View Order',
            url("/admin/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        return $this->getDatabaseNotificationData(
            'writer_started_working_admin',
            "Writer Started Working",
            "{$writerName} has started working on order #{$this->order->order_number} for {$clientName}.",
            'play',
            'info',
            'medium',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'writer_name' => $writerName,
                'client_name' => $clientName,
                'work_started_at' => $this->order->work_started_at ? $this->order->work_started_at->toISOString() : now()->toISOString(),
                'writer_deadline' => $this->order->writer_deadline ? $this->order->writer_deadline->toISOString() : null,
                'pages' => $this->order->pages,
                'academic_level' => $this->order->academic_level,
                'subject' => $this->order->subject
            ]
        );
    }
} 