<?php

namespace App\Notifications;

use App\Models\Order;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterStartedWorkingNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     */
    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('writer_started_working', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $writerName = $notifiable->name ?? 'Writer';
        $startTime = $this->order->work_started_at ? $this->order->work_started_at->format('M j, Y g:i A') : 'Just now';
        $deadline = $this->order->writer_deadline ? $this->order->writer_deadline->format('M j, Y g:i A') : 'Not set';
        $timeRemaining = $this->order->writer_deadline ? now()->diffForHumans($this->order->writer_deadline, ['parts' => 2]) : 'No deadline set';

        $mailMessage = (new MailMessage)
            ->subject("Work Started Successfully - Order #{$this->order->order_number}")
            ->greeting("Hello {$writerName},")
            ->line("Great! You have successfully started working on your assigned order.")
            ->line("Order Details:")
            ->line("- Order #: {$this->order->order_number}")
            ->line("- Title: {$this->order->title}")
            ->line("- Work Started: {$startTime}")
            ->line("- Deadline: {$deadline}")
            ->line("- Time Remaining: {$timeRemaining}")
            ->line("- Pages: {$this->order->pages}")
            ->line("- Academic Level: {$this->order->academic_level}")
            ->line("- Subject: {$this->order->subject}");

        $mailMessage->line("You are now actively working on this assignment. Please ensure you submit your work before the deadline.")
            ->action('View Assignment Details', url("/writer/orders/{$this->order->id}"))
            ->line('Good luck with your work!');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $startTime = $this->order->work_started_at ? $this->order->work_started_at->format('M j, Y g:i A') : 'Just now';
        $deadline = $this->order->writer_deadline ? $this->order->writer_deadline->format('M j, Y g:i A') : 'Not set';

        // Create view assignment action
        $action = $this->createAction(
            'View Assignment',
            url("/writer/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        return $this->getDatabaseNotificationData(
            'writer_started_working',
            "Work Started Successfully",
            "You have started working on order #{$this->order->order_number}. Work started at {$startTime}, deadline is {$deadline}.",
            'play',
            'success',
            'medium',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'work_started_at' => $this->order->work_started_at ? $this->order->work_started_at->toISOString() : now()->toISOString(),
                'writer_deadline' => $this->order->writer_deadline ? $this->order->writer_deadline->toISOString() : null,
                'pages' => $this->order->pages,
                'academic_level' => $this->order->academic_level,
                'subject' => $this->order->subject
            ]
        );
    }
} 