<?php

namespace App\Notifications;

use App\Models\User;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterTestResultNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $user;
    protected $score;
    protected $passed;
    protected $correctAnswers;
    protected $totalQuestions;
    protected $isAdminNotification;
    protected $adminName;

    /**
     * Create a new notification instance.
     *
     * @param User $user
     * @param float $score
     * @param bool $passed
     * @param int $correctAnswers
     * @param int $totalQuestions
     * @param bool $isAdminNotification
     * @param string|null $adminName
     */
    public function __construct(
        User $user,
        float $score,
        bool $passed,
        int $correctAnswers,
        int $totalQuestions,
        bool $isAdminNotification = false,
        ?string $adminName = null
    ) {
        $this->user = $user;
        $this->score = $score;
        $this->passed = $passed;
        $this->correctAnswers = $correctAnswers;
        $this->totalQuestions = $totalQuestions;
        $this->isAdminNotification = $isAdminNotification;
        $this->adminName = $adminName;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        if ($this->isAdminNotification) {
            return $this->toAdminMail($notifiable);
        }

        $mailMessage = (new MailMessage)
            ->subject($this->passed ? 'Congratulations! You Passed the Multiple-Choice Test' : 'Multiple-Choice Test Results')
            ->greeting('Hello ' . $this->user->name . ',');

        if ($this->passed) {
            $mailMessage->line('Congratulations! You have successfully passed the multiple-choice qualification test.')
                ->line('Your score: ' . round($this->score, 1) . '% (' . $this->correctAnswers . ' out of ' . $this->totalQuestions . ' correct)')
                ->line('You can now proceed to the essay test, which is the final step in our writer qualification process.')
                ->line('The essay test will assess your writing skills, research abilities, and adherence to academic standards.')
                ->action('Start Essay Test', route('writer.essay.instructions'));
        } else {
            $retryDate = now()->addMonths(3)->format('F j, Y');

            $mailMessage->line('Thank you for taking the multiple-choice qualification test.')
                ->line('Your score: ' . round($this->score, 1) . '% (' . $this->correctAnswers . ' out of ' . $this->totalQuestions . ' correct)')
                ->line('Unfortunately, you did not meet the passing score of 90%.')
                ->line('You can retry the test after 3 months from now (' . $retryDate . ').')
                ->line('We recommend reviewing academic writing standards and citation formats before your next attempt.')
                ->action('View Test Results', route('writer.test.results'));
        }

        return $mailMessage->line('Thank you for your interest in joining our writing team!');
    }

    /**
     * Get the admin mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    private function toAdminMail(object $notifiable): MailMessage
    {
        // Get admin name if available
        $adminName = $notifiable->name ?? $this->adminName ?? 'Admin';

        return (new MailMessage)
            ->subject('Writer Test Completion: ' . $this->user->name)
            ->greeting('Hello ' . $adminName . ',')
            ->line('A writer has completed the multiple-choice qualification test.')
            ->line('Writer: ' . $this->user->name . ' (' . $this->user->email . ')')
            ->line('Score: ' . round($this->score, 1) . '% (' . $this->correctAnswers . ' out of ' . $this->totalQuestions . ' correct)')
            ->line('Result: ' . ($this->passed ? 'Passed' : 'Failed'))
            ->line($this->passed
                ? 'The writer can now proceed to the essay test.'
                : 'The writer can retry after 3 months.')
            ->action('View Test Results', route('admin.writers.tests'))
            ->line('This is an automated notification.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        // Determine actions based on passed status and notification recipient
        $actions = [];

        if ($this->isAdminNotification) {
            // Admin actions
            $actions[] = $this->createAction(
                'View Test Results',
                url(route('admin.writers.tests')),
                'GET',
                'primary'
            );

            if ($this->passed) {
                $actions[] = $this->createAction(
                    'View Writer Profile',
                    url(route('admin.users.show', $this->user->id)),
                    'GET',
                    'secondary'
                );
            }
        } else {
            // Writer actions
            if ($this->passed) {
                $actions[] = $this->createAction(
                    'Start Essay Test',
                    url(route('writer.essay.instructions')),
                    'GET',
                    'primary'
                );

                $actions[] = $this->createAction(
                    'Writer Guidelines',
                    url('/writer/resources/guides'),
                    'GET',
                    'secondary'
                );
            } else {
                $actions[] = $this->createAction(
                    'View Test Results',
                    url(route('writer.test.results')),
                    'GET',
                    'primary'
                );

                $actions[] = $this->createAction(
                    'Study Resources',
                    url('/writer/resources/training'),
                    'GET',
                    'secondary'
                );
            }
        }

        // Determine icon, color, title and message based on passed status and recipient
        $icon = $this->passed ? 'check-circle' : 'x-circle';
        $color = $this->passed ? 'success' : 'warning';

        if ($this->isAdminNotification) {
            $title = 'Writer Test Completion: ' . $this->user->name;
            $message = "Writer {$this->user->name} has " . ($this->passed ? "passed" : "failed") . " the multiple-choice test with a score of " . round($this->score, 1) . "%.";
        } else {
            $title = $this->passed
                ? "Congratulations! Test Passed"
                : "Multiple-Choice Test Results";

            $message = $this->passed
                ? "You've passed the multiple-choice test with a score of " . round($this->score, 1) . "%. You can now proceed to the essay test."
                : "You scored " . round($this->score, 1) . "% on the multiple-choice test. The required passing score is 90%.";
        }

        // Calculate retry date if failed
        $retryDate = $this->passed ? null : now()->addMonths(3)->toISOString();

        // Return standardized notification data
        return $this->getDatabaseNotificationData(
            'test_result',
            $title,
            $message,
            $icon,
            $color,
            'high',
            'qualification',
            $actions,
            now()->addMonths($this->passed ? 3 : 6), // Longer expiration if failed
            [
                'user_id' => $this->user->id,
                'user_name' => $this->user->name,
                'user_email' => $this->user->email,
                'score' => $this->score,
                'passed' => $this->passed,
                'correct_answers' => $this->correctAnswers,
                'total_questions' => $this->totalQuestions,
                'is_admin_notification' => $this->isAdminNotification,
                'test_date' => now()->toISOString(),
                'retry_date' => $retryDate,
                'passing_threshold' => 90,
            ]
        );
    }
}
