<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\OrderFile;
use App\Traits\DatabaseNotificationTrait;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WriterWorkSubmittedNotification extends Notification implements ShouldQueue
{
    use Queueable, DatabaseNotificationTrait;

    protected $order;
    protected $uploadedFiles;
    protected $isLateSubmission;

    /**
     * Create a new notification instance.
     *
     * @param Order $order
     * @param array $uploadedFiles
     * @param bool $isLateSubmission
     */
    public function __construct(Order $order, array $uploadedFiles = [], bool $isLateSubmission = false)
    {
        $this->order = $order;
        $this->uploadedFiles = $uploadedFiles;
        $this->isLateSubmission = $isLateSubmission;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param object $notifiable
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return $notifiable->preferredChannels('writer_work_submitted', ['database', 'mail']);
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param object $notifiable
     * @return MailMessage
     */
    public function toMail(object $notifiable): MailMessage
    {
        $writerName = $notifiable->name ?? 'Writer';
        $submissionTime = $this->order->work_submitted_at ? $this->order->work_submitted_at->format('M j, Y g:i A') : 'Just now';
        $fileCount = count($this->uploadedFiles);
        $deadline = $this->order->writer_deadline ? $this->order->writer_deadline->format('M j, Y g:i A') : 'Not set';

        $mailMessage = (new MailMessage)
            ->subject("Work Submitted Successfully - Order #{$this->order->order_number}")
            ->greeting("Hello {$writerName},")
            ->line("Congratulations! You have successfully submitted your work for review.")
            ->line("Order Details:")
            ->line("- Order #: {$this->order->order_number}")
            ->line("- Title: {$this->order->title}")
            ->line("- Submission Time: {$submissionTime}")
            ->line("- Deadline: {$deadline}")
            ->line("- Files Submitted: {$fileCount} file(s)");

        if (!empty($this->uploadedFiles)) {
            $mailMessage->line("Submitted files:");
            foreach ($this->uploadedFiles as $file) {
                $mailMessage->line("- {$file->file_name} ({$file->file_size} bytes)");
            }
        }

        if ($this->isLateSubmission) {
            $mailMessage->line("⚠️ Note: This submission was made after the deadline. Admin will review and assess any penalties.")
                ->line("Please ensure future submissions are made on time to avoid penalties.");
        } else {
            $mailMessage->line("✅ Your work was submitted on time! Great job meeting the deadline.");
        }

        $mailMessage->line("Our admin team will now review your work and upload it to the client's account.")
            ->line("You'll be notified once the client has reviewed and approved the work.")
            ->action('View Assignment Details', url("/writer/orders/{$this->order->id}"))
            ->line('Thank you for your excellent work!');

        return $mailMessage;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param object $notifiable
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $submissionTime = $this->order->work_submitted_at ? $this->order->work_submitted_at->format('M j, Y g:i A') : 'Just now';
        $fileCount = count($this->uploadedFiles);
        $deadline = $this->order->writer_deadline ? $this->order->writer_deadline->format('M j, Y g:i A') : 'Not set';

        // Create view assignment action
        $action = $this->createAction(
            'View Assignment',
            url("/writer/orders/{$this->order->id}"),
            'GET',
            'primary'
        );

        $statusMessage = $this->isLateSubmission 
            ? "Work submitted successfully (LATE). Admin will review and assess any penalties."
            : "Work submitted successfully and on time! Admin will review and upload to client.";

        return $this->getDatabaseNotificationData(
            'writer_work_submitted',
            "Work Submitted Successfully",
            $statusMessage,
            'upload',
            $this->isLateSubmission ? 'warning' : 'success',
            'high',
            'order',
            [$action],
            null,
            [
                'order_id' => $this->order->id,
                'order_number' => $this->order->order_number,
                'title' => $this->order->title,
                'submission_time' => $this->order->work_submitted_at ? $this->order->work_submitted_at->toISOString() : now()->toISOString(),
                'deadline' => $this->order->writer_deadline ? $this->order->writer_deadline->toISOString() : null,
                'file_count' => $fileCount,
                'is_late_submission' => $this->isLateSubmission,
                'files' => $this->uploadedFiles->map(function ($file) {
                    return [
                        'name' => $file->file_name,
                        'size' => $file->file_size,
                        'type' => $file->file_type
                    ];
                })->toArray()
            ]
        );
    }
} 