<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Carbon\Carbon;

class BlogFileProcessor
{
    private array $config;
    private string $uploadDisk;
    private string $tempDisk;

    public function __construct()
    {
        $this->config = config('blog_uploads');
        $this->uploadDisk = $this->config['storage']['upload_disk'];
        $this->tempDisk = $this->config['storage']['temp_disk'];
    }

    /**
     * Process uploaded blog files
     */
    public function processUploadedFiles(string $uploadId): array
    {
        try {
            $uploadPath = $this->config['storage']['directories']['uploads'] . '/' . $uploadId;
            $files = Storage::disk($this->uploadDisk)->files($uploadPath);

            $results = [];

            foreach ($files as $file) {
                $result = $this->processFile($file, $uploadId);
                $results[] = $result;
            }

            // Log processing results
            if ($this->config['logging']['log_processing']) {
                Log::info('Blog files processed', [
                    'upload_id' => $uploadId,
                    'files_processed' => count($results),
                    'results' => $results,
                ]);
            }

            return [
                'success' => true,
                'upload_id' => $uploadId,
                'processed_files' => $results,
                'total_files' => count($results),
            ];
        } catch (\Exception $e) {
            Log::error('Blog file processing failed', [
                'upload_id' => $uploadId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return [
                'success' => false,
                'error' => 'Processing failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Process individual file
     */
    private function processFile(string $filePath, string $uploadId): array
    {
        $fileName = basename($filePath);
        $fileContent = Storage::disk($this->uploadDisk)->get($filePath);

        try {
            // Parse JavaScript content
            $parsedData = $this->parseJavaScriptContent($fileContent);

            // Validate parsed data
            $validation = $this->validateParsedData($parsedData);

            if (!$validation['valid']) {
                return [
                    'filename' => $fileName,
                    'status' => 'failed',
                    'errors' => $validation['errors'],
                ];
            }

            // Extract blog post data
            $blogData = $this->extractBlogData($parsedData);

            // Generate processed file
            $processedPath = $this->saveProcessedFile($blogData, $fileName, $uploadId);

            return [
                'filename' => $fileName,
                'status' => 'success',
                'processed_path' => $processedPath,
                'blog_data' => $blogData,
                'processed_at' => Carbon::now()->toISOString(),
            ];
        } catch (\Exception $e) {
            Log::error('File processing error', [
                'filename' => $fileName,
                'error' => $e->getMessage(),
            ]);

            return [
                'filename' => $fileName,
                'status' => 'error',
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Parse JavaScript content to extract data - COMPLETELY REWRITTEN
     */
    private function parseJavaScriptContent(string $content): array
    {
        Log::info('Starting robust JavaScript parsing...');

        // Remove comments first
        $content = $this->removeComments($content);

        // Extract the main object content
        $exportContent = $this->extractExportContent($content);

        if (empty($exportContent)) {
            throw new \Exception('No valid export default found in JavaScript file');
        }

        // Use a more robust parsing approach
        $phpArray = $this->parseJavaScriptObject($exportContent);

        Log::info('Robust parsing completed. Found ' . count($phpArray) . ' fields: ' . implode(', ', array_keys($phpArray)));

        return $phpArray;
    }

    /**
     * Extract export content from JavaScript
     */
    private function extractExportContent(string $content): ?string
    {
        // Method 1: Direct export default { ... }
        if (preg_match('/export\s+default\s+(\{[\s\S]*\});?\s*$/i', $content, $matches)) {
            return $matches[1];
        }

        // Method 2: Variable export (const post = { ... }; export default post;)
        if (preg_match('/const\s+(\w+)\s*=\s*(\{[\s\S]*\});\s*export\s+default\s+\1;?/i', $content, $matches)) {
            return $matches[2];
        }

        return null;
    }

    /**
     * Parse JavaScript object using character-by-character approach
     */
    private function parseJavaScriptObject(string $jsObject): array
    {
        // Remove outer braces
        $jsObject = trim($jsObject, '{}');

        $result = [];
        $i = 0;
        $length = strlen($jsObject);

        while ($i < $length) {
            // Skip whitespace and comments
            $i = $this->skipWhitespace($jsObject, $i);
            if ($i >= $length) break;

            // Parse field name
            $fieldResult = $this->parseFieldName($jsObject, $i);
            if (!$fieldResult) {
                $i++;
                continue;
            }

            $fieldName = $fieldResult['name'];
            $i = $fieldResult['position'];

            Log::debug("Parsing field: {$fieldName}");

            // Skip whitespace after field name
            $i = $this->skipWhitespace($jsObject, $i);

            // Expect colon
            if ($i >= $length || $jsObject[$i] !== ':') {
                Log::warning("Expected ':' after field name {$fieldName}");
                $i++;
                continue;
            }
            $i++; // Skip colon

            // Skip whitespace after colon
            $i = $this->skipWhitespace($jsObject, $i);

            // Parse field value
            $valueResult = $this->parseFieldValue($jsObject, $i);
            if (!$valueResult) {
                Log::warning("Failed to parse value for field {$fieldName}");
                $i++;
                continue;
            }

            $result[$fieldName] = $valueResult['value'];
            $i = $valueResult['position'];

            Log::debug("Successfully parsed field: {$fieldName}");

            // Skip whitespace after value
            $i = $this->skipWhitespace($jsObject, $i);

            // Skip comma if present
            if ($i < $length && $jsObject[$i] === ',') {
                $i++;
            }
        }

        return $result;
    }

    /**
     * Skip whitespace and comments
     */
    private function skipWhitespace(string $content, int $start): int
    {
        $i = $start;
        $length = strlen($content);

        while ($i < $length) {
            $char = $content[$i];

            // Skip whitespace
            if (ctype_space($char)) {
                $i++;
                continue;
            }

            // Skip single-line comments
            if ($i + 1 < $length && $content[$i] === '/' && $content[$i + 1] === '/') {
                while ($i < $length && $content[$i] !== "\n") {
                    $i++;
                }
                continue;
            }

            // Skip multi-line comments
            if ($i + 1 < $length && $content[$i] === '/' && $content[$i + 1] === '*') {
                $i += 2;
                while ($i + 1 < $length && !($content[$i] === '*' && $content[$i + 1] === '/')) {
                    $i++;
                }
                $i += 2;
                continue;
            }

            break;
        }

        return $i;
    }

    /**
     * Parse field name
     */
    private function parseFieldName(string $content, int $start): ?array
    {
        $i = $start;
        $length = strlen($content);

        // Handle quoted field names
        if ($content[$i] === '"' || $content[$i] === "'") {
            $quote = $content[$i];
            $i++;
            $fieldName = '';

            while ($i < $length && $content[$i] !== $quote) {
                if ($content[$i] === '\\' && $i + 1 < $length) {
                    $i++; // Skip escape character
                }
                $fieldName .= $content[$i];
                $i++;
            }

            if ($i < $length) {
                $i++; // Skip closing quote
            }

            return ['name' => $fieldName, 'position' => $i];
        }

        // Handle unquoted field names
        $fieldName = '';
        while ($i < $length && preg_match('/[a-zA-Z0-9_$]/', $content[$i])) {
            $fieldName .= $content[$i];
            $i++;
        }

        if (empty($fieldName)) {
            return null;
        }

        return ['name' => $fieldName, 'position' => $i];
    }

    /**
     * Parse field value with proper nesting support
     */
    private function parseFieldValue(string $content, int $start): ?array
    {
        $i = $start;
        $length = strlen($content);

        if ($i >= $length) {
            return null;
        }

        $char = $content[$i];

        // Handle different value types
        switch ($char) {
            case '"':
            case "'":
                return $this->parseStringValue($content, $i);

            case '`':
                return $this->parseTemplateString($content, $i);

            case '{':
                return $this->parseObjectValue($content, $i);

            case '[':
                return $this->parseArrayValue($content, $i);

            default:
                return $this->parsePrimitiveValue($content, $i);
        }
    }

    /**
     * Parse string value
     */
    private function parseStringValue(string $content, int $start): ?array
    {
        $quote = $content[$start];
        $i = $start + 1;
        $length = strlen($content);
        $value = '';

        while ($i < $length && $content[$i] !== $quote) {
            if ($content[$i] === '\\' && $i + 1 < $length) {
                $i++; // Skip escape character
            }
            $value .= $content[$i];
            $i++;
        }

        if ($i < $length) {
            $i++; // Skip closing quote
        }

        return ['value' => $value, 'position' => $i];
    }

    /**
     * Parse template string (backticks)
     */
    private function parseTemplateString(string $content, int $start): ?array
    {
        $i = $start + 1; // Skip opening backtick
        $length = strlen($content);
        $value = '';
        $braceLevel = 0;

        while ($i < $length) {
            $char = $content[$i];

            if ($char === '`' && $braceLevel === 0) {
                $i++; // Skip closing backtick
                break;
            }

            if ($char === '$' && $i + 1 < $length && $content[$i + 1] === '{') {
                $braceLevel++;
                $value .= $char;
                $i++;
                continue;
            }

            if ($char === '}' && $braceLevel > 0) {
                $braceLevel--;
            }

            $value .= $char;
            $i++;
        }

        return ['value' => $value, 'position' => $i];
    }

    /**
     * Parse object value
     */
    private function parseObjectValue(string $content, int $start): ?array
    {
        $i = $start + 1; // Skip opening brace
        $length = strlen($content);
        $braceLevel = 1;
        $objectContent = '';

        while ($i < $length && $braceLevel > 0) {
            $char = $content[$i];

            if ($char === '{') {
                $braceLevel++;
            } elseif ($char === '}') {
                $braceLevel--;
            }

            if ($braceLevel > 0) {
                $objectContent .= $char;
            }

            $i++;
        }

        // Parse the object content recursively
        $parsedObject = $this->parseJavaScriptObject($objectContent);

        return ['value' => $parsedObject, 'position' => $i];
    }

    /**
     * Parse array value
     */
    private function parseArrayValue(string $content, int $start): ?array
    {
        $i = $start + 1; // Skip opening bracket
        $length = strlen($content);
        $bracketLevel = 1;
        $arrayContent = '';

        while ($i < $length && $bracketLevel > 0) {
            $char = $content[$i];

            if ($char === '[') {
                $bracketLevel++;
            } elseif ($char === ']') {
                $bracketLevel--;
            }

            if ($bracketLevel > 0) {
                $arrayContent .= $char;
            }

            $i++;
        }

        // Parse array elements
        $elements = [];
        $elementStart = 0;
        $elementLength = strlen($arrayContent);

        while ($elementStart < $elementLength) {
            $elementStart = $this->skipWhitespace($arrayContent, $elementStart);
            if ($elementStart >= $elementLength) break;

            $elementResult = $this->parseFieldValue($arrayContent, $elementStart);
            if ($elementResult) {
                $elements[] = $elementResult['value'];
                $elementStart = $elementResult['position'];

                // Skip whitespace and comma
                $elementStart = $this->skipWhitespace($arrayContent, $elementStart);
                if ($elementStart < $elementLength && $arrayContent[$elementStart] === ',') {
                    $elementStart++;
                }
            } else {
                break;
            }
        }
        return ['value' => $elements, 'position' => $i];
    }

    /**
     * Parse primitive value (number, boolean, null)
     */
    private function parsePrimitiveValue(string $content, int $start): ?array
    {
        $i = $start;
        $length = strlen($content);
        $value = '';

        // Read until comma, brace, bracket, or end
        while ($i < $length) {
            $char = $content[$i];

            if ($char === ',' || $char === '}' || $char === ']' || ctype_space($char)) {
                break;
            }

            $value .= $char;
            $i++;
        }

        $value = trim($value);

        // Convert to appropriate type
        if ($value === 'true') {
            return ['value' => true, 'position' => $i];
        } elseif ($value === 'false') {
            return ['value' => false, 'position' => $i];
        } elseif ($value === 'null') {
            return ['value' => null, 'position' => $i];
        } elseif (is_numeric($value)) {
            return ['value' => strpos($value, '.') !== false ? (float)$value : (int)$value, 'position' => $i];
        }

        return ['value' => $value, 'position' => $i];
    }

    /**
     * Remove JavaScript comments
     */
    private function removeComments(string $content): string
    {
        // Remove single-line comments
        $content = preg_replace('/\/\/.*$/m', '', $content);

        // Remove multi-line comments
        $content = preg_replace('/\/\*[\s\S]*?\*\//', '', $content);

        return $content;
    }

    /**
     * Validate parsed data structure
     */
    private function validateParsedData(array $data): array
    {
        $errors = [];

        // Check required fields
        foreach ($this->config['file_validation']['required_fields'] as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                $errors[] = "Missing required field: {$field}";
            }
        }

        // Validate specific field types
        if (isset($data['publishedAt'])) {
            if (!$this->isValidDate($data['publishedAt'])) {
                $errors[] = "Invalid publishedAt date format";
            }
        }

        if (isset($data['readTime'])) {
            if (!is_numeric($data['readTime']) || $data['readTime'] < 0) {
                $errors[] = "Invalid readTime value";
            }
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors,
        ];
    }

    /**
     * Extract blog data for database storage
     */
    private function extractBlogData(array $data): array
    {
        // Log::info('Available fields in parsed data: ' . implode(', ', array_keys($data)));

        // // Check for specific fields
        // Log::info('Checking specific fields:');
        // Log::info('subtitle: ' . (isset($data['subtitle']) ? 'FOUND' : 'NOT FOUND'));
        // Log::info('featuredImage: ' . (isset($data['featuredImage']) ? 'FOUND' : 'NOT FOUND'));
        // Log::info('structuredData: ' . (isset($data['structuredData']) ? 'FOUND' : 'NOT FOUND'));
        // Log::info('tableOfContents: ' . (isset($data['tableOfContents']) ? 'FOUND' : 'NOT FOUND'));
        // Log::info('socialSharing: ' . (isset($data['socialSharing']) ? 'FOUND' : 'NOT FOUND'));
        // Log::info('analytics: ' . (isset($data['analytics']) ? 'FOUND' : 'NOT FOUND'));

        $extractedData = [
            'title' => $data['title'] ?? '',
            'subtitle' => $data['subtitle'] ?? null,
            'slug' => $data['slug'] ?? Str::slug($data['title'] ?? ''),
            'excerpt' => $data['excerpt'] ?? '',
            'content' => $data['content'] ?? '',
            'category' => $data['category'] ?? 'general',
            'author' => $data['author'] ?? 'Academic Scribe Team',
            'published_at' => $data['publishedAt'] ?? Carbon::now()->toISOString(),
            'read_time' => $data['readTime'] ?? 5,
            'difficulty' => $data['difficulty'] ?? 'beginner',
            'keywords' => $data['keywords'] ?? [],
            'tags' => $data['tags'] ?? [],
            'featured_image' => $data['featuredImage'] ?? null,
            'featured_image_alt' => $data['featuredImageAlt'] ?? null,
            'meta_data' => [
                'title' => $data['meta']['title'] ?? $data['title'] ?? '',
                'description' => $data['meta']['description'] ?? $data['excerpt'] ?? '',
                'keywords' => $data['meta']['keywords'] ?? implode(', ', $data['keywords'] ?? []),
                'canonical' => $data['meta']['canonical'] ?? null,
                'openGraph' => $data['meta']['openGraph'] ?? [],
                'twitter' => $data['meta']['twitter'] ?? [],
            ],
            'structured_data' => $data['structuredData'] ?? [],
            'table_of_contents' => $data['tableOfContents'] ?? [],
            'social_sharing' => $data['socialSharing'] ?? [],
            'analytics' => $data['analytics'] ?? [],
        ];

        // Log::info('Final extracted data keys: ' . implode(', ', array_keys($extractedData)));
        // Log::info('Final subtitle value: ' . ($extractedData['subtitle'] ?? 'NULL'));
        // Log::info('Final featured_image value: ' . ($extractedData['featured_image'] ?? 'NULL'));

        return $extractedData;
    }

    /**
     * Save processed file
     */
    private function saveProcessedFile(array $blogData, string $originalFileName, string $uploadId): string
    {
        $processedPath = $this->config['storage']['directories']['processed'] . '/' . $uploadId;
        $fileName = pathinfo($originalFileName, PATHINFO_FILENAME) . '_processed.json';

        $filePath = $processedPath . '/' . $fileName;

        Storage::disk($this->uploadDisk)->put($filePath, json_encode($blogData, JSON_PRETTY_PRINT));

        return $filePath;
    }

    /**
     * Validate date format
     */
    private function isValidDate(string $date): bool
    {
        try {
            Carbon::parse($date);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}
