<?php

namespace App\Services;

use App\Models\PageContent;
use Illuminate\Support\Facades\DB;

class ContentBlockService
{
    /**
     * Get content blocks for a page
     */
    public function getPageContentBlocks(int $pageId, bool $activeOnly = true): \Illuminate\Database\Eloquent\Collection
    {
        $query = PageContent::where('page_id', $pageId);

        if ($activeOnly) {
            $query->where('is_active', true);
        }

        return $query->orderBy('order')->get();
    }

    /**
     * Create a content block
     */
    public function createContentBlock(array $data): PageContent
    {
        return PageContent::create($data);
    }

    /**
     * Update a content block
     */
    public function updateContentBlock(PageContent $contentBlock, array $data): PageContent
    {
        $contentBlock->update($data);
        return $contentBlock->fresh();
    }

    /**
     * Delete a content block
     */
    public function deleteContentBlock(PageContent $contentBlock): bool
    {
        return $contentBlock->delete();
    }

    /**
     * Reorder content blocks for a page
     */
    public function reorderContentBlocks(int $pageId, array $blockOrders): bool
    {
        return DB::transaction(function () use ($pageId, $blockOrders) {
            foreach ($blockOrders as $blockId => $order) {
                PageContent::where('id', $blockId)
                    ->where('page_id', $pageId)
                    ->update(['order' => $order]);
            }
            return true;
        });
    }

    /**
     * Toggle content block status
     */
    public function toggleContentBlockStatus(PageContent $contentBlock): PageContent
    {
        $contentBlock->update(['is_active' => !$contentBlock->is_active]);
        return $contentBlock->fresh();
    }

    /**
     * Get content blocks by type
     */
    public function getContentBlocksByType(int $pageId, string $blockType): \Illuminate\Database\Eloquent\Collection
    {
        return PageContent::where('page_id', $pageId)
            ->where('block_type', $blockType)
            ->where('is_active', true)
            ->orderBy('order')
            ->get();
    }

    /**
     * Get content block by key
     */
    public function getContentBlockByKey(int $pageId, string $blockKey): ?PageContent
    {
        return PageContent::where('page_id', $pageId)
            ->where('block_key', $blockKey)
            ->where('is_active', true)
            ->first();
    }

    /**
     * Duplicate a content block
     */
    public function duplicateContentBlock(PageContent $contentBlock, int $newPageId = null): PageContent
    {
        $data = $contentBlock->toArray();
        unset($data['id'], $data['created_at'], $data['updated_at']);

        if ($newPageId) {
            $data['page_id'] = $newPageId;
        }

        // Modify block key to avoid conflicts
        if ($data['block_key']) {
            $data['block_key'] = $data['block_key'] . '-copy-' . time();
        }

        return PageContent::create($data);
    }

    /**
     * Get content blocks statistics for a page
     */
    public function getContentBlocksStats(int $pageId): array
    {
        $blocks = PageContent::where('page_id', $pageId)->get();

        return [
            'total_blocks' => $blocks->count(),
            'active_blocks' => $blocks->where('is_active', true)->count(),
            'inactive_blocks' => $blocks->where('is_active', false)->count(),
            'block_types' => $blocks->groupBy('block_type')->map->count(),
            'latest_update' => $blocks->max('updated_at'),
        ];
    }

    /**
     * Search content blocks
     */
    public function searchContentBlocks(string $searchTerm, int $pageId = null): \Illuminate\Database\Eloquent\Collection
    {
        $query = PageContent::query();

        if ($pageId) {
            $query->where('page_id', $pageId);
        }

        return $query->where(function ($q) use ($searchTerm) {
            $q->where('block_type', 'like', '%' . $searchTerm . '%')
              ->orWhere('block_key', 'like', '%' . $searchTerm . '%')
              ->orWhereRaw('JSON_SEARCH(content, "one", ?) IS NOT NULL', ['%' . $searchTerm . '%']);
        })
        ->orderBy('updated_at', 'desc')
        ->get();
    }

    /**
     * Validate content block structure
     */
    public function validateContentBlock(array $data): array
    {
        $errors = [];

        // Required fields
        if (empty($data['page_id'])) {
            $errors[] = 'Page ID is required';
        }

        if (empty($data['block_type'])) {
            $errors[] = 'Block type is required';
        }

        if (empty($data['content'])) {
            $errors[] = 'Content is required';
        }

        // Validate content is JSON
        if (isset($data['content']) && !is_array($data['content']) && !is_string($data['content'])) {
            $errors[] = 'Content must be valid JSON or array';
        }

        // Validate block type is allowed
        $allowedBlockTypes = [
            'hero', 'overview', 'benefits', 'process', 'testimonials', 
            'faqs', 'contact_form', 'cta', 'features', 'pricing', 
            'gallery', 'text_content', 'video', 'custom'
        ];

        if (isset($data['block_type']) && !in_array($data['block_type'], $allowedBlockTypes)) {
            $errors[] = 'Invalid block type';
        }

        return $errors;
    }

    /**
     * Get content block template
     */
    public function getContentBlockTemplate(string $blockType): array
    {
        $templates = [
            'hero' => [
                'title' => '',
                'subtitle' => '',
                'description' => '',
                'cta_text' => '',
                'cta_link' => '',
                'background_image' => '',
                'background_color' => ''
            ],
            'overview' => [
                'title' => '',
                'description' => '',
                'features' => [],
                'image' => ''
            ],
            'benefits' => [
                'title' => '',
                'subtitle' => '',
                'benefits' => []
            ],
            'process' => [
                'title' => '',
                'subtitle' => '',
                'steps' => []
            ],
            'testimonials' => [
                'title' => '',
                'subtitle' => '',
                'testimonials' => []
            ],
            'faqs' => [
                'title' => '',
                'subtitle' => '',
                'faqs' => []
            ],
            'contact_form' => [
                'title' => '',
                'description' => '',
                'form_fields' => [],
                'submit_text' => 'Submit'
            ],
            'cta' => [
                'title' => '',
                'description' => '',
                'button_text' => '',
                'button_link' => '',
                'background_color' => ''
            ],
            'features' => [
                'title' => '',
                'subtitle' => '',
                'features' => []
            ],
            'pricing' => [
                'title' => '',
                'subtitle' => '',
                'plans' => []
            ],
            'gallery' => [
                'title' => '',
                'images' => []
            ],
            'text_content' => [
                'title' => '',
                'content' => '',
                'layout' => 'single_column'
            ],
            'video' => [
                'title' => '',
                'description' => '',
                'video_url' => '',
                'thumbnail' => ''
            ],
            'custom' => []
        ];

        return $templates[$blockType] ?? [];
    }
}
