<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;
use Illuminate\Support\Facades\Log;

class ImageOptimizationService
{
    protected $config;
    protected $imageManager;

    public function __construct()
    {
        $this->config = $this->getDefaultConfig();
        $this->imageManager = new ImageManager(new Driver());
    }

    /**
     * Get default configuration with fallbacks
     */
    protected function getDefaultConfig(): array
    {
        $configFile = config('image-optimization');

        // If config file doesn't exist, use defaults
        if (!$configFile) {
            // Check if WebP is supported
            $webpSupported = function_exists('imagewebp') && gd_info()['WebP Support'];
            
            return [
                'default_quality' => 85,
                'enable_avif' => true,
                'enable_webp' => $webpSupported,
                'progressive_jpeg' => true,
                'presets' => [
                    'blog' => [
                        'formats' => [
                            'webp' => ['width' => 1200, 'height' => 675, 'quality' => 80],
                            'fallback' => ['width' => 1200, 'height' => 675, 'quality' => 85, 'format' => 'jpg'],
                        ]
                    ],
                    'hero' => [
                        'formats' => [
                            'webp' => ['width' => 1920, 'height' => 1080, 'quality' => 85],
                            'fallback' => ['width' => 1920, 'height' => 1080, 'quality' => 90, 'format' => 'jpg'],
                        ]
                    ],
                    'service' => [
                        'formats' => [
                            'webp' => ['width' => 800, 'height' => 600, 'quality' => 80],
                            'fallback' => ['width' => 800, 'height' => 600, 'quality' => 85, 'format' => 'jpg'],
                        ]
                    ],
                    'thumbnail' => [
                        'formats' => [
                            'webp' => ['width' => 400, 'height' => 300, 'quality' => 75],
                            'fallback' => ['width' => 400, 'height' => 300, 'quality' => 80, 'format' => 'jpg'],
                        ]
                    ],
                    'avatar' => [
                        'formats' => [
                            'webp' => ['width' => 200, 'height' => 200, 'quality' => 80],
                            'fallback' => ['width' => 200, 'height' => 200, 'quality' => 85, 'format' => 'jpg'],
                        ]
                    ],
                    'logo' => [
                        'formats' => [
                            'webp' => ['width' => 300, 'height' => 120, 'quality' => 90],
                            'fallback' => ['width' => 300, 'height' => 120, 'quality' => 95, 'format' => 'png'],
                        ]
                    ],
                ],
                'optimization' => [
                    'auto_orient' => true,
                    'sharpen' => false,
                    'strip_metadata' => true,
                    'compression_level' => 9,
                ],
                'processing' => [
                    'memory_limit' => '256M',
                ],
                'development' => [
                    'log_processing' => false,
                ],
                'responsive' => [
                    'lazy_loading' => true,
                ],
                'performance' => [
                    'batch_size' => 10,
                ],
            ];
        }

        return $configFile;
    }

    /**
     * Get optimized image URLs for a given image path and type
     */
    public function getOptimizedImageUrls(string $imagePath, string $conversionType = 'blog'): array
    {
        // Check if optimized versions exist
        $pathInfo = pathinfo($imagePath);
        $directory = $pathInfo['dirname'];
        $filename = $pathInfo['filename'];

        $urls = [];
        $preset = $this->config['presets'][$conversionType] ?? $this->config['presets']['blog'];

        if (!$preset) {
            return ['fallback' => $imagePath];
        }

        foreach ($preset['formats'] as $format => $settings) {
            // Skip WebP if not supported
            if ($format === 'webp' && !function_exists('imagewebp')) {
                continue;
            }
            
            $optimizedPath = $this->getOptimizedPath($directory, $filename, $format, $settings);

            if (File::exists(public_path($optimizedPath))) {
                $urls[$format] = $optimizedPath;
            }
        }

        // Always include original as fallback if no fallback was created
        if (!isset($urls['fallback'])) {
            $urls['fallback'] = $imagePath;
        }

        return $urls;
    }

    /**
     * Convert and store a static image using Intervention Image
     */
    public function convertStaticImage(string $staticImagePath, string $conversionType = 'blog', string $collection = 'images'): array
    {
        $fullPath = public_path(ltrim($staticImagePath, '/'));

        if (!file_exists($fullPath)) {
            throw new \Exception("Image not found: {$staticImagePath}");
        }

        // Get conversion settings from config
        $preset = $this->config['presets'][$conversionType] ?? $this->config['presets']['blog'];

        if (!$preset || !isset($preset['formats'])) {
            throw new \Exception("Invalid preset configuration for: {$conversionType}");
        }

        $optimizedUrls = [];

        // Get path info
        $pathInfo = pathinfo($staticImagePath);
        $directory = $pathInfo['dirname'];
        $filename = $pathInfo['filename'];

        // Log processing if enabled
        if ($this->config['development']['log_processing'] ?? false) {
            Log::info("Processing image: {$staticImagePath} with preset: {$conversionType}");
        }

        // Create optimized versions
        foreach ($preset['formats'] as $format => $settings) {
            // Skip WebP if not supported
            if ($format === 'webp' && !function_exists('imagewebp')) {
                Log::info("Skipping WebP conversion for {$staticImagePath} - WebP not supported");
                continue;
            }
            
            try {
                $optimizedPath = $this->createOptimizedImage(
                    $fullPath,
                    $directory,
                    $filename,
                    $format,
                    $settings
                );

                if ($optimizedPath) {
                    $optimizedUrls[$format] = $optimizedPath;

                    if ($this->config['development']['log_processing'] ?? false) {
                        Log::info("Created {$format} version: {$optimizedPath}");
                    }
                }
            } catch (\Exception $e) {
                Log::warning("Failed to create {$format} version of {$staticImagePath}: " . $e->getMessage());
            }
        }

        // Always include original as fallback if no fallback was created
        if (!isset($optimizedUrls['fallback'])) {
            $optimizedUrls['fallback'] = $staticImagePath;
        }

        return $optimizedUrls;
    }

    /**
     * Create optimized image using Intervention Image with config settings
     */
    protected function createOptimizedImage(string $sourcePath, string $directory, string $filename, string $format, array $settings): ?string
    {
        // Get output format
        $outputFormat = $settings['format'] ?? $format;

        // Skip AVIF if not enabled (GD doesn't support AVIF well)
        if ($format === 'avif') {
            return null; // Skip AVIF for now with GD driver
        }

        // Skip WebP if not enabled
        if ($format === 'webp' && !($this->config['enable_webp'] ?? true)) {
            return null;
        }

        // Create output filename
        $outputFilename = $filename . '.' . $outputFormat;
        $outputPath = public_path($directory . '/' . $outputFilename);

        // Skip if file already exists
        if (File::exists($outputPath)) {
            return '/' . ltrim($directory . '/' . $outputFilename, '/');
        }

        // Ensure directory exists
        $outputDir = dirname($outputPath);
        if (!File::exists($outputDir)) {
            File::makeDirectory($outputDir, 0755, true);
        }

        try {
            // Set memory limit if specified
            if (isset($this->config['processing']['memory_limit'])) {
                ini_set('memory_limit', $this->config['processing']['memory_limit']);
            }

            // Load image using Intervention Image v3 syntax
            $image = $this->imageManager->read($sourcePath);

            // Resize if dimensions are specified
            if (isset($settings['width']) && isset($settings['height'])) {
                // Use contain() to preserve aspect ratio without cropping
                // This ensures logos and images aren't cropped
                $image->contain($settings['width'], $settings['height']);
            } elseif (isset($settings['width'])) {
                $image->scale(width: $settings['width']);
            } elseif (isset($settings['height'])) {
                $image->scale(height: $settings['height']);
            }

            // Set quality
            $quality = $settings['quality'] ?? ($this->config['default_quality'] ?? 85);

            // Save in the specified format
            switch ($outputFormat) {
                case 'webp':
                    // Check if WebP is supported
                    if (function_exists('imagewebp') && gd_info()['WebP Support']) {
                        $image->toWebp($quality)->save($outputPath);
                    } else {
                        // Fallback to JPEG if WebP is not supported
                        Log::warning("WebP not supported, falling back to JPEG for: {$outputPath}");
                        $image->toJpeg($quality)->save($outputPath);
                    }
                    break;
                case 'jpg':
                case 'jpeg':
                    $image->toJpeg($quality)->save($outputPath);
                    break;
                case 'png':
                    $image->toPng()->save($outputPath);
                    break;
                default:
                    $image->save($outputPath);
            }

            return '/' . ltrim($directory . '/' . $outputFilename, '/');
        } catch (\Exception $e) {
            // If processing fails, copy original for fallback
            if ($format === 'fallback') {
                copy($sourcePath, $outputPath);
                return '/' . ltrim($directory . '/' . $outputFilename, '/');
            }
            throw $e;
        }
    }

    /**
     * Get the expected path for an optimized image
     */
    protected function getOptimizedPath(string $directory, string $filename, string $format, array $settings): string
    {
        $outputFormat = $settings['format'] ?? $format;
        $outputFilename = $filename . '.' . $outputFormat;
        return '/' . ltrim($directory . '/' . $outputFilename, '/');
    }

    /**
     * Get responsive image data for React components
     */
    public function getResponsiveImageData(string $imagePath, string $conversionType = 'blog', string $alt = ''): array
    {
        $urls = $this->getOptimizedImageUrls($imagePath, $conversionType);
        $preset = $this->config['presets'][$conversionType] ?? $this->config['presets']['blog'];

        $sources = [];

        // Add WebP source if available
        if (isset($urls['webp'])) {
            $sources[] = [
                'srcSet' => $urls['webp'],
                'type' => 'image/webp',
            ];
        }

        return [
            'sources' => $sources,
            'fallback' => [
                'src' => $urls['fallback'] ?? $imagePath,
                'alt' => $alt,
                'width' => $preset['formats']['fallback']['width'] ?? null,
                'height' => $preset['formats']['fallback']['height'] ?? null,
            ],
            'loading' => ($this->config['responsive']['lazy_loading'] ?? true) ? 'lazy' : 'eager',
            'decoding' => 'async',
        ];
    }

    /**
     * Batch convert multiple static images
     */
    public function batchConvertStaticImages(array $imagePaths, string $conversionType = 'blog'): array
    {
        $results = [];
        $batchSize = $this->config['performance']['batch_size'] ?? 10;
        $batches = array_chunk($imagePaths, $batchSize);

        foreach ($batches as $batch) {
            foreach ($batch as $path) {
                try {
                    $results[$path] = $this->convertStaticImage($path, $conversionType);
                } catch (\Exception $e) {
                    $results[$path] = ['error' => $e->getMessage()];
                    Log::error("Failed to process image {$path}: " . $e->getMessage());
                }
            }
        }

        return $results;
    }
}
