<?php

namespace App\Services;

use Exception;
use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Notifications\Notification;
use App\Repositories\NotificationRepository;
use Illuminate\Support\Facades\Notification as NotificationFacade;

class NotificationService
{
    /**
     * @var NotificationRepository
     */
    protected $repository;

    /**
     * Create a new notification service instance.
     *
     * @param NotificationRepository $repository
     * @return void
     */
    public function __construct(NotificationRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Send a notification to a specific user.
     *
     * @param User $user
     * @param Notification $notification
     * @return void
     */
    public function sendToUser(User $user, Notification $notification)
    {
        try {
            // Check if user can receive notifications
            if (!$this->canReceiveNotifications($user)) {
                Log::info("Notification not sent to user {$user->id}: user cannot receive notifications");
                return;
            }

            NotificationFacade::send($user, $notification);

            // Invalidate notification cache for this user
            $this->invalidateUserCache($user->id);

            Log::info("Notification sent to user {$user->id}", [
                'notification_type' => get_class($notification),
                'user_id' => $user->id
            ]);
        } catch (Exception $e) {
            Log::error("Failed to send notification to user {$user->id}", [
                'exception' => $e->getMessage(),
                'notification_type' => get_class($notification),
                'user_id' => $user->id
            ]);
        }
    }

    /**
     * Send a notification to multiple users.
     *
     * @param Collection|array $users
     * @param Notification $notification
     * @return void
     */
    public function sendToUsers($users, Notification $notification)
    {
        if (is_array($users)) {
            $users = collect($users);
        }

        // Ensure we have a collection
        if (!$users instanceof Collection) {
            throw new \InvalidArgumentException('Users must be a Collection or array');
        }

        // For large collections, process in chunks
        if ($users->count() > 100) {
            $users->chunk(100)->each(function ($chunk) use ($notification) {
                $this->processBatchNotification($chunk, $notification);
            });
        } else {
            $this->processBatchNotification($users, $notification);
        }
    }

    /**
     * Send a notification to users with a specific role.
     *
     * @param string $role
     * @param Notification $notification
     * @return void
     */
    public function sendToRole(string $role, Notification $notification)
    {
        try {
            $users = User::where('user_type', $role)->get();

            if ($users->isEmpty()) {
                Log::info("No users found with role: {$role}");
                return;
            }

            $this->sendToUsers($users, $notification);

            Log::info("Notification sent to users with role: {$role}", [
                'notification_type' => get_class($notification),
                'role' => $role,
                'user_count' => $users->count()
            ]);
        } catch (Exception $e) {
            Log::error("Failed to send notification to role {$role}", [
                'exception' => $e->getMessage(),
                'notification_type' => get_class($notification)
            ]);
        }
    }

    /**
     * Send a notification to all users.
     *
     * @param Notification $notification
     * @return void
     */
    public function broadcast(Notification $notification)
    {
        try {
            // This could be resource-intensive for large user bases
            // Consider using queues or batch processing
            User::query()
                ->whereNull('deactivated_at') // Only active users
                ->chunk(100, function ($users) use ($notification) {
                    $this->sendToUsers($users, $notification);
                });

            Log::info("Broadcast notification sent", [
                'notification_type' => get_class($notification)
            ]);
        } catch (Exception $e) {
            Log::error("Failed to broadcast notification", [
                'exception' => $e->getMessage(),
                'notification_type' => get_class($notification)
            ]);
        }
    }

    /**
     * Get notifications for a specific user.
     *
     * @param User $user
     * @param array $filters
     * @return \Illuminate\Pagination\LengthAwarePaginator
     */
    public function getForUser(User $user, array $filters = [])
    {
        return $this->repository->getForUser($user->id, $filters);
    }

    /**
     * Get unread notifications for a specific user.
     *
     * @param User $user
     * @param array $filters
     * @return \Illuminate\Pagination\LengthAwarePaginator
     */
    public function getUnreadForUser(User $user, array $filters = [])
    {
        $filters['read'] = false;
        return $this->repository->getForUser($user->id, $filters);
    }

    /**
     * Mark notifications as read.
     *
     * @param array|string $notificationIds
     * @param User|null $user
     * @return bool
     */
    public function markAsRead($notificationIds, ?User $user = null)
    {
        try {
            // If it's a single ID, convert to array
            if (!is_array($notificationIds)) {
                $notificationIds = [$notificationIds];
            }

            // Log the input parameters
            Log::info('Marking notifications as read', [
                'notification_ids' => $notificationIds,
                'user_id' => $user ? $user->id : 'null'
            ]);

            // If no notifications provided, return false
            if (empty($notificationIds)) {
                Log::warning('No notification IDs provided to mark as read');
                return false;
            }

            // If user is provided, ensure they own these notifications
            $userNotificationIds = [];
            if ($user !== null) {
                // Get notifications that actually belong to this user
                $userNotificationIds = $user->notifications()
                    ->whereIn('id', $notificationIds)
                    ->pluck('id')
                    ->toArray();

                // Log if any notifications don't belong to the user
                $nonUserNotifications = array_diff($notificationIds, $userNotificationIds);
                if (!empty($nonUserNotifications)) {
                    Log::warning('Some notifications do not belong to the user', [
                        'user_id' => $user->id,
                        'non_user_notifications' => $nonUserNotifications
                    ]);
                }

                // Only proceed with notifications that belong to this user
                $notificationIds = $userNotificationIds;
            }

            // If after filtering, we have no valid notifications, return false
            if (empty($notificationIds)) {
                Log::warning('No valid notification IDs to mark as read after filtering');
                return false;
            }

            // Begin transaction for database consistency
            DB::beginTransaction();

            // Get the current state of notifications before updating
            $beforeUpdate = DB::table('notifications')
                ->whereIn('id', $notificationIds)
                ->select('id', 'read_at')
                ->get()
                ->keyBy('id')
                ->toArray();

            Log::info('Notification states before update', [
                'before_update' => $beforeUpdate
            ]);

            // Perform the update using direct database query for better control
            $result = DB::table('notifications')
                ->whereIn('id', $notificationIds)
                ->whereNull('read_at')  // Only update if not already read
                ->update(['read_at' => now()]);

            // Get the state after update to verify changes
            $afterUpdate = DB::table('notifications')
                ->whereIn('id', $notificationIds)
                ->select('id', 'read_at')
                ->get()
                ->keyBy('id')
                ->toArray();

            Log::info('Notification states after update', [
                'after_update' => $afterUpdate,
                'rows_updated' => $result
            ]);

            // Commit the transaction
            DB::commit();

            // If user is provided, invalidate their cache
            if ($user !== null) {
                $this->invalidateUserCache($user->id);
            }

            // Consider the operation successful if we updated at least one row
            // or if all notifications were already marked as read
            $success = $result > 0 || count(array_filter($beforeUpdate, function ($item) {
                return $item->read_at !== null;
            })) === count($notificationIds);

            Log::info('Mark as read operation completed', [
                'success' => $success,
                'rows_updated' => $result
            ]);

            return $success;
        } catch (Exception $e) {
            // Roll back transaction on error
            DB::rollBack();

            Log::error('Error marking notifications as read', [
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'notification_ids' => $notificationIds,
                'user_id' => $user ? $user->id : 'null'
            ]);

            return false;
        }
    }

    /**
     * Mark all notifications as read for a user.
     *
     * @param User $user
     * @return bool
     */
    public function markAllAsRead(User $user)
    {
        try {
            Log::info('Marking all notifications as read for user', [
                'user_id' => $user->id
            ]);

            // Begin transaction
            DB::beginTransaction();

            // Get count of unread notifications before update
            $unreadCount = DB::table('notifications')
                ->where('notifiable_type', get_class($user))
                ->where('notifiable_id', $user->id)
                ->whereNull('read_at')
                ->count();

            Log::info('Unread notifications count before marking all as read', [
                'user_id' => $user->id,
                'unread_count' => $unreadCount
            ]);

            // If no unread notifications, return early
            if ($unreadCount === 0) {
                Log::info('No unread notifications to mark as read', [
                    'user_id' => $user->id
                ]);
                return true;
            }

            // Perform the update
            $result = DB::table('notifications')
                ->where('notifiable_type', get_class($user))
                ->where('notifiable_id', $user->id)
                ->whereNull('read_at')
                ->update(['read_at' => now()]);

            // Verify the update
            $remainingUnread = DB::table('notifications')
                ->where('notifiable_type', get_class($user))
                ->where('notifiable_id', $user->id)
                ->whereNull('read_at')
                ->count();

            Log::info('Mark all as read operation completed', [
                'user_id' => $user->id,
                'rows_updated' => $result,
                'remaining_unread' => $remainingUnread
            ]);

            // Commit transaction
            DB::commit();

            // Invalidate user cache
            $this->invalidateUserCache($user->id);

            return $result > 0;
        } catch (Exception $e) {
            // Roll back transaction on error
            DB::rollBack();

            Log::error('Error marking all notifications as read', [
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => $user->id
            ]);

            return false;
        }
    }

    /**
     * Mark a notification as unread.
     *
     * @param string $notificationId
     * @param User|null $user
     * @return bool
     */
    public function markAsUnread(string $notificationId, ?User $user = null)
    {
        try {
            Log::info('Marking notification as unread', [
                'notification_id' => $notificationId,
                'user_id' => $user ? $user->id : 'null'
            ]);

            // Ensure the notification exists and belongs to the user if specified
            $query = DB::table('notifications')->where('id', $notificationId);

            if ($user !== null) {
                $query->where('notifiable_type', get_class($user))
                    ->where('notifiable_id', $user->id);
            }

            $notification = $query->first();

            if (!$notification) {
                Log::warning('Notification not found or does not belong to user', [
                    'notification_id' => $notificationId,
                    'user_id' => $user ? $user->id : 'null'
                ]);
                return false;
            }

            // Begin transaction
            DB::beginTransaction();

            // Update the notification to mark as unread
            $result = DB::table('notifications')
                ->where('id', $notificationId)
                ->update(['read_at' => null]);

            // Verify the update
            $updatedNotification = DB::table('notifications')
                ->where('id', $notificationId)
                ->first();

            Log::info('Notification state after marking as unread', [
                'notification_id' => $notificationId,
                'read_at' => $updatedNotification->read_at,
                'success' => $updatedNotification->read_at === null
            ]);

            // Commit transaction
            DB::commit();

            // Invalidate cache if user provided
            if ($user !== null) {
                $this->invalidateUserCache($user->id);
            }

            return $result > 0;
        } catch (Exception $e) {
            // Roll back transaction on error
            DB::rollBack();

            Log::error('Error marking notification as unread', [
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'notification_id' => $notificationId,
                'user_id' => $user ? $user->id : 'null'
            ]);

            return false;
        }
    }

    /**
     * Get unread notification count for a user.
     *
     * @param User $user
     * @return int
     */
    public function getUnreadCount(User $user)
    {
        try {
            $cacheKey = "user:{$user->id}:unread_notifications_count";

            return Cache::remember($cacheKey, now()->addMinutes(5), function () use ($user) {
                $count = DB::table('notifications')
                    ->where('notifiable_type', get_class($user))
                    ->where('notifiable_id', $user->id)
                    ->whereNull('read_at')
                    ->count();

                Log::info('Retrieved unread notification count', [
                    'user_id' => $user->id,
                    'count' => $count
                ]);

                return $count;
            });
        } catch (Exception $e) {
            Log::error('Error getting unread notification count', [
                'exception' => $e->getMessage(),
                'user_id' => $user->id
            ]);

            // Return 0 as a safe default
            return 0;
        }
    }

    /**
     * Process a batch of notifications.
     *
     * @param Collection $users
     * @param Notification $notification
     * @return void
     */
    protected function processBatchNotification(Collection $users, Notification $notification)
    {
        try {
            // Filter out users who can't receive notifications
            $eligibleUsers = $users->filter(function ($user) {
                return $this->canReceiveNotifications($user);
            });

            if ($eligibleUsers->isEmpty()) {
                Log::info("No eligible users to receive notification");
                return;
            }

            NotificationFacade::send($eligibleUsers, $notification);

            // Invalidate cache for all users
            $eligibleUsers->each(function ($user) {
                $this->invalidateUserCache($user->id);
            });

            Log::info("Batch notification sent", [
                'notification_type' => get_class($notification),
                'user_count' => $eligibleUsers->count()
            ]);
        } catch (Exception $e) {
            Log::error("Failed to send batch notification", [
                'exception' => $e->getMessage(),
                'notification_type' => get_class($notification)
            ]);
        }
    }

    /**
     * Determine if a user can receive notifications.
     *
     * @param User $user
     * @return bool
     */
    protected function canReceiveNotifications(User $user)
    {
        // Check if user exists and is active
        if (!$user || ($user->deactivated_at !== null)) {
            return false;
        }

        // Additional checks could be added here
        // For example, check if user has opted out of all notifications

        return true;
    }

    /**
     * Invalidate notification cache for a user.
     *
     * @param int $userId
     * @return void
     */
    public function invalidateUserCache(int $userId)
    {
        try {
            Log::info('Invalidating notification cache for user', [
                'user_id' => $userId
            ]);

            Cache::forget("user:{$userId}:unread_notifications_count");
            Cache::forget("user:{$userId}:recent_notifications");
        } catch (Exception $e) {
            Log::error('Error invalidating user notification cache', [
                'exception' => $e->getMessage(),
                'user_id' => $userId
            ]);
        }
    }

    /**
     * Toggle the read status of a notification.
     *
     * @param string $notificationId
     * @param User|null $user
     * @return bool
     */
    public function toggleReadStatus(string $notificationId, ?User $user = null)
    {
        try {
            Log::info('Toggling notification read status', [
                'notification_id' => $notificationId,
                'user_id' => $user ? $user->id : 'null'
            ]);

            // Ensure the notification exists and belongs to the user if specified
            $query = DB::table('notifications')->where('id', $notificationId);

            if ($user !== null) {
                $query->where('notifiable_type', get_class($user))
                    ->where('notifiable_id', $user->id);
            }

            $notification = $query->first();

            if (!$notification) {
                Log::warning('Notification not found or does not belong to user', [
                    'notification_id' => $notificationId,
                    'user_id' => $user ? $user->id : 'null'
                ]);
                return false;
            }

            // Begin transaction
            DB::beginTransaction();

            // Toggle read status based on current status
            if ($notification->read_at === null) {
                // Mark as read
                $result = DB::table('notifications')
                    ->where('id', $notificationId)
                    ->update(['read_at' => now()]);
                $newStatus = 'read';
            } else {
                // Mark as unread
                $result = DB::table('notifications')
                    ->where('id', $notificationId)
                    ->update(['read_at' => null]);
                $newStatus = 'unread';
            }

            // Verify the update
            $updatedNotification = DB::table('notifications')
                ->where('id', $notificationId)
                ->first();

            Log::info('Notification read status toggled', [
                'notification_id' => $notificationId,
                'new_status' => $newStatus,
                'read_at' => $updatedNotification->read_at,
                'success' => ($newStatus === 'read' && $updatedNotification->read_at !== null) ||
                    ($newStatus === 'unread' && $updatedNotification->read_at === null)
            ]);

            // Commit transaction
            DB::commit();

            // Invalidate cache if user provided
            if ($user !== null) {
                $this->invalidateUserCache($user->id);
            }

            return $result > 0;
        } catch (Exception $e) {
            // Roll back transaction on error
            DB::rollBack();

            Log::error('Error toggling notification read status', [
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'notification_id' => $notificationId,
                'user_id' => $user ? $user->id : 'null'
            ]);

            return false;
        }
    }
}
