<?php

namespace App\Services;

use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;

class OrderContextService
{
    /**
     * Store order context in session before authentication
     *
     * @param array $orderData
     * @param int $currentStep
     * @return bool
     */
    public static function storeOrderContext(array $orderData, int $currentStep): bool
    {
        try {
            // Validate input data
            if (empty($orderData) || $currentStep < 1 || $currentStep > 5) {
                return false;
            }

            $context = [
                'data' => $orderData,
                'step' => $currentStep,
                'timestamp' => now()->timestamp,
                'session_id' => Session::getId(),
                'csrf_token' => csrf_token() // Store current CSRF token
            ];

            Session::put('order_context', $context);
            
            Log::info('Order context stored successfully', [
                'step' => $currentStep,
                'session_id' => Session::getId(),
                'has_data' => !empty($orderData)
            ]);
            
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to store order context: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Retrieve order context from session
     *
     * @return array|null
     */
    public static function getOrderContext(): ?array
    {
        $context = Session::get('order_context');
        
        if (!$context) {
            return null;
        }

        // Check if context is still valid (not too old) - increased to 2 hours for order forms
        $maxAge = 120 * 60; // 2 hours - more reasonable for order forms
        if (now()->timestamp - $context['timestamp'] > $maxAge) {
            self::clearOrderContext();
            Log::info('Order context expired and cleared');
            return null;
        }

        // Don't check session ID mismatch during authentication flow
        // Session regeneration happens during login/register, but context should still be valid
        // The session ID check was too strict and caused context to be lost during authentication

        return $context;
    }

    /**
     * Clear order context from session
     *
     * @return void
     */
    public static function clearOrderContext(): void
    {
        Session::forget('order_context');
        Log::info('Order context cleared from session');
    }

    /**
     * Check if order context exists and is valid
     *
     * @return bool
     */
    public static function hasValidOrderContext(): bool
    {
        return self::getOrderContext() !== null;
    }

    /**
     * Get order data from context
     *
     * @return array|null
     */
    public static function getOrderData(): ?array
    {
        $context = self::getOrderContext();
        return $context['data'] ?? null;
    }

    /**
     * Get current step from context
     *
     * @return int
     */
    public static function getCurrentStep(): int
    {
        $context = self::getOrderContext();
        return $context['step'] ?? 1;
    }

    /**
     * Update order context with new data
     *
     * @param array $orderData
     * @param int $currentStep
     * @return bool
     */
    public static function updateOrderContext(array $orderData, int $currentStep): bool
    {
        try {
            $context = self::getOrderContext();
            
            if ($context) {
                $context['data'] = $orderData;
                $context['step'] = $currentStep;
                $context['timestamp'] = now()->timestamp;
                $context['session_id'] = Session::getId(); // Update session ID
                $context['csrf_token'] = csrf_token(); // Update CSRF token
                
                Session::put('order_context', $context);
                
                Log::info('Order context updated successfully', [
                    'step' => $currentStep,
                    'session_id' => Session::getId()
                ]);
                
                return true;
            }
            
            return false;
        } catch (\Exception $e) {
            Log::error('Failed to update order context: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Validate order context data
     *
     * @param array $context
     * @return bool
     */
    public static function validateContext(array $context): bool
    {
        return isset($context['data']) && 
               isset($context['step']) && 
               isset($context['timestamp']) &&
               is_array($context['data']) &&
               is_numeric($context['step']) &&
               $context['step'] >= 1;
    }

    /**
     * Check if order context needs CSRF token refresh
     *
     * @return bool
     */
    public static function needsTokenRefresh(): bool
    {
        $context = self::getOrderContext();
        
        if (!$context) {
            return false;
        }

        // Check if stored CSRF token is different from current
        return isset($context['csrf_token']) && $context['csrf_token'] !== csrf_token();
    }

    /**
     * Refresh CSRF token in order context
     *
     * @return bool
     */
    public static function refreshContextToken(): bool
    {
        try {
            $context = self::getOrderContext();
            
            if ($context) {
                $context['csrf_token'] = csrf_token();
                $context['session_id'] = Session::getId();
                $context['timestamp'] = now()->timestamp;
                
                Session::put('order_context', $context);
                
                Log::info('Order context CSRF token refreshed');
                return true;
            }
            
            return false;
        } catch (\Exception $e) {
            Log::error('Failed to refresh order context token: ' . $e->getMessage());
            return false;
        }
    }
} 