<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderStatusHistory;
use App\Services\WriterPaymentService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class OrderStatusService
{
    protected $writerPaymentService;

    public function __construct(WriterPaymentService $writerPaymentService)
    {
        $this->writerPaymentService = $writerPaymentService;
    }

    /**
     * Change the order status with proper validation and tracking.
     *
     * @param Order $order
     * @param string $newStatus
     * @param string|null $reason
     * @param array $metadata
     * @return bool
     */
    public function changeStatus(Order $order, string $newStatus, ?string $reason = null, array $metadata = []): bool
    {
        // Validate the status transition
        if (!$order->canTransitionTo($newStatus)) {
            throw new \InvalidArgumentException(
                "Invalid status transition from {$order->order_status} to {$newStatus}"
            );
        }

        return DB::transaction(function () use ($order, $newStatus, $reason, $metadata) {
            $oldStatus = $order->order_status;
            
            // Update order status and tracking fields
            $order->order_status = $newStatus;
            $order->status_changed_at = now();
            $order->status_changed_by = Auth::id();
            $order->status_change_reason = $reason;

            // Update specific tracking fields based on status
            $this->updateTrackingFields($order, $newStatus);

            // Save the order
            $order->save();

            // Record the status change in history
            $this->recordStatusChange($order, $oldStatus, $newStatus, $reason, $metadata);

            // Automatically update writer payment status when order is approved
            if ($newStatus === Order::ORDER_STATUS_APPROVED && $order->writer_id) {
                $this->writerPaymentService->updateOrderPaymentStatus($order);
            }

            return true;
        });
    }

    /**
     * Update tracking fields based on the new status.
     *
     * @param Order $order
     * @param string $newStatus
     */
    private function updateTrackingFields(Order $order, string $newStatus): void
    {
        switch ($newStatus) {
            case Order::ORDER_STATUS_ASSIGNED:
                $order->writer_assigned_at = now();
                break;
            
            case Order::ORDER_STATUS_IN_PROGRESS:
                $order->work_started_at = now();
                break;
            
            case Order::ORDER_STATUS_SUBMITTED:
                $order->work_submitted_at = now();
                break;
            
            case Order::ORDER_STATUS_UNDER_REVIEW:
                $order->admin_reviewed_at = now();
                $order->client_review_started_at = now();
                break;
            
            case Order::ORDER_STATUS_APPROVED:
                $order->client_approved_at = now();
                break;
            
            case Order::ORDER_STATUS_REVISION_REQUESTED:
                $order->revision_count = ($order->revision_count ?? 0) + 1;
                break;
        }
    }

    /**
     * Record the status change in history.
     *
     * @param Order $order
     * @param string $oldStatus
     * @param string $newStatus
     * @param string|null $reason
     * @param array $metadata
     */
    private function recordStatusChange(Order $order, string $oldStatus, string $newStatus, ?string $reason, array $metadata): void
    {
        OrderStatusHistory::create([
            'order_id' => $order->id,
            'from_status' => $oldStatus,
            'to_status' => $newStatus,
            'changed_by' => Auth::id(),
            'reason' => $reason,
            'metadata' => $metadata,
        ]);
    }

    /**
     * Automatically change order status when payment is completed.
     *
     * @param Order $order
     * @return bool
     */
    public function handlePaymentCompleted(Order $order): bool
    {
        // Handle transition from pending_payment to bidding (new orders)
        if ($order->payment_status === Order::PAYMENT_STATUS_PAID && $order->order_status === Order::ORDER_STATUS_PENDING_PAYMENT) {
            $statusChanged = $this->changeStatus(
                $order,
                Order::ORDER_STATUS_BIDDING,
                'Payment completed - order now available for bidding'
            );

            // If status change was successful, notify qualified writers
            if ($statusChanged) {
                try {
                    $writerNotificationService = app(\App\Services\WriterNotificationService::class);
                    $writerNotificationService->notifyQualifiedWriters($order);
                } catch (\Exception $e) {
                    \Log::error('Failed to notify writers about new order available for bidding', [
                        'order_id' => $order->id,
                        'order_number' => $order->order_number,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            return $statusChanged;
        }

        // Handle transition from pending to bidding (legacy orders - backward compatibility)
        if ($order->payment_status === Order::PAYMENT_STATUS_PAID && $order->order_status === Order::ORDER_STATUS_PENDING) {
            $statusChanged = $this->changeStatus(
                $order,
                Order::ORDER_STATUS_BIDDING,
                'Payment completed - order now available for bidding'
            );

            // If status change was successful, notify qualified writers
            if ($statusChanged) {
                try {
                    $writerNotificationService = app(\App\Services\WriterNotificationService::class);
                    $writerNotificationService->notifyQualifiedWriters($order);
                } catch (\Exception $e) {
                    \Log::error('Failed to notify writers about new order available for bidding', [
                        'order_id' => $order->id,
                        'order_number' => $order->order_number,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            return $statusChanged;
        }

        // If order is already in bidding status, notify writers about the available order
        if ($order->payment_status === Order::PAYMENT_STATUS_PAID && $order->order_status === Order::ORDER_STATUS_BIDDING) {
            try {
                $writerNotificationService = app(\App\Services\WriterNotificationService::class);
                $writerNotificationService->notifyQualifiedWriters($order);
                Log::info('Writer notifications sent for order already in bidding status', [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number
                ]);
                return true;
            } catch (\Exception $e) {
                Log::error('Failed to send writer notifications for order in bidding status', [
                    'order_id' => $order->id,
                    'error' => $e->getMessage()
                ]);
                return false;
            }
        }

        return false;
    }

    /**
     * Handle bid acceptance and assign writer.
     *
     * @param Order $order
     * @param int $writerId
     * @param int $bidId
     * @return bool
     */
    public function handleBidAccepted(Order $order, int $writerId, int $bidId): bool
    {
        if ($order->order_status !== Order::ORDER_STATUS_BIDDING) {
            throw new \InvalidArgumentException('Order must be in bidding status to accept a bid');
        }

        // Set the writer_id before changing status
        $order->writer_id = $writerId;

        return $this->changeStatus(
            $order,
            Order::ORDER_STATUS_ASSIGNED,
            'Bid accepted - writer assigned to order',
            ['writer_id' => $writerId, 'bid_id' => $bidId]
        );
    }

    /**
     * Handle work submission by writer.
     *
     * @param Order $order
     * @return bool
     */
    public function handleWorkSubmitted(Order $order): bool
    {
        if ($order->order_status !== Order::ORDER_STATUS_IN_PROGRESS) {
            throw new \InvalidArgumentException('Order must be in progress to submit work');
        }

        return $this->changeStatus(
            $order,
            Order::ORDER_STATUS_SUBMITTED,
            'Work submitted by writer - awaiting admin review'
        );
    }

    /**
     * Handle admin review completion.
     *
     * @param Order $order
     * @return bool
     */
    public function handleAdminReviewCompleted(Order $order): bool
    {
        if ($order->order_status !== Order::ORDER_STATUS_SUBMITTED) {
            throw new \InvalidArgumentException('Order must be submitted to complete admin review');
        }

        return $this->changeStatus(
            $order,
            Order::ORDER_STATUS_UNDER_REVIEW,
            'Admin review completed - work sent to client for approval'
        );
    }

    /**
     * Handle client approval.
     *
     * @param Order $order
     * @return bool
     */
    public function handleClientApproval(Order $order): bool
    {
        if ($order->order_status !== Order::ORDER_STATUS_UNDER_REVIEW) {
            throw new \InvalidArgumentException('Order must be under review to be approved');
        }

        return $this->changeStatus(
            $order,
            Order::ORDER_STATUS_APPROVED,
            'Order approved by client'
        );
    }

    /**
     * Handle client rejection and request revision.
     *
     * @param Order $order
     * @param string $revisionNotes
     * @return bool
     */
    public function handleClientRejection(Order $order, string $revisionNotes): bool
    {
        if ($order->order_status !== Order::ORDER_STATUS_UNDER_REVIEW) {
            throw new \InvalidArgumentException('Order must be under review to request revision');
        }

        $order->revision_notes = $revisionNotes;

        return $this->changeStatus(
            $order,
            Order::ORDER_STATUS_REVISION_REQUESTED,
            'Revision requested by client',
            ['revision_notes' => $revisionNotes]
        );
    }

    /**
     * Handle auto-approval after 48 hours.
     *
     * @param Order $order
     * @return bool
     */
    public function handleAutoApproval(Order $order): bool
    {
        if ($order->order_status !== Order::ORDER_STATUS_UNDER_REVIEW) {
            return false;
        }

        $order->auto_approval_at = now();

        return $this->changeStatus(
            $order,
            Order::ORDER_STATUS_APPROVED,
            'Order auto-approved after 48 hours of no client response'
        );
    }

    /**
     * Handle order cancellation.
     *
     * @param Order $order
     * @param string $reason
     * @return bool
     */
    public function handleCancellation(Order $order, string $reason): bool
    {
        if (in_array($order->order_status, [Order::ORDER_STATUS_APPROVED, Order::ORDER_STATUS_CANCELLED])) {
            throw new \InvalidArgumentException('Cannot cancel completed or already cancelled order');
        }

        return $this->changeStatus(
            $order,
            Order::ORDER_STATUS_CANCELLED,
            $reason
        );
    }

    /**
     * Get orders that need auto-approval (48 hours after client review started).
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getOrdersNeedingAutoApproval()
    {
        return Order::where('order_status', Order::ORDER_STATUS_UNDER_REVIEW)
            ->where('client_review_started_at', '<=', now()->subHours(48))
            ->whereNull('auto_approval_at')
            ->get();
    }

    /**
     * Get orders that need 24-hour reminder.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getOrdersNeedingReminder()
    {
        return Order::where('order_status', Order::ORDER_STATUS_UNDER_REVIEW)
            ->where('client_review_started_at', '<=', now()->subHours(24))
            ->where('client_review_started_at', '>', now()->subHours(48))
            ->whereNull('client_approved_at')
            ->get();
    }
} 