<?php

namespace App\Services\Payments;

use Stripe\Stripe;
use Stripe\Webhook;
use App\Models\Order;
use App\Models\Payment;
use Stripe\PaymentIntent;
use App\Models\PaymentMethod;
use Illuminate\Support\Facades\Log;
use Stripe\Exception\ApiErrorException;
use App\Services\AdminNotificationService;
use App\Services\OrderStatusService;
use App\Notifications\AdminPaymentNotification;
use App\Notifications\PaymentFailedNotification;
use App\Notifications\PaymentRefundedNotification;
use App\Notifications\PaymentSuccessfulNotification;

class StripePaymentService extends AbstractPaymentService
{
    /**
     * Create a new Stripe payment service instance.
     *
     * @param PaymentMethod $paymentMethod
     * @param OrderStatusService $orderStatusService
     */
    public function __construct(PaymentMethod $paymentMethod, OrderStatusService $orderStatusService)
    {
        parent::__construct($paymentMethod, $orderStatusService);

        // Initialize Stripe with API key
        $this->initializeStripe();
    }

    /**
     * Initialize Stripe with API key.
     *
     * @return void
     */
    protected function initializeStripe(): void
    {
        $secretKey = $this->paymentMethod->getConfig('secret_key', env('STRIPE_SECRET'));

        if (empty($secretKey)) {
            throw new \RuntimeException('Stripe secret key is not configured');
        }

        Stripe::setApiKey($secretKey);

        // Set API version (optional)
        Stripe::setApiVersion('2022-11-15');
    }

    /**
     * Initialize a payment for an order.
     *
     * @param Order $order
     * @return array Payment initialization data
     */
    public function initializePayment(Order $order): array
    {
        // Create a pending payment record
        $payment = $this->createPaymentRecord($order);

        try {
            // Convert amount to cents (Stripe uses smallest currency unit)
            $amountInCents = (int)($order->net_amount * 100);

            // Create a PaymentIntent with the order's currency
            $paymentIntent = PaymentIntent::create([
                'amount' => $amountInCents,
                'currency' => strtolower($order->currency), // Use order's currency
                'description' => "Order #{$order->order_number}",
                'metadata' => [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                    'payment_id' => $payment->id
                ],
                // Optional automatic payment methods
                'automatic_payment_methods' => [
                    'enabled' => true,
                ],
            ]);

            // Update payment record with Stripe payment intent ID
            $this->updatePaymentRecord(
                $payment,
                Payment::STATUS_PENDING,
                $paymentIntent->id,
                ['stripe_payment_intent' => [
                    'id' => $paymentIntent->id,
                    'client_secret' => $paymentIntent->client_secret
                ]]
            );

            return [
                'success' => true,
                'payment_id' => $payment->id,
                'client_secret' => $paymentIntent->client_secret,
                'payment_intent_id' => $paymentIntent->id,
                'client_config' => $this->getClientConfig($order->currency) // Pass currency to client config
            ];
        } catch (ApiErrorException $e) {
            $this->handlePaymentError($order, $payment, $e);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }






    /**
     * Process a payment after user confirmation.
     *
     * @param Order $order
     * @param array $paymentData
     * @return Payment
     */
    public function processPayment(Order $order, array $paymentData): Payment
    {
        // Find existing payment or create a new one
        $payment = Payment::where('order_id', $order->id)
            ->where('payment_method', $this->paymentMethod->name)
            ->where('status', Payment::STATUS_PENDING)
            ->latest()
            ->first();

        if (!$payment) {
            $payment = $this->createPaymentRecord($order);
        }

        try {
            $paymentIntentId = $paymentData['payment_intent_id'] ?? null;

            if (!$paymentIntentId) {
                throw new \Exception('Stripe payment intent ID is missing');
            }

            // Retrieve the payment intent to check its status
            $paymentIntent = PaymentIntent::retrieve($paymentIntentId);

            if ($paymentIntent->status === 'succeeded') {
                // Verify amount (convert to cents for comparison)
                $amountInCents = (int)($order->net_amount * 100);

                // Verify both amount and currency
                if (
                    $paymentIntent->amount === $amountInCents &&
                    strtolower($paymentIntent->currency) === strtolower($order->currency)
                ) {

                    // Update the payment record
                    $payment = $this->updatePaymentRecord(
                        $payment,
                        Payment::STATUS_COMPLETED,
                        $paymentIntentId,
                        ['stripe_payment_intent' => $paymentIntent->toArray()]
                    );

                    // Update the order
                    $order->payment_status = 'paid';
                    $order->payment_transaction_id = $paymentIntentId;
                    $order->payment_method = 'Stripe';
                    $order->order_status = 'bidding';
                    $order->payment_date = now();
                    $order->save();

                    return $payment;
                } else {
                    // Log the mismatch details for debugging
                    Log::error('Payment verification failed', [
                        'order_id' => $order->id,
                        'expected_amount' => $amountInCents,
                        'actual_amount' => $paymentIntent->amount,
                        'expected_currency' => strtolower($order->currency),
                        'actual_currency' => $paymentIntent->currency
                    ]);

                    throw new \Exception('Payment amount or currency mismatch');
                }
            } else {
                throw new \Exception('Payment not completed: ' . $paymentIntent->status);
            }
        } catch (\Exception $e) {
            return $this->handlePaymentError($order, $payment, $e);
        }
    }
    /**
     * Verify a payment using provider-specific methods.
     *
     * @param string $transactionId
     * @return bool
     */
    public function verifyPayment(string $transactionId): bool
    {
        try {
            $paymentIntent = PaymentIntent::retrieve($transactionId);
            return $paymentIntent->status === 'succeeded';
        } catch (ApiErrorException $e) {
            Log::error('Stripe verification error: ' . $e->getMessage(), [
                'transaction_id' => $transactionId
            ]);
            return false;
        }
    }

    /**
     * Handle webhook events from Stripe.
     *
     * @param array $payload
     * @return bool
     */
    public function handleWebhook(array $payload): bool
    {
        Log::info('Stripe webhook received', ['event_type' => $payload['type'] ?? 'unknown']);

        try {
            // Verify webhook signature
            $webhookSecret = $this->paymentMethod->getConfig('webhook_secret', env('STRIPE_WEBHOOK_SECRET'));

            if (empty($webhookSecret)) {
                Log::warning('Stripe webhook secret is not configured');
            } else {
                $sigHeader = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
                $event = Webhook::constructEvent(
                    file_get_contents('php://input'),
                    $sigHeader,
                    $webhookSecret
                );

                // Replace payload with verified event data
                $payload = $event->toArray();
            }

            $eventType = $payload['type'] ?? '';
            $object = $payload['data']['object'] ?? [];

            if (empty($eventType) || empty($object)) {
                Log::error('Invalid Stripe webhook payload', ['payload' => $payload]);
                return false;
            }

            // Handle different event types
            switch ($eventType) {
                case 'payment_intent.succeeded':
                    return $this->handlePaymentSucceeded($object);

                case 'payment_intent.payment_failed':
                    return $this->handlePaymentFailed($object);

                case 'charge.refunded':
                    return $this->handleChargeRefunded($object);

                default:
                    Log::info('Unhandled Stripe webhook event', ['event_type' => $eventType]);
                    return true; // Return true for unhandled events to acknowledge receipt
            }
        } catch (\Exception $e) {
            Log::error('Error processing Stripe webhook: ' . $e->getMessage(), [
                'payload' => $payload,
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Handle a successful payment intent webhook event.
     *
     * @param array $object
     * @return bool
     */
    protected function handlePaymentSucceeded(array $object): bool
    {
        $paymentIntentId = $object['id'] ?? null;

        if (!$paymentIntentId) {
            Log::error('Missing payment intent ID in Stripe webhook', ['object' => $object]);
            return false;
        }

        // Find the payment by transaction ID
        $payment = Payment::where('transaction_id', $paymentIntentId)->first();

        if (!$payment) {
            // Try to find by metadata
            $orderId = $object['metadata']['order_id'] ?? null;

            if ($orderId) {
                $payment = Payment::where('order_id', $orderId)
                    ->where('payment_method', $this->paymentMethod->name)
                    ->latest()
                    ->first();
            }

            if (!$payment) {
                Log::error('Payment not found for Stripe payment intent', [
                    'payment_intent_id' => $paymentIntentId,
                    'order_id' => $orderId ?? 'not provided'
                ]);
                return false;
            }
        }

        // Update the payment status
        $this->updatePaymentRecord(
            $payment,
            Payment::STATUS_COMPLETED,
            $paymentIntentId,
            ['webhook_data' => $object]
        );

        // Get the order and user
        $order = $payment->order;
        if ($order && $order->user) {
            // Notify the user
            $order->user->notify(new PaymentSuccessfulNotification($order, $payment));

            // Notify admins
            AdminNotificationService::notifyAllAdmins(
                new AdminPaymentNotification($order, $payment, $order->user->name)
            );
        }

        return true;
    }

    /**
     * Handle a failed payment intent webhook event.
     *
     * @param array $object
     * @return bool
     */
    protected function handlePaymentFailed(array $object): bool
    {
        $paymentIntentId = $object['id'] ?? null;

        if (!$paymentIntentId) {
            Log::error('Missing payment intent ID in Stripe webhook', ['object' => $object]);
            return false;
        }

        // Find the payment by transaction ID
        $payment = Payment::where('transaction_id', $paymentIntentId)->first();

        if ($payment) {
            // Extract error message from the webhook payload
            $errorMessage = $object['last_payment_error']['message'] ?? 'Payment failed';

            // Update the payment status
            $this->updatePaymentRecord(
                $payment,
                Payment::STATUS_FAILED,
                $paymentIntentId,
                [
                    'webhook_data' => $object,
                    'error' => $errorMessage
                ]
            );

            // Get the order and user
            $order = $payment->order;
            if ($order && $order->user) {
                // Notify the user
                $order->user->notify(new PaymentFailedNotification($order, $payment, $errorMessage));

                // Notify admins
                AdminNotificationService::notifyAllAdmins(
                    new AdminPaymentNotification($order, $payment, $order->user->name, true, $errorMessage)
                );
            }

            return true;
        }

        Log::error('Payment not found for Stripe payment intent', ['payment_intent_id' => $paymentIntentId]);
        return false;
    }


    /**
     * Handle a refunded charge webhook event.
     *
     * @param array $object
     * @return bool
     */
    // protected function handleChargeRefunded(array $object): bool
    // {
    //     $chargeId = $object['id'] ?? null;
    //     $paymentIntentId = $object['payment_intent'] ?? null;

    //     if (!$paymentIntentId && !$chargeId) {
    //         Log::error('Missing payment intent ID in Stripe webhook', ['object' => $object]);
    //         return false;
    //     }

    //     // Find the payment by transaction ID (payment intent ID)
    //     $payment = null;

    //     if ($paymentIntentId) {
    //         $payment = Payment::where('transaction_id', $paymentIntentId)->first();
    //     }

    //     if (!$payment && $chargeId) {
    //         // Try to find by charge ID in metadata
    //         $payment = Payment::whereJsonContains('metadata->stripe_charge_id', $chargeId)->first();
    //     }

    //     if ($payment) {
    //         // Update the payment status
    //         $this->updatePaymentRecord(
    //             $payment,
    //             Payment::STATUS_REFUNDED,
    //             $payment->transaction_id, // Keep the original transaction ID
    //             ['webhook_data' => $object]
    //         );

    //         return true;
    //     }

    //     Log::error('Payment not found for Stripe refund', [
    //         'charge_id' => $chargeId,
    //         'payment_intent_id' => $paymentIntentId
    //     ]);
    //     return false;
    // }

    protected function handleChargeRefunded(array $object): bool
    {
        $chargeId = $object['id'] ?? null;
        $paymentIntentId = $object['payment_intent'] ?? null;

        if (!$paymentIntentId && !$chargeId) {
            Log::error('Missing payment intent ID in Stripe webhook', ['object' => $object]);
            return false;
        }

        // Find the payment by transaction ID (payment intent ID)
        $payment = null;

        if ($paymentIntentId) {
            $payment = Payment::where('transaction_id', $paymentIntentId)->first();
        }

        if (!$payment && $chargeId) {
            // Try to find by charge ID in metadata
            $payment = Payment::whereJsonContains('metadata->stripe_charge_id', $chargeId)->first();
        }

        if ($payment) {
            // Get refund amount
            $refundAmount = $object['amount_refunded'] / 100; // Convert from cents
            $isFullRefund = $object['refunded'] === true;

            // Get refund reason if available
            $reason = $object['refunds']['data'][0]['reason'] ?? '';

            // Update the payment status
            $this->updatePaymentRecord(
                $payment,
                $isFullRefund ? Payment::STATUS_REFUNDED : Payment::STATUS_PARTIALLY_REFUNDED,
                $payment->transaction_id, // Keep the original transaction ID
                ['webhook_data' => $object]
            );

            // Get the order and user to send notifications
            $order = $payment->order;
            if ($order && $order->user) {
                try {
                    // Notify the user about the refund
                    $order->user->notify(new PaymentRefundedNotification(
                        $order,
                        $payment,
                        $refundAmount,
                        $isFullRefund,
                        $reason
                    ));

                    // Notify admins
                    AdminNotificationService::notifyAllAdmins(
                        new AdminPaymentNotification(
                            $order,
                            $payment,
                            $order->user->name,
                            false,
                            "Refund processed: " . ($isFullRefund ? "Full" : "Partial") . " - $refundAmount"
                        )
                    );

                    Log::info('Refund notifications sent for Stripe payment', [
                        'payment_id' => $payment->id,
                        'order_id' => $order->id,
                        'amount' => $refundAmount,
                        'is_full_refund' => $isFullRefund
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to send refund notifications', [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            }

            return true;
        }

        Log::error('Payment not found for Stripe refund', [
            'charge_id' => $chargeId,
            'payment_intent_id' => $paymentIntentId
        ]);
        return false;
    }


    /**
     * Get the client-side configuration needed for Stripe.
     *
     * @param string|null $currency Currency code to use (defaults to USD)
     * @return array
     */
    public function getClientConfig(?string $currency = null): array
    {
        return [
            'publishableKey' => $this->paymentMethod->getConfig('publishable_key', env('STRIPE_KEY')),
            'currency' => strtolower($currency ?? 'usd'),
        ];
    }





    /**
     * Test connection with Stripe API.
     *
     * @return array
     */
    public function testConnection(): array
    {
        try {
            // Check if required config values exist
            $secretKey = $this->paymentMethod->getConfig('secret_key', env('STRIPE_SECRET'));

            if (empty($secretKey)) {
                return [
                    'success' => false,
                    'message' => 'Missing required Stripe API secret key',
                ];
            }

            // Initialize Stripe with API key
            \Stripe\Stripe::setApiKey($secretKey);

            // Make a simple API call to check connectivity
            $balance = \Stripe\Balance::retrieve();

            if ($balance && isset($balance->available)) {
                return [
                    'success' => true,
                    'message' => 'Successfully connected to Stripe API',
                    'details' => [
                        'available_currencies' => count($balance->available),
                        'default_currency' => count($balance->available) > 0 ? $balance->available[0]->currency : 'N/A'
                    ],
                ];
            }

            return [
                'success' => true,
                'message' => 'Connected to Stripe API but no balance information available',
            ];
        } catch (\Exception $e) {
            Log::error('Stripe connection test failed', [
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Stripe connection test failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Process a refund for a payment.
     *
     * @param Payment $payment
     * @param float $amount
     * @param string $reason
     * @return array
     */
    public function processRefund(Payment $payment, float $amount, string $reason): array
    {
        try {
            // Verify the payment belongs to Stripe
            if ($payment->payment_method !== PaymentMethod::STRIPE) {
                return [
                    'success' => false,
                    'message' => 'Invalid payment method for refund'
                ];
            }

            // Check if payment can be refunded (uses our enhanced canBeRefunded method)
            if (!$payment->canBeRefunded()) {
                return [
                    'success' => false,
                    'message' => 'This payment cannot be refunded. Current status: ' . $payment->status
                ];
            }

            // Validate refund amount against available refundable amount
            $refundableAmount = $payment->getRefundableAmount();
            if ($amount > $refundableAmount) {
                return [
                    'success' => false,
                    'message' => "Cannot refund more than the available amount ({$refundableAmount})"
                ];
            }

            // Original payment intent ID is required for refund
            $paymentIntentId = $payment->transaction_id;
            if (empty($paymentIntentId)) {
                return [
                    'success' => false,
                    'message' => 'Missing transaction ID for refund'
                ];
            }

            // Initialize Stripe
            \Stripe\Stripe::setApiKey($this->paymentMethod->getConfig('secret_key', env('STRIPE_SECRET')));

            // Convert amount to cents for Stripe
            $amountInCents = (int)($amount * 100);

            // Create refund
            $refund = \Stripe\Refund::create([
                'payment_intent' => $paymentIntentId,
                'amount' => $amountInCents,
                'reason' => 'requested_by_customer',
                'metadata' => [
                    'reason' => substr($reason, 0, 500), // Limit reason length
                    'refunded_by' => 'admin',
                    'order_id' => $payment->order->id ?? null,
                    'payment_id' => $payment->id
                ]
            ]);

            if ($refund && $refund->status === 'succeeded') {
                Log::info('Stripe refund processed successfully', [
                    'payment_id' => $payment->id,
                    'refund_id' => $refund->id,
                    'amount' => $amount,
                    'is_full_refund' => ($amount >= $payment->amount)
                ]);

                // Update payment status based on refund amount
                $payment->updateStatusAfterRefund($amount);

                // Update order payment status if order exists
                if ($payment->order) {
                    $payment->order->updatePaymentStatusAfterRefund($payment, $amount);
                }

                return [
                    'success' => true,
                    'message' => 'Refund processed successfully',
                    'transaction_id' => $refund->id,
                    'details' => $refund->toArray(),
                    'is_full_refund' => ($amount >= $payment->amount)
                ];
            } else {
                Log::error('Stripe refund failed or pending', [
                    'payment_id' => $payment->id,
                    'refund' => $refund->toArray()
                ]);

                return [
                    'success' => false,
                    'message' => 'Stripe refund failed or pending: ' . $refund->status,
                    'details' => $refund->toArray()
                ];
            }
        } catch (\Stripe\Exception\CardException $e) {
            // Card errors (e.g., insufficient funds)
            Log::error('Stripe card error: ' . $e->getMessage(), [
                'payment_id' => $payment->id,
                'amount' => $amount,
                'error_code' => $e->getStripeCode(),
                'error_type' => 'card_error'
            ]);

            return [
                'success' => false,
                'message' => 'Card error: ' . $e->getMessage(),
                'error_type' => 'card_error'
            ];
        } catch (\Stripe\Exception\RateLimitException $e) {
            // Too many requests made to the API too quickly
            Log::error('Stripe rate limit error: ' . $e->getMessage(), [
                'payment_id' => $payment->id,
                'amount' => $amount
            ]);

            return [
                'success' => false,
                'message' => 'Rate limit error, please try again later',
                'error_type' => 'rate_limit'
            ];
        } catch (\Stripe\Exception\InvalidRequestException $e) {
            // Invalid parameters were supplied to Stripe's API
            Log::error('Stripe invalid request: ' . $e->getMessage(), [
                'payment_id' => $payment->id,
                'amount' => $amount,
                'param' => $e->getStripeParam()
            ]);

            return [
                'success' => false,
                'message' => 'Invalid request: ' . $e->getMessage(),
                'error_type' => 'invalid_request'
            ];
        } catch (\Stripe\Exception\AuthenticationException $e) {
            // Authentication with Stripe's API failed
            Log::error('Stripe authentication error: ' . $e->getMessage(), [
                'payment_id' => $payment->id
            ]);

            return [
                'success' => false,
                'message' => 'Authentication with Stripe failed',
                'error_type' => 'authentication_error'
            ];
        } catch (\Stripe\Exception\ApiConnectionException $e) {
            // Network communication with Stripe failed
            Log::error('Stripe API connection error: ' . $e->getMessage(), [
                'payment_id' => $payment->id
            ]);

            return [
                'success' => false,
                'message' => 'Network communication with Stripe failed',
                'error_type' => 'api_connection_error'
            ];
        } catch (\Stripe\Exception\ApiErrorException $e) {
            // Generic API error
            Log::error('Stripe API error: ' . $e->getMessage(), [
                'payment_id' => $payment->id,
                'amount' => $amount,
                'http_status' => $e->getHttpStatus()
            ]);

            return [
                'success' => false,
                'message' => 'Stripe API error: ' . $e->getMessage(),
                'error_type' => 'api_error'
            ];
        } catch (\Exception $e) {
            // Any other error
            Log::error('Stripe refund error: ' . $e->getMessage(), [
                'payment_id' => $payment->id,
                'amount' => $amount,
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'message' => 'Error processing Stripe refund: ' . $e->getMessage(),
                'error_type' => 'unknown_error'
            ];
        }
    }
}
