<?php

namespace App\Services;

use App\Models\Payment;
use App\Models\Refund;
use App\Services\Payments\PaymentServiceFactory;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class RefundService
{
    /**
     * Validate that a refund can be processed for the given payment and amount.
     *
     * @param Payment $payment
     * @param float $amount
     * @return array
     */
    public function validateRefund(Payment $payment, float $amount): array
    {
        // Check if payment can be refunded
        if (!$payment->canBeRefunded()) {
            return [
                'valid' => false,
                'message' => "This payment cannot be refunded. Current status: {$payment->status}"
            ];
        }

        // Validate refund amount
        $refundableAmount = $payment->getRefundableAmount();
        if ($amount <= 0) {
            return [
                'valid' => false,
                'message' => 'Refund amount must be greater than zero'
            ];
        }

        if ($amount > $refundableAmount) {
            return [
                'valid' => false,
                'message' => "Cannot refund more than the available amount ({$refundableAmount})"
            ];
        }

        return ['valid' => true];
    }

    /**
     * Process a refund for a payment.
     *
     * @param Payment $payment
     * @param float $amount
     * @param string $reason
     * @param string|null $notes
     * @param int|null $processedBy
     * @return array
     */
    public function processRefund(
        Payment $payment,
        float $amount,
        string $reason,
        ?string $notes = null,
        ?int $processedBy = null
    ): array {
        // Validate the refund
        $validation = $this->validateRefund($payment, $amount);
        if (!$validation['valid']) {
            return [
                'success' => false,
                'message' => $validation['message']
            ];
        }

        // Use a database transaction for the entire refund process
        DB::beginTransaction();

        try {
            // Create a pending refund record
            $refund = new Refund([
                'payment_id' => $payment->id,
                'amount' => $amount,
                'currency' => $payment->currency ?? 'USD',
                'status' => Refund::STATUS_PENDING,
                'reason' => $reason,
                'notes' => $notes,
                'processed_by' => $processedBy ?? Auth::id(),
            ]);
            $refund->save();

            // Get the appropriate payment service
            $paymentService = PaymentServiceFactory::create($payment->payment_method);

            // Process the refund through the payment gateway
            $result = $paymentService->processRefund($payment, $amount, $reason);

            if ($result['success']) {
                // Update refund record with success details
                $refund->markAsCompleted(
                    $result['transaction_id'] ?? null,
                    ['gateway_response' => $result['details'] ?? []]
                );

                // Log the successful refund
                Log::info('Refund processed successfully', [
                    'payment_id' => $payment->id,
                    'refund_id' => $refund->id,
                    'amount' => $amount,
                    'is_full_refund' => $result['is_full_refund'] ?? ($amount >= $payment->amount),
                    'payment_status' => $payment->status,
                    'order_payment_status' => $payment->order->payment_status ?? null
                ]);

                // Commit the transaction
                DB::commit();

                return [
                    'success' => true,
                    'message' => 'Refund processed successfully',
                    'refund' => $refund,
                    'payment' => $payment->fresh(),
                    'order' => $payment->order ? $payment->order->fresh() : null
                ];
            } else {
                // Mark refund as failed
                $refund->markAsFailed(
                    $result['message'] ?? 'Unknown error',
                    ['gateway_response' => $result['details'] ?? []]
                );

                // Commit the transaction - we still want to save the failed refund record
                DB::commit();

                return [
                    'success' => false,
                    'message' => $result['message'] ?? 'Failed to process refund',
                    'refund_id' => $refund->id,
                    'error_type' => $result['error_type'] ?? 'unknown'
                ];
            }
        } catch (\Exception $e) {
            // Rollback the transaction
            DB::rollBack();

            Log::error('Refund processing error', [
                'payment_id' => $payment->id,
                'amount' => $amount,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return [
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Get the refund history for a payment.
     *
     * @param Payment $payment
     * @return array
     */
    public function getRefundHistory(Payment $payment): array
    {
        $refunds = $payment->refunds()
            ->with('processor:id,name')
            ->latest()
            ->get()
            ->map(function ($refund) {
                return [
                    'id' => $refund->id,
                    'amount' => $refund->amount,
                    'currency' => $refund->currency,
                    'status' => $refund->status,
                    'reason' => $refund->reason,
                    'notes' => $refund->notes,
                    'processed_at' => $refund->processed_at,
                    'processor' => $refund->processor ? [
                        'id' => $refund->processor->id,
                        'name' => $refund->processor->name,
                    ] : null,
                    'created_at' => $refund->created_at,
                    'is_full_refund' => $refund->isFullRefund(),
                    'refund_percentage' => round($refund->getRefundPercentage(), 2)
                ];
            });

        return [
            'refunds' => $refunds,
            'payment' => [
                'id' => $payment->id,
                'amount' => $payment->amount,
                'status' => $payment->status,
                'refunded_amount' => $payment->getRefundedAmount(),
                'refundable_amount' => $payment->getRefundableAmount(),
                'is_fully_refunded' => $payment->isFullyRefunded(),
                'can_be_refunded' => $payment->canBeRefunded(),
            ],
            'order' => $payment->order ? [
                'id' => $payment->order->id,
                'payment_status' => $payment->order->payment_status,
                'order_status' => $payment->order->order_status,
                'is_refunded' => $payment->order->isRefunded(),
                'is_partially_refunded' => $payment->order->isPartiallyRefunded()
            ] : null,
        ];
    }

    /**
     * Get detailed refund information for an order.
     *
     * @param int $orderId
     * @return array
     */
    public function getOrderRefundInfo(int $orderId): array
    {
        // Load the order with payments and refunds
        $order = \App\Models\Order::with([
            'payments.refunds' => function ($query) {
                $query->where('status', Refund::STATUS_COMPLETED);
            }
        ])->findOrFail($orderId);

        // Default refund info
        $refundInfo = [
            'has_refunds' => false,
            'refunded_amount' => 0,
            'refund_date' => null,
            'refund_reason' => null,
            'is_fully_refunded' => $order->isRefunded(),
            'is_partially_refunded' => $order->isPartiallyRefunded(),
            'refunds' => []
        ];

        // If order has payments with refunds
        if ($order->payments->isNotEmpty()) {
            $refundedAmount = 0;
            $latestRefundDate = null;
            $refunds = [];

            foreach ($order->payments as $payment) {
                if ($payment->refunds->isEmpty()) {
                    continue;
                }

                // Calculate total refunded for this payment
                $paymentRefundedAmount = $payment->getRefundedAmount();
                $refundedAmount += $paymentRefundedAmount;

                // Track latest refund
                foreach ($payment->refunds as $refund) {
                    // Add refund details
                    $refunds[] = [
                        'id' => $refund->id,
                        'amount' => (float) $refund->amount,
                        'reason' => $refund->reason,
                        'notes' => $refund->notes,
                        'status' => $refund->status,
                        'date' => $refund->processed_at ? $refund->processed_at->toDateTimeString() : $refund->created_at->toDateTimeString(),
                        'currency' => $refund->currency ?? $payment->currency ?? ($payment->order->currency ?? 'USD'),
                        'payment_id' => $payment->id,
                        'processor' => $refund->processor ? [
                            'id' => $refund->processor->id,
                            'name' => $refund->processor->name
                        ] : null
                    ];

                    // Update latest refund date
                    if (!$latestRefundDate || ($refund->processed_at && $refund->processed_at->isAfter($latestRefundDate))) {
                        $latestRefundDate = $refund->processed_at;
                        $refundInfo['refund_reason'] = $refund->reason;
                    }
                }
            }

            // Update refund info
            $refundInfo['has_refunds'] = count($refunds) > 0;
            $refundInfo['refunded_amount'] = $refundedAmount;
            $refundInfo['refund_date'] = $latestRefundDate ? $latestRefundDate->toDateTimeString() : null;
            $refundInfo['refunds'] = $refunds;
        }

        return $refundInfo;
    }
}
