<?php

namespace App\Services;

use App\Models\OrderRevision;
use App\Models\Order;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class RevisionQualityAssessmentService
{
    /**
     * Assess the quality of a completed revision and calculate payment impacts.
     */
    public function assessQuality(OrderRevision $revision, int $qualityRating, ?string $qualityNotes = null): array
    {
        try {
            // Validate quality rating
            if ($qualityRating < 1 || $qualityRating > 10) {
                throw new \InvalidArgumentException('Quality rating must be between 1 and 10');
            }

            // Calculate payment impacts based on quality rating
            $paymentImpacts = $this->calculatePaymentImpacts($revision, $qualityRating);
            
            // Calculate delay penalties if applicable
            $delayPenalty = $this->calculateDelayPenalty($revision);
            
            // Update the revision with assessment results
            $revision->update([
                'quality_rating' => $qualityRating,
                'quality_notes' => $qualityNotes,
                'quality_bonus' => $paymentImpacts['bonus'],
                'quality_penalty' => $paymentImpacts['penalty'],
                'payment_delay_hours' => $delayPenalty['hours'],
                'payment_impact' => $paymentImpacts['total'] + $delayPenalty['amount'],
            ]);

            // Log the assessment
            Log::info('Revision quality assessed', [
                'revision_id' => $revision->id,
                'order_id' => $revision->order_id,
                'writer_id' => $revision->assigned_to,
                'quality_rating' => $qualityRating,
                'bonus' => $paymentImpacts['bonus'],
                'penalty' => $paymentImpacts['penalty'],
                'delay_penalty' => $delayPenalty['amount'],
                'total_impact' => $revision->payment_impact,
            ]);

            return [
                'success' => true,
                'quality_rating' => $qualityRating,
                'bonus' => $paymentImpacts['bonus'],
                'penalty' => $paymentImpacts['penalty'],
                'delay_penalty' => $delayPenalty['amount'],
                'total_impact' => $revision->payment_impact,
                'message' => 'Quality assessment completed successfully',
            ];

        } catch (\Exception $e) {
            Log::error('Failed to assess revision quality', [
                'revision_id' => $revision->id,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Failed to assess quality: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Calculate payment impacts based on quality rating.
     */
    private function calculatePaymentImpacts(OrderRevision $revision, int $qualityRating): array
    {
        $qualityBonus = config('revisions.payment_impact.quality_bonus', 5.00);
        $qualityPenalty = config('revisions.payment_impact.quality_penalty', 10.00);
        
        $bonus = 0;
        $penalty = 0;
        
        // Quality rating 9-10: Full bonus
        if ($qualityRating >= 9) {
            $bonus = $qualityBonus;
        }
        // Quality rating 7-8: No impact
        elseif ($qualityRating >= 7) {
            // No bonus or penalty
        }
        // Quality rating 5-6: 60% penalty
        elseif ($qualityRating >= 5) {
            $penalty = $qualityPenalty * 0.6;
        }
        // Quality rating 1-4: Full penalty
        else {
            $penalty = $qualityPenalty;
        }
        
        return [
            'bonus' => $bonus,
            'penalty' => $penalty,
            'total' => $bonus - $penalty,
        ];
    }

    /**
     * Calculate delay penalty if revision is overdue.
     */
    private function calculateDelayPenalty(OrderRevision $revision): array
    {
        if (!$revision->isOverdue() || $revision->status !== 'completed') {
            return ['hours' => 0, 'amount' => 0];
        }
        
        $overdueHours = $revision->getOverdueHours();
        $delayPenaltyPerHour = config('revisions.payment_impact.delay_penalty_per_hour', 0.50);
        
        $penaltyAmount = $overdueHours * $delayPenaltyPerHour;
        
        return [
            'hours' => $overdueHours,
            'amount' => $penaltyAmount,
        ];
    }

    /**
     * Reject a revision and apply penalties.
     */
    public function rejectRevision(OrderRevision $revision, string $rejectionReason, ?string $adminNotes = null): array
    {
        try {
            // Apply rejection penalty (full quality penalty)
            $qualityPenalty = config('revisions.payment_impact.quality_penalty', 10.00);
            
            $revision->update([
                'status' => 'rejected',
                'rejected_at' => now(),
                'quality_rating' => 1, // Lowest rating for rejected revisions
                'quality_penalty' => $qualityPenalty,
                'quality_notes' => $rejectionReason,
                'admin_notes' => $adminNotes,
                'payment_impact' => -$qualityPenalty, // Negative impact
            ]);

            Log::info('Revision rejected', [
                'revision_id' => $revision->id,
                'order_id' => $revision->order_id,
                'writer_id' => $revision->assigned_to,
                'rejection_reason' => $rejectionReason,
                'penalty_applied' => $qualityPenalty,
            ]);

            return [
                'success' => true,
                'message' => 'Revision rejected successfully',
                'penalty_applied' => $qualityPenalty,
            ];

        } catch (\Exception $e) {
            Log::error('Failed to reject revision', [
                'revision_id' => $revision->id,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Failed to reject revision: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get quality assessment statistics for a writer.
     */
    public function getWriterQualityStats(int $writerId): array
    {
        $revisions = OrderRevision::where('assigned_to', $writerId)
            ->whereNotNull('quality_rating')
            ->get();

        if ($revisions->isEmpty()) {
            return [
                'total_revisions' => 0,
                'average_rating' => 0,
                'total_bonus' => 0,
                'total_penalty' => 0,
                'net_impact' => 0,
                'quality_distribution' => [],
            ];
        }

        $totalRevisions = $revisions->count();
        $averageRating = $revisions->avg('quality_rating');
        $totalBonus = $revisions->sum('quality_bonus');
        $totalPenalty = $revisions->sum('quality_penalty');
        $netImpact = $totalBonus - $totalPenalty;

        // Quality distribution
        $qualityDistribution = [];
        for ($i = 1; $i <= 10; $i++) {
            $count = $revisions->where('quality_rating', $i)->count();
            if ($count > 0) {
                $qualityDistribution[$i] = $count;
            }
        }

        return [
            'total_revisions' => $totalRevisions,
            'average_rating' => round($averageRating, 2),
            'total_bonus' => $totalBonus,
            'total_penalty' => $totalPenalty,
            'net_impact' => $netImpact,
            'quality_distribution' => $qualityDistribution,
        ];
    }

    /**
     * Get payment impact summary for an order.
     */
    public function getOrderPaymentImpactSummary(Order $order): array
    {
        $revisions = $order->revisions()->whereNotNull('quality_rating')->get();
        
        $totalBonus = $revisions->sum('quality_bonus');
        $totalPenalty = $revisions->sum('quality_penalty');
        $totalDelayPenalty = $revisions->sum('payment_delay_hours');
        $netImpact = $totalBonus - $totalPenalty - $totalDelayPenalty;

        return [
            'total_revisions_assessed' => $revisions->count(),
            'total_bonus' => $totalBonus,
            'total_penalty' => $totalPenalty,
            'total_delay_penalty' => $totalDelayPenalty,
            'net_impact' => $netImpact,
            'revisions' => $revisions->map(function ($revision) {
                return [
                    'id' => $revision->id,
                    'type' => $revision->revision_type,
                    'quality_rating' => $revision->quality_rating,
                    'bonus' => $revision->quality_bonus,
                    'penalty' => $revision->quality_penalty,
                    'delay_penalty' => $revision->payment_delay_hours,
                    'total_impact' => $revision->payment_impact,
                    'completed_at' => $revision->completed_at,
                ];
            }),
        ];
    }
} 