<?php

namespace App\Services;

use App\Models\Order;
use App\Models\User;
use App\Models\WriterPaymentRecord;
use App\Models\WriterWithdrawalRequest;
use Illuminate\Support\Facades\DB;

class WriterPaymentService
{
    /**
     * Calculate total pending amount for a writer.
     */
    public function calculatePendingAmount($writerId): float
    {
        return Order::where('writer_id', $writerId)
            ->whereIn('writer_payment_status', Order::WRITER_PENDING_STATUSES)
            ->sum('writer_amount') ?? 0.0;
    }

    /**
     * Calculate total available amount for a writer.
     */
    public function calculateAvailableAmount($writerId): float
    {
        return Order::where('writer_id', $writerId)
            ->whereIn('writer_payment_status', Order::WRITER_AVAILABLE_STATUSES)
            ->where('writer_payment_status', '!=', Order::WRITER_PAYMENT_STATUS_WITHDRAWN)
            ->sum('writer_amount') ?? 0.0;
    }

    /**
     * Get pending orders for a writer.
     */
    public function getPendingOrders($writerId)
    {
        return Order::where('writer_id', $writerId)
            ->whereIn('writer_payment_status', Order::WRITER_PENDING_STATUSES)
            ->with(['user'])
            ->orderBy('id', 'desc')
            ->get();
    }

    /**
     * Get available orders for a writer.
     */
    public function getAvailableOrders($writerId)
    {
        return Order::where('writer_id', $writerId)
            ->whereIn('writer_payment_status', Order::WRITER_AVAILABLE_STATUSES)
            ->where('writer_payment_status', '!=', Order::WRITER_PAYMENT_STATUS_WITHDRAWN)
            ->with(['user'])
            ->orderBy('writer_payment_released_at', 'desc')
            ->orderBy('id', 'desc')
            ->get();
    }

    /**
     * Get writer payment summary.
     */
    public function getWriterPaymentSummary($writerId): array
    {
        $pendingAmount = $this->calculatePendingAmount($writerId);
        $availableAmount = $this->calculateAvailableAmount($writerId);
        
        // Get total withdrawn amount
        $withdrawnAmount = Order::where('writer_id', $writerId)
            ->where('writer_payment_status', Order::WRITER_PAYMENT_STATUS_WITHDRAWN)
            ->sum('writer_amount') ?? 0.0;

        // Get total earnings (all time)
        $totalEarnings = Order::where('writer_id', $writerId)
            ->whereNotNull('writer_amount')
            ->sum('writer_amount') ?? 0.0;

        return [
            'pending_amount' => $pendingAmount,
            'available_amount' => $availableAmount,
            'withdrawn_amount' => $withdrawnAmount,
            'total_earnings' => $totalEarnings,
            'pending_orders_count' => $this->getPendingOrders($writerId)->count(),
            'available_orders_count' => $this->getAvailableOrders($writerId)->count(),
        ];
    }

    /**
     * Update writer payment status for an order when order status changes.
     */
    public function updateOrderPaymentStatus(Order $order)
    {
        if (!$order->writer_id) {
            return;
        }

        $order->updateWriterPaymentStatus();
        
        // Create or update payment record
        $paymentRecord = $order->latestWriterPaymentRecord;
        
        if (!$paymentRecord) {
            $order->createWriterPaymentRecord();
        } else {
            // Update existing payment record status
            $this->updatePaymentRecordStatus($paymentRecord, $order);
        }
    }

    /**
     * Update payment record status based on order.
     */
    private function updatePaymentRecordStatus(WriterPaymentRecord $paymentRecord, Order $order)
    {
        $newStatus = $this->mapOrderStatusToPaymentRecordStatus($order->writer_payment_status);
        
        if ($newStatus !== $paymentRecord->status) {
            $paymentRecord->status = $newStatus;
            
            // Set timestamps based on status
            if ($newStatus === WriterPaymentRecord::STATUS_AVAILABLE && !$paymentRecord->available_at) {
                $paymentRecord->available_at = now();
            }
            
            if ($order->order_status === Order::ORDER_STATUS_APPROVED && !$paymentRecord->earned_at) {
                $paymentRecord->earned_at = now();
            }
            
            $paymentRecord->save();
        }
    }

    /**
     * Map order payment status to payment record status.
     */
    private function mapOrderStatusToPaymentRecordStatus(string $orderPaymentStatus): string
    {
        switch ($orderPaymentStatus) {
            case Order::WRITER_PAYMENT_STATUS_AVAILABLE:
                return WriterPaymentRecord::STATUS_AVAILABLE;
            case Order::WRITER_PAYMENT_STATUS_WITHDRAWN:
                return WriterPaymentRecord::STATUS_WITHDRAWN;
            case Order::WRITER_PAYMENT_STATUS_DISPUTED:
                return WriterPaymentRecord::STATUS_DISPUTED;
            default:
                return WriterPaymentRecord::STATUS_PENDING;
        }
    }

    /**
     * Check if writer can make a withdrawal request.
     */
    public function canWriterWithdraw($writerId): array
    {
        $availableAmount = $this->calculateAvailableAmount($writerId);
        $nextWithdrawalDate = WriterWithdrawalRequest::getNextWithdrawalDate();
        $now = now();
        $minimumWithdrawal = 5.00; // Minimum $5 requirement
        
        // Check if today is a valid withdrawal date
        $isValidDate = WriterWithdrawalRequest::isValidWithdrawalDate($now);
        
        // Check if writer has pending withdrawal requests
        $hasPendingRequest = WriterWithdrawalRequest::where('writer_id', $writerId)
            ->where('status', WriterWithdrawalRequest::STATUS_REQUESTED)
            ->exists();

        // Check if available amount meets minimum requirement
        $meetsMinimum = $availableAmount >= $minimumWithdrawal;

        return [
            'can_withdraw' => $availableAmount > 0 && $isValidDate && !$hasPendingRequest && $meetsMinimum,
            'available_amount' => $availableAmount,
            'minimum_withdrawal' => $minimumWithdrawal,
            'meets_minimum' => $meetsMinimum,
            'next_withdrawal_date' => $nextWithdrawalDate,
            'is_valid_date' => $isValidDate,
            'has_pending_request' => $hasPendingRequest,
            'reasons' => $this->getWithdrawalRestrictionReasons($availableAmount, $isValidDate, $hasPendingRequest, $meetsMinimum)
        ];
    }

    /**
     * Get reasons why withdrawal might be restricted.
     */
    private function getWithdrawalRestrictionReasons(float $availableAmount, bool $isValidDate, bool $hasPendingRequest, bool $meetsMinimum = true): array
    {
        $reasons = [];
        
        if ($availableAmount <= 0) {
            $reasons[] = 'No available amount for withdrawal';
        } elseif (!$meetsMinimum) {
            $reasons[] = 'Minimum withdrawal amount is $5.00';
        }
        
        if (!$isValidDate) {
            $reasons[] = 'Withdrawals are only allowed on the 15th and 30th of each month';
        }
        
        if ($hasPendingRequest) {
            $reasons[] = 'You already have a pending withdrawal request';
        }
        
        return $reasons;
    }

    /**
     * Create a withdrawal request for a writer.
     */
    public function createWithdrawalRequest($writerId, float $amount, string $paymentMethod, array $paymentDetails): WriterWithdrawalRequest
    {
        $withdrawalCheck = $this->canWriterWithdraw($writerId);
        
        if (!$withdrawalCheck['can_withdraw']) {
            throw new \Exception('Withdrawal not allowed: ' . implode(', ', $withdrawalCheck['reasons']));
        }
        
        if ($amount > $withdrawalCheck['available_amount']) {
            throw new \Exception('Requested amount exceeds available amount');
        }

        return DB::transaction(function () use ($writerId, $amount, $paymentMethod, $paymentDetails) {
            $withdrawalRequest = WriterWithdrawalRequest::create([
                'writer_id' => $writerId,
                'requested_amount' => $amount,
                'currency' => 'USD', // Default currency, could be dynamic
                'withdrawal_date' => now()->toDateString(),
                'status' => WriterWithdrawalRequest::STATUS_REQUESTED,
                'payment_method' => $paymentMethod,
                'payment_details' => $paymentDetails,
            ]);

            // Mark the corresponding orders as processing withdrawal
            $orders = $this->getAvailableOrders($writerId);
            $remainingAmount = $amount;
            
            foreach ($orders as $order) {
                if ($remainingAmount <= 0) break;
                
                $orderAmount = min($remainingAmount, $order->writer_amount);
                $remainingAmount -= $orderAmount;
                
                // For now, we'll handle this in a later phase with partial withdrawals
                // This is a simplified version for Phase 1
            }

            return $withdrawalRequest;
        });
    }

    /**
     * Sync all orders' writer payment status.
     * Useful for initial setup or maintenance.
     */
    public function syncAllOrderPaymentStatuses()
    {
        $orders = Order::whereNotNull('writer_id')->get();
        
        foreach ($orders as $order) {
            $this->updateOrderPaymentStatus($order);
        }
    }
}