<?php

namespace App\Traits;

use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

trait DatabaseNotificationTrait
{
    /**
     * Get common notification data for the database channel.
     *
     * @param string $type Notification type key
     * @param string $title Human-readable notification title
     * @param string $message Main notification message body
     * @param string $icon Icon name (Lucide icon)
     * @param string $color Color class (success, danger, warning, info, default)
     * @param string $priority Priority level (high, medium, low)
     * @param string $category Category (order, payment, account, system, marketing)
     * @param array|null $actions Actions that can be taken from this notification
     * @param \DateTime|null $expiresAt When the notification should expire
     * @param array $data Type-specific data
     * @return array
     */
    protected function getDatabaseNotificationData(
        string $type,
        string $title,
        string $message,
        string $icon = 'bell',
        string $color = 'default',
        string $priority = 'medium',
        string $category = 'system',
        ?array $actions = null,
        ?\DateTime $expiresAt = null,
        array $data = []
    ): array {
        try {
            // Base notification data
            $notificationData = [
                'type' => $type,
                'title' => $title,
                'message' => $message,
                'icon' => $icon,
                'color' => $color,
                'priority' => $priority,
                'category' => $category,
                'expires_at' => $expiresAt ? $expiresAt->format('Y-m-d H:i:s') : null,
            ];

            // Add actions if provided
            if (!empty($actions)) {
                $notificationData['actions'] = $actions;
            }

            // Merge with type-specific data
            if (!empty($data)) {
                $notificationData['data'] = $data;
            }

            return $notificationData;
        } catch (\Exception $e) {
            Log::error('Error creating database notification data', [
                'exception' => $e->getMessage(),
                'type' => $type
            ]);

            // Return minimal data to prevent complete failure
            return [
                'type' => $type,
                'title' => $title,
                'message' => $message,
                'icon' => 'bell',
                'color' => 'default',
                'priority' => 'medium',
                'category' => 'system',
            ];
        }
    }

    /**
     * Create an action button configuration.
     *
     * @param string $text Button text
     * @param string $url URL to navigate to
     * @param string $method HTTP method (GET, POST, DELETE)
     * @param string $style Button style (primary, secondary, danger)
     * @return array
     */
    protected function createAction(
        string $text,
        string $url,
        string $method = 'GET',
        string $style = 'primary'
    ): array {
        return [
            'text' => $text,
            'url' => $url,
            'method' => $method,
            'style' => $style,
        ];
    }
}
