<?php

namespace Database\Factories;

use App\Models\Order;
use App\Models\OrderRevision;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\OrderRevision>
 */
class OrderRevisionFactory extends Factory
{
    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'order_id' => Order::factory(),
            'requested_by' => User::factory(),
            'assigned_to' => null,
            'revision_type' => $this->faker->randomElement([
                OrderRevision::REVISION_TYPE_CONTENT,
                OrderRevision::REVISION_TYPE_FORMATTING,
                OrderRevision::REVISION_TYPE_RESEARCH,
                OrderRevision::REVISION_TYPE_CITATION,
            ]),
            'revision_priority' => $this->faker->randomElement([
                OrderRevision::PRIORITY_LOW,
                OrderRevision::PRIORITY_MEDIUM,
                OrderRevision::PRIORITY_HIGH,
            ]),
            'revision_section' => $this->faker->optional()->sentence(3),
            'revision_reason' => $this->faker->sentence(),
            'specific_instructions' => $this->faker->optional()->paragraph(),
            'status' => 'requested',
            'revision_status' => OrderRevision::STATUS_REQUESTED,
            'revision_deadline' => $this->faker->dateTimeBetween('now', '+1 week'),
            'admin_notes' => $this->faker->optional()->paragraph(),
            'revision_metadata' => [],
            'requested_at' => now(),
            'revision_acknowledged_at' => null,
            'started_at' => null,
            'revision_submitted_at' => null,
            'revision_reviewed_at' => null,
            'revision_number' => $this->faker->numberBetween(1, 10),
        ];
    }

    /**
     * Indicate that the revision is acknowledged.
     */
    public function acknowledged(): static
    {
        return $this->state(fn (array $attributes) => [
            'revision_status' => OrderRevision::STATUS_ACKNOWLEDGED,
            'revision_acknowledged_at' => now(),
        ]);
    }

    /**
     * Indicate that the revision is in progress.
     */
    public function inProgress(): static
    {
        return $this->state(fn (array $attributes) => [
            'revision_status' => OrderRevision::STATUS_IN_PROGRESS,
            'revision_acknowledged_at' => now()->subHours(1),
            'revision_started_at' => now(),
        ]);
    }

    /**
     * Indicate that the revision is submitted.
     */
    public function submitted(): static
    {
        return $this->state(fn (array $attributes) => [
            'revision_status' => OrderRevision::STATUS_SUBMITTED,
            'revision_acknowledged_at' => now()->subHours(2),
            'revision_started_at' => now()->subHours(1),
            'revision_submitted_at' => now(),
        ]);
    }

    /**
     * Indicate that the revision is reviewed.
     */
    public function reviewed(): static
    {
        return $this->state(fn (array $attributes) => [
            'revision_status' => OrderRevision::STATUS_REVIEWED,
            'revision_acknowledged_at' => now()->subHours(3),
            'revision_started_at' => now()->subHours(2),
            'revision_submitted_at' => now()->subHours(1),
            'revision_reviewed_at' => now(),
        ]);
    }

    /**
     * Indicate that the revision is overdue.
     */
    public function overdue(): static
    {
        return $this->state(fn (array $attributes) => [
            'revision_deadline' => now()->subHours(2),
        ]);
    }

    /**
     * Indicate that the revision is high priority.
     */
    public function highPriority(): static
    {
        return $this->state(fn (array $attributes) => [
            'revision_priority' => OrderRevision::PRIORITY_HIGH,
        ]);
    }

    /**
     * Indicate that the revision is assigned to a writer.
     */
    public function assigned(User $writer = null): static
    {
        return $this->state(fn (array $attributes) => [
            'assigned_to' => $writer ? $writer->id : User::factory()->create(['user_type' => 'writer'])->id,
        ]);
    }
}
