<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateWriterInfractionsTable extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('writer_infractions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('writer_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('order_id')->nullable()->constrained()->comment('Related order if applicable');
            $table->foreignId('reported_by')->constrained('users')->comment('Admin who reported the infraction');
            $table->foreignId('reviewed_by')->nullable()->constrained('users')->comment('Admin who reviewed the infraction');
            
            // Infraction details
            $table->enum('infraction_type', [
                'deadline_violation',
                'quality_issues',
                'plagiarism',
                'communication_failure',
                'policy_violation',
                'unprofessional_behavior',
                'client_complaint',
                'other'
            ]);
            
            $table->enum('severity_level', [
                'warning',
                'minor',
                'major',
                'critical'
            ])->default('minor');
            
            $table->text('description');
            $table->text('evidence')->nullable()->comment('Supporting evidence or documentation');
            $table->text('admin_notes')->nullable();
            
            // Status tracking
            $table->enum('status', [
                'reported',
                'under_review',
                'confirmed',
                'disputed',
                'resolved',
                'appealed',
                'appeal_approved',
                'appeal_rejected'
            ])->default('reported');
            
            // Timeline tracking
            $table->timestamp('occurred_at')->nullable()->comment('When the infraction occurred');
            $table->timestamp('reported_at')->useCurrent();
            $table->timestamp('reviewed_at')->nullable();
            $table->timestamp('resolved_at')->nullable();
            $table->timestamp('appeal_deadline')->nullable()->comment('Deadline for writer to appeal');
            
            // Appeal process
            $table->text('writer_response')->nullable()->comment('Writer\'s response to the infraction');
            $table->text('appeal_reason')->nullable()->comment('Reason for appeal if disputed');
            $table->text('appeal_decision_notes')->nullable()->comment('Admin notes on appeal decision');
            
            // Impact assessment
            $table->decimal('estimated_cost_impact', 10, 2)->default(0)->comment('Estimated financial impact');
            $table->integer('client_satisfaction_impact')->nullable()->comment('Impact on client satisfaction 1-5');
            $table->text('impact_assessment')->nullable();
            
            // Progressive discipline
            $table->integer('repeat_count')->default(1)->comment('How many times this type of infraction has occurred');
            $table->boolean('escalated')->default(false)->comment('If this infraction triggered escalation');
            $table->text('escalation_notes')->nullable();
            
            // Related infractions
            $table->foreignId('related_infraction_id')->nullable()->constrained('writer_infractions')->comment('Link to related infraction');
            
            $table->timestamps();
            
            // Indexes for performance
            $table->index(['writer_id', 'status']);
            $table->index(['infraction_type', 'severity_level']);
            $table->index('occurred_at');
            $table->index('reported_at');
            $table->index('severity_level');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('writer_infractions');
    }
} 