<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateWriterFinesTable extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('writer_fines', function (Blueprint $table) {
            $table->id();
            $table->foreignId('writer_id')->constrained('users')->onDelete('cascade');
            $table->foreignId('infraction_id')->nullable()->constrained('writer_infractions')->comment('Related infraction if applicable');
            $table->foreignId('order_id')->nullable()->constrained()->comment('Related order if applicable');
            $table->foreignId('applied_by')->constrained('users')->comment('Admin who applied the fine');
            $table->foreignId('reviewed_by')->nullable()->constrained('users')->comment('Admin who reviewed the fine');
            
            // Fine details
            $table->enum('fine_type', [
                'fixed_amount',
                'percentage_based',
                'progressive',
                'opportunity_cost',
                'quality_penalty',
                'deadline_violation',
                'other'
            ]);
            
            $table->decimal('fine_amount', 10, 2)->comment('Amount of the fine');
            $table->string('currency', 3)->default('USD');
            $table->text('fine_reason');
            $table->text('calculation_details')->nullable()->comment('How the fine amount was calculated');
            
            // Fine application
            $table->enum('application_method', [
                'immediate_deduction',
                'future_earning_deduction',
                'withdrawal_deduction',
                'bonus_cancellation',
                'other'
            ])->default('future_earning_deduction');
            
            $table->enum('status', [
                'pending',
                'applied',
                'disputed',
                'reduced',
                'waived',
                'reversed'
            ])->default('pending');
            
            // Timeline tracking
            $table->timestamp('applied_at')->useCurrent();
            $table->timestamp('effective_from')->nullable()->comment('When the fine becomes effective');
            $table->timestamp('deducted_at')->nullable()->comment('When the fine was actually deducted');
            $table->timestamp('resolved_at')->nullable();
            
            // Payment tracking
            $table->decimal('amount_deducted', 10, 2)->default(0)->comment('Actual amount deducted');
            $table->decimal('remaining_amount', 10, 2)->default(0)->comment('Remaining fine amount to be deducted');
            $table->foreignId('payment_record_id')->nullable()->constrained('writer_payment_records')->comment('Payment record where fine was deducted');
            
            // Appeal and review process
            $table->text('writer_appeal')->nullable()->comment('Writer\'s appeal against the fine');
            $table->text('appeal_decision')->nullable()->comment('Decision on the appeal');
            $table->text('review_notes')->nullable()->comment('Admin notes on fine review');
            
            // Fine configuration
            $table->json('fine_configuration')->nullable()->comment('Configuration used for fine calculation');
            $table->boolean('is_automatic')->default(false)->comment('If fine was applied automatically');
            $table->text('manual_adjustment_notes')->nullable()->comment('Notes if fine was manually adjusted');
            
            // Related fines
            $table->foreignId('related_fine_id')->nullable()->constrained('writer_fines')->comment('Link to related fine');
            $table->integer('installment_number')->nullable()->comment('If fine is paid in installments');
            
            $table->timestamps();
            
            // Indexes for performance
            $table->index(['writer_id', 'status']);
            $table->index(['fine_type', 'status']);
            $table->index('applied_at');
            $table->index('effective_from');
            $table->index('status');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('writer_fines');
    }
} 