<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

return new class extends Migration
{
    public function up(): void
    {
        DB::transaction(function () {
            $rows = DB::table('page_content')
                ->where('block_key', 'like', '%_longform_%')
                ->get();

            foreach ($rows as $row) {
                [$prefix, $suffix] = explode('_longform_', $row->block_key, 2) + [null, null];

                if (!$prefix || !$suffix) {
                    continue;
                }

                $content = json_decode($row->content, true) ?? [];
                $sections = $content['sections'] ?? [];
                $html = $content['description_html'] ?? $content['description'] ?? null;

                if (!$html && !empty($sections)) {
                    $html = $this->combineSectionsHtml($sections);
                }

                $html = $html ?? '';

                $normalized = [
                    'eyebrow' => $content['eyebrow'] ?? 'Deep Dive',
                    'title' => $content['title'] ?? Str::headline(str_replace('_', ' ', $suffix)),
                    'description' => $html,
                    'description_html' => $html,
                    'preferredHeight' => isset($content['preferredHeight'])
                        ? (int) $content['preferredHeight']
                        : 480,
                    'sections' => [],
                ];

                DB::table('page_content')
                    ->where('id', $row->id)
                    ->update([
                        'block_key' => $prefix . '_' . $suffix,
                        'block_type' => 'service_scrollable',
                        'content' => json_encode($normalized),
                    ]);
            }
        });
    }

    public function down(): void
    {
        DB::transaction(function () {
            $rows = DB::table('page_content')
                ->where('block_key', 'not like', '%_longform_%')
                ->where('block_type', 'service_scrollable')
                ->get();

            foreach ($rows as $row) {
                $content = json_decode($row->content, true) ?? [];
                $html = $content['description_html'] ?? $content['description'] ?? '';

                [$prefix, $suffix] = explode('_', $row->block_key, 2) + [null, null];

                if (!$prefix || !$suffix) {
                    continue;
                }

                $legacy = $content;
                $legacy['sections'] = [
                    [
                        'title' => null,
                        'body_html' => $html,
                        'body' => $html,
                        'is_active' => true,
                    ],
                ];

                DB::table('page_content')
                    ->where('id', $row->id)
                    ->update([
                        'block_key' => $prefix . '_longform_' . $suffix,
                        'block_type' => 'service_longform',
                        'content' => json_encode($legacy),
                    ]);
            }
        });
    }

    private function combineSectionsHtml(array $sections): string
    {
        return collect($sections)
            ->filter(fn ($section) => ($section['is_active'] ?? true) !== false)
            ->filter(fn ($section) => !empty($section['body_html']) || !empty($section['body']))
            ->map(function ($section) {
                $title = trim((string) ($section['title'] ?? ''));
                $body = $section['body_html'] ?? $section['body'] ?? '';

                if ($body === '') {
                    return '';
                }

                $heading = $title !== '' ? '<h3>' . e($title) . '</h3>' : '';

                return $heading . $body;
            })
            ->filter()
            ->implode('<br /><br />');
    }
};










