<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up(): void
    {
        $prefixes = ['affordable', 'buy', 'cheap'];

        DB::transaction(function () use ($prefixes) {
            foreach ($prefixes as $prefix) {
                $key = $prefix . '_faqs';

                $row = DB::table('page_content')
                    ->where('block_key', $key)
                    ->first();

                if (!$row) {
                    continue;
                }

                $content = json_decode($row->content, true) ?? [];

                if (!empty($content['faqs'])) {
                    // Already structured
                    continue;
                }

                $html = $content['description_html'] ?? $content['description'] ?? '';

                if (!$html) {
                    continue;
                }

                $faqs = $this->parseFaqsFromHtml($html);

                if (empty($faqs)) {
                    continue;
                }

                $content['faqs'] = $faqs;
                $content['description'] = '<p>Find quick answers to common questions about our service.</p>';
                $content['description_html'] = $content['description'];

                DB::table('page_content')
                    ->where('id', $row->id)
                    ->update([
                        'content' => json_encode($content),
                    ]);
            }
        });
    }

    public function down(): void
    {
        // No-op: we don't want to drop the structured data once created.
    }

    private function parseFaqsFromHtml(string $html): array
    {
        $faqs = [];

        $cursor = 0;
        $length = mb_strlen($html);

        while (true) {
            $qPos = $this->stripos($html, 'Q:', $cursor);

            if ($qPos === false) {
                break;
            }

            $aPos = $this->stripos($html, 'A:', $qPos);

            if ($aPos === false) {
                break;
            }

            $nextQ = $this->stripos($html, 'Q:', $aPos + 2);

            $questionChunk = mb_substr($html, $qPos + 2, $aPos - ($qPos + 2));
            $answerChunk = $nextQ === false
                ? mb_substr($html, $aPos + 2)
                : mb_substr($html, $aPos + 2, $nextQ - ($aPos + 2));

            $question = trim(strip_tags($questionChunk));
            $answerHtml = trim($answerChunk);

            if ($question === '' || $answerHtml === '') {
                $cursor = $aPos + 2;
                continue;
            }

            $faqs[] = [
                'question' => $question,
                'answer' => trim(strip_tags($answerHtml)),
                'answer_html' => $answerHtml,
                'is_active' => true,
            ];

            $cursor = $nextQ === false ? $length : $nextQ;
        }

        return $faqs;
    }

    private function stripos(string $haystack, string $needle, int $offset = 0)
    {
        $pos = mb_stripos($haystack, $needle, $offset, 'UTF-8');

        return $pos === false ? false : $pos;
    }
};
