<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use App\Models\Page;
use App\Models\PageSeo;
use App\Models\PageContent;

class BuyIntentPagesSeeder extends Seeder
{
    public function run(): void
    {
        DB::transaction(function () {
            $pages = [
                [
                    'slug' => 'affordable-essay-writing',
                    'title' => 'Affordable Essay Writing Service',
                    'template' => 'Services/AffordableEssayWriting',
                    'block_prefix' => 'affordable',
                    'meta_title' => 'Affordable Essay Writing Service | High Quality, Fair Prices',
                    'meta_description' => 'Get high‑quality essays at fair prices. Transparent rates, expert writers, originality checks, and on‑time delivery.'
                ],
                [
                    'slug' => 'buy-essays-online',
                    'title' => 'Buy Essay Writing Service',
                    'template' => 'Services/BuyEssayWriting',
                    'block_prefix' => 'buy',
                    'meta_title' => 'Buy Essays Online | Fast, 100% Original',
                    'meta_description' => 'Buy essays online from vetted writers. Original, on‑time, secure checkout. View samples and transparent pricing.'
                ],
                [
                    'slug' => 'cheap-essays-online',
                    'title' => 'Cheap Essay Writing Service',
                    'template' => 'Services/CheapEssayWriting',
                    'block_prefix' => 'cheap',
                    'meta_title' => 'Cheap Essays Online You Can Trust (Honest, Transparent)',
                    'meta_description' => 'Cheap essay options with clear quality safeguards. See how we balance price and standards to protect your grade.'
                ],
            ];

            foreach ($pages as $config) {
                /** @var Page $page */
                $page = Page::firstOrCreate(
                    ['slug' => $config['slug']],
                    [
                        'title' => $config['title'],
                        'page_type' => 'service',
                        'status' => 'published',
                        'template' => $config['template'],
                        'seo_priority' => 80,
                        'published_at' => now(),
                    ]
                );

                // SEO
                $structured = [
                    'service' => [
                        '@context' => 'https://schema.org',
                        '@type' => 'Service',
                        'name' => $config['title'],
                        'areaServed' => 'Global',
                        'provider' => [
                            '@type' => 'Organization',
                            'name' => 'Academic Scribe'
                        ],
                    ],
                    'faq' => [
                        '@context' => 'https://schema.org',
                        '@type' => 'FAQPage',
                        'mainEntity' => [],
                    ],
                    'breadcrumb' => [
                        '@context' => 'https://schema.org',
                        '@type' => 'BreadcrumbList',
                        'itemListElement' => [
                            [
                                '@type' => 'ListItem',
                                'position' => 1,
                                'name' => 'Services',
                                'item' => url('/services'),
                            ],
                            [
                                '@type' => 'ListItem',
                                'position' => 2,
                                'name' => $config['title'],
                                'item' => url('/services/' . $config['slug']),
                            ],
                        ],
                    ],
                    'organization' => [
                        '@context' => 'https://schema.org',
                        '@type' => 'Organization',
                        'name' => 'Academic Scribe',
                        'url' => url('/'),
                    ],
                ];

                PageSeo::updateOrCreate(
                    ['page_id' => $page->id],
                    [
                        'meta_title' => $config['meta_title'],
                        'meta_description' => $config['meta_description'],
                        'canonical_url' => url('/services/' . $config['slug']),
                        'og_url' => url('/services/' . $config['slug']),
                        'robots' => 'index, follow',
                        'language' => 'en-US',
                        'structured_data' => $structured,
                    ]
                );

                // Content blocks (canonical schema)
                $blockDefinitions = [
                    ['suffix' => 'hero', 'file_suffix' => 'hero', 'block_type' => 'service_hero'],
                    ['suffix' => 'overview', 'file_suffix' => 'overview', 'block_type' => 'service_overview'],
                    ['suffix' => 'process', 'file_suffix' => 'process', 'block_type' => 'service_process'],
                    ['suffix' => 'benefits', 'file_suffix' => 'benefits', 'block_type' => 'service_benefits'],
                    ['suffix' => 'testimonials', 'file_suffix' => 'testimonials', 'block_type' => 'service_testimonials'],
                    ['suffix' => 'faqs', 'file_suffix' => 'faqs', 'block_type' => 'service_faqs'],
                    ['suffix' => 'cta', 'file_suffix' => 'cta', 'block_type' => 'service_cta'],
                    ['suffix' => 'related_services', 'file_suffix' => 'related_services', 'block_type' => 'service_links', 'link_category' => 'related'],
                    ['suffix' => 'support_pages', 'file_suffix' => 'support_pages', 'block_type' => 'service_links', 'link_category' => 'support'],
                    ['suffix' => 'resource_links', 'file_suffix' => 'resource_links', 'block_type' => 'service_links', 'link_category' => 'resources'],
                    // Scrollable deep-dive sections
                    ['suffix' => 'trust', 'file_suffix' => 'trust', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'writer_selection', 'file_suffix' => 'writer_selection', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'reasons', 'file_suffix' => 'reasons', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'originality', 'file_suffix' => 'originality', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'join_experts', 'file_suffix' => 'join_experts', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'types', 'file_suffix' => 'types', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'hiring_steps', 'file_suffix' => 'hiring_steps', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'writer_standout', 'file_suffix' => 'writer_standout', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'service_benefits', 'file_suffix' => 'service_benefits', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'peace_of_mind', 'file_suffix' => 'peace_of_mind', 'block_type' => 'service_scrollable'],
                    ['suffix' => 'quality_score', 'file_suffix' => 'quality_score', 'block_type' => 'service_scrollable'],
                ];

                foreach ($blockDefinitions as $index => $definition) {
                    $blockKeySuffix = $definition['suffix'];

                    $blockKey = $config['block_prefix'] . '_' . $blockKeySuffix;
                    $content = $this->loadMarkdownBlock($config['slug'], $config['block_prefix'], $definition, $blockKey, $config['title']);

                    PageContent::updateOrCreate(
                        [
                            'page_id' => $page->id,
                            'block_key' => $blockKey,
                        ],
                        [
                            'block_type' => $definition['block_type'],
                            'content' => $content,
                            'order' => $definition['order'] ?? $index + 1,
                            'is_active' => true,
                        ]
                    );
                }
            }
        });
    }

    private function loadMarkdownBlock(string $slug, string $prefix, array $definition, string $blockKey, string $pageTitle): array
    {
        $base = resource_path('content/services/' . $slug);
        $fileSuffix = $definition['file_suffix'] ?? $definition['suffix'];
        $path = $base . '/' . $prefix . '_' . $fileSuffix . '.md';

        if (File::exists($path)) {
            $markdown = File::get($path);
        } else {
            // Minimal placeholder content
            $title = Str::headline(str_replace('_', ' ', $blockKey));
            $markdown = "# {$title}\n\nContent placeholder for {$blockKey}. Replace with final copy.";
        }

        // Map to the structure expected by frontend components
        return $this->mapMarkdownToBlock($definition, $blockKey, $pageTitle, $markdown);
    }

    private function mapMarkdownToBlock(array $definition, string $blockKey, string $pageTitle, string $markdown): array
    {
        $html = $this->toHtml($markdown);

        switch ($definition['block_type']) {
            case 'service_hero':
                return [
                    'title' => $pageTitle,
                    'description' => $html,
                    'description_html' => $html,
                    'keyPoints' => [],
                    'ctaText' => 'Get a price in 30 seconds',
                    'ctaLink' => url('/place-order'),
                    'trustSignals' => [],
                ];
            case 'service_overview':
                return [
                    'title' => 'Overview',
                    'description' => $html,
                    'description_html' => $html,
                    'features' => [],
                ];
            case 'service_process':
                return [
                    'title' => 'How it works',
                    'description' => 'Our ordering process in four steps.',
                    'description_html' => '<p>Our ordering process in four steps.</p>',
                    'steps' => [
                        [
                            'number' => 1,
                            'icon' => 'ClipboardList',
                            'title' => 'Tell us about your essay',
                            'description' => 'Share details, rubric, deadline',
                            'description_html' => '<p>Share details, rubric, deadline.</p>',
                            'is_active' => true,
                        ],
                        [
                            'number' => 2,
                            'icon' => 'UserCheck',
                            'title' => 'Get matched to a writer',
                            'description' => 'We assign a vetted expert',
                            'description_html' => '<p>We assign a vetted expert.</p>',
                            'is_active' => true,
                        ],
                        [
                            'number' => 3,
                            'icon' => 'Edit3',
                            'title' => 'Review drafts and request edits',
                            'description' => 'We iterate until satisfied',
                            'description_html' => '<p>We iterate until satisfied.</p>',
                            'is_active' => true,
                        ],
                        [
                            'number' => 4,
                            'icon' => 'Send',
                            'title' => 'Receive final paper',
                            'description' => 'On time, plagiarism-checked',
                            'description_html' => '<p>On time, plagiarism-checked.</p>',
                            'is_active' => true,
                        ],
                    ],
                ];
            case 'service_benefits':
                return [
                    'title' => 'Why choose us',
                    'description' => $html,
                    'description_html' => $html,
                    'benefits' => [],
                    'ctaText' => 'See pricing',
                    'ctaLink' => url('/pricing-page'),
                    'guarantees' => [
                        ['text' => 'Originality', 'is_active' => true],
                        ['text' => 'On-time delivery', 'is_active' => true],
                        ['text' => 'Free revisions', 'is_active' => true],
                    ],
                ];
            case 'service_testimonials':
                return [
                    'title' => 'What students say',
                    'description' => $html,
                    'description_html' => $html,
                    'testimonials' => [],
                    'trustMetrics' => [],
                ];
            case 'service_faqs':
                return [
                    'title' => 'FAQs',
                    'description' => $html,
                    'description_html' => $html,
                    'faqs' => [],
                ];
            case 'service_cta':
                return [
                    'title' => 'Ready to get started?',
                    'description' => $html,
                    'description_html' => $html,
                    'benefits' => [
                        ['text' => 'Fast quotes', 'is_active' => true],
                        ['text' => 'Secure checkout', 'is_active' => true],
                        ['text' => 'Expert writers', 'is_active' => true],
                    ],
                    'primaryButtonText' => 'Get a price',
                    'primaryButtonLink' => url('/place-order'),
                    'secondaryButtonText' => 'Talk to an expert',
                    'secondaryButtonLink' => url('/contact'),
                    'bgColor' => 'blue',
                    'urgencyText' => 'Limited availability this week',
                    'guarantees' => [
                        ['text' => 'Plagiarism-free', 'is_active' => true],
                        ['text' => 'On-time delivery', 'is_active' => true],
                    ],
                    'trustBadges' => [],
                ];
            case 'service_links':
                $category = $definition['link_category'] ?? 'related';

                return [
                    'title' => Str::headline(str_replace('_', ' ', Str::after($blockKey, '_'))),
                    'description' => $html,
                    'description_html' => $html,
                    'linkCategory' => $category,
                    'links' => [],
                ];
            case 'service_scrollable':
                return [
                    'eyebrow' => 'Deep Dive',
                    'title' => Str::headline(str_replace('_', ' ', Str::after($blockKey, '_'))),
                    'description' => $html,
                    'description_html' => $html,
                    'preferredHeight' => 480,
                    'sections' => [],
                ];
        }

        return [
            'title' => Str::headline($blockKey),
            'description' => $html,
            'description_html' => $html,
        ];
    }

    private function toHtml(string $markdown): string
    {
        return trim(Str::markdown($markdown));
    }
}


