<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Page;
use App\Models\PageSeo;
use App\Models\PageContent;

class SeedServiceLiteratureReviews extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create/update the page for Literature Reviews service
        $page = Page::updateOrCreate(
            ['slug' => 'literature-reviews'],
            [
                'title' => 'Literature Review Writing Service',
                'page_type' => 'service',
                'status' => 'published',
                'template' => 'Services/LiteratureReviews',
                'seo_priority' => 85,
                'published_at' => now(),
            ]
        );

        // Structured data with all schemas
        $structuredData = [
            'service' => [
                '@context' => 'https://schema.org',
                '@type' => 'Service',
                'name' => 'Professional Literature Review Writing Service',
                'description' => 'Expert literature review writing service providing systematic literature review help with professional writers, comprehensive analysis, meta-analysis support, and perfect academic formatting for research projects and dissertations.',
                'provider' => [
                    '@type' => 'Organization',
                    'name' => 'Academic Scribe',
                    'url' => 'https://academicscribe.com',
                    'logo' => 'https://academicscribe.com/images/logo.png',
                    'contactPoint' => [
                        '@type' => 'ContactPoint',
                        'telephone' => '+1-555-123-4567',
                        'contactType' => 'customer service',
                        'availableLanguage' => ['English'],
                        'areaServed' => 'Worldwide',
                    ],
                    'address' => [
                        '@type' => 'PostalAddress',
                        'addressCountry' => 'US',
                        'addressRegion' => 'Global',
                    ],
                    'sameAs' => [
                        'https://facebook.com/academicscribe',
                        'https://twitter.com/academicscribe',
                        'https://linkedin.com/company/academicscribe',
                    ],
                ],
                'serviceType' => 'Academic Writing Service',
                'areaServed' => 'Worldwide',
                'availableChannel' => [
                    '@type' => 'ServiceChannel',
                    'serviceUrl' => 'https://academicscribe.com/services/literature-reviews',
                    'servicePhone' => '+1-555-123-4567',
                    'availableLanguage' => 'English',
                ],
                'hasOfferCatalog' => [
                    '@type' => 'OfferCatalog',
                    'name' => 'Literature Review Writing Services',
                    'itemListElement' => [
                        [
                            '@type' => 'Offer',
                            'itemOffered' => [
                                '@type' => 'Service',
                                'name' => 'Systematic Literature Review Help',
                                'description' => 'Rigorous systematic literature reviews with comprehensive methodology and PRISMA guidelines',
                            ],
                            'price' => 'Starting at $18.99',
                            'priceCurrency' => 'USD',
                            'availability' => 'https://schema.org/InStock',
                        ],
                        [
                            '@type' => 'Offer',
                            'itemOffered' => [
                                '@type' => 'Service',
                                'name' => 'Narrative Literature Review Writing',
                                'description' => 'Comprehensive narrative literature reviews with thematic organization and critical analysis',
                            ],
                            'price' => 'Starting at $16.99',
                            'priceCurrency' => 'USD',
                            'availability' => 'https://schema.org/InStock',
                        ],
                        [
                            '@type' => 'Offer',
                            'itemOffered' => [
                                '@type' => 'Service',
                                'name' => 'Meta-Analysis Literature Review',
                                'description' => 'Statistical synthesis and meta-analysis of quantitative research studies with forest plots',
                            ],
                            'price' => 'Starting at $22.99',
                            'priceCurrency' => 'USD',
                            'availability' => 'https://schema.org/InStock',
                        ],
                        [
                            '@type' => 'Offer',
                            'itemOffered' => [
                                '@type' => 'Service',
                                'name' => 'Scoping Review Writing',
                                'description' => 'Comprehensive scoping reviews to map research landscapes and identify knowledge gaps',
                            ],
                            'price' => 'Starting at $19.99',
                            'priceCurrency' => 'USD',
                            'availability' => 'https://schema.org/InStock',
                        ],
                    ],
                ],
            ],
            'organization' => [
                '@context' => 'https://schema.org',
                '@type' => 'Organization',
                'name' => 'Academic Scribe',
                'description' => 'Professional academic writing services with expert literature review writers providing systematic literature review help and comprehensive academic literature synthesis for researchers worldwide.',
                'url' => 'https://academicscribe.com',
                'logo' => 'https://academicscribe.com/images/logo.png',
                'image' => 'https://academicscribe.com/images/literature-review-writing-service.jpg',
                'contactPoint' => [
                    '@type' => 'ContactPoint',
                    'telephone' => '+1-555-123-4567',
                    'contactType' => 'customer service',
                    'availableLanguage' => ['English'],
                    'areaServed' => 'Worldwide',
                ],
                'address' => [
                    '@type' => 'PostalAddress',
                    'addressCountry' => 'US',
                    'addressRegion' => 'Global',
                ],
                'sameAs' => [
                    'https://facebook.com/academicscribe',
                    'https://twitter.com/academicscribe',
                    'https://linkedin.com/company/academicscribe',
                ],
            ],
            'faq' => [
                '@context' => 'https://schema.org',
                '@type' => 'FAQPage',
                'mainEntity' => [
                    [
                        '@type' => 'Question',
                        'name' => 'What types of literature reviews does your literature review writing service handle?',
                        'acceptedAnswer' => [
                            '@type' => 'Answer',
                            'text' => 'Our professional literature review writing service handles all types of literature reviews including systematic literature reviews, narrative literature reviews, meta-analysis reviews, scoping reviews, integrative literature reviews, critical literature reviews, and rapid reviews. Our systematic literature review help covers all academic disciplines including healthcare, education, psychology, business, social sciences, and specialized research fields across all universities and institutions worldwide.',
                        ],
                    ],
                    [
                        '@type' => 'Question',
                        'name' => 'What qualifications do your literature review writers have?',
                        'acceptedAnswer' => [
                            '@type' => 'Answer',
                            'text' => 'Our literature review writers hold advanced degrees (Master\'s and PhD) in their respective fields with extensive literature review writing and systematic review experience. Each professional literature review writer undergoes rigorous screening including academic credential verification, literature review portfolio assessment, systematic review methodology expertise testing, and critical analysis skills evaluation. Our literature review assistance team includes published researchers, systematic review specialists, and academic experts with proven track records in comprehensive literature synthesis.',
                        ],
                    ],
                ],
            ],
            'breadcrumb' => [
                '@context' => 'https://schema.org',
                '@type' => 'BreadcrumbList',
                'itemListElement' => [
                    [
                        '@type' => 'ListItem',
                        'position' => 1,
                        'name' => 'Home',
                        'item' => 'https://academicscribe.com',
                    ],
                    [
                        '@type' => 'ListItem',
                        'position' => 2,
                        'name' => 'Services',
                        'item' => 'https://academicscribe.com/services',
                    ],
                    [
                        '@type' => 'ListItem',
                        'position' => 3,
                        'name' => 'Literature Review Writing Service',
                        'item' => 'https://academicscribe.com/services/literature-reviews',
                    ],
                ],
            ],
            'website' => [
                '@context' => 'https://schema.org',
                '@type' => 'WebSite',
                'name' => 'Academic Scribe - Literature Review Writing Service',
                'description' => 'Professional literature review writers providing systematic literature review help and comprehensive academic literature synthesis for research projects and dissertations.',
                'url' => 'https://academicscribe.com/services/literature-reviews',
                'potentialAction' => [
                    '@type' => 'SearchAction',
                    'target' => 'https://academicscribe.com/search?q={search_term_string}',
                    'query-input' => 'required name=search_term_string',
                ],
            ],
            'webpage' => [
                '@context' => 'https://schema.org',
                '@type' => 'WebPage',
                'name' => 'Professional Literature Review Writing Service | Systematic Literature Review Help',
                'description' => 'Expert literature review writing service with professional writers. Get comprehensive systematic literature review help with meta-analysis, thematic synthesis, and guaranteed results.',
                'url' => 'https://academicscribe.com/services/literature-reviews',
                'isPartOf' => [
                    '@type' => 'WebSite',
                    'name' => 'Academic Scribe',
                    'url' => 'https://academicscribe.com',
                ],
                'about' => [
                    '@type' => 'Service',
                    'name' => 'Literature Review Writing Service',
                ],
                'mainEntity' => [
                    '@type' => 'Service',
                    'name' => 'Professional Literature Review Writing Service',
                ],
            ],
        ];

        // SEO data
        PageSeo::updateOrCreate(
            ['page_id' => $page->id],
            [
                'meta_title' => 'Professional Literature Review Writing Service | Systematic Literature Review Help | Expert Academic Writers | Academic Scribe',
                'meta_description' => 'Get expert literature review writing service with professional academic writers. Comprehensive systematic literature review help with meta-analysis, thematic synthesis, and 24/7 support. Order now for guaranteed results!',
                'meta_keywords' => 'literature review writing service, professional literature review writing, systematic literature review help, literature review assistance, literature review writers, academic literature review service, literature review writing help, systematic review writing, meta-analysis literature review, narrative literature review writing, literature synthesis service, academic literature analysis, literature review writing company, literature review writing experts',
                'canonical_url' => 'https://academicscribe.com/services/literature-reviews',
                'og_url' => 'https://academicscribe.com/services/literature-reviews',
                'robots' => 'index, follow, max-snippet:-1, max-image-preview:large, max-video-preview:-1',
                'language' => 'en-US',
                'structured_data' => $structuredData,
                'twitter_card' => 'summary_large_image',
                'twitter_title' => 'Professional Literature Review Writing Service | Systematic Literature Review Help',
                'twitter_description' => 'Expert literature review writing service with professional writers. Get comprehensive systematic literature review help with meta-analysis and guaranteed results.',
                'twitter_image' => 'https://academicscribe.com/images/literature-review-writing-service-twitter.jpg',
                'og_title' => 'Professional Literature Review Writing Service | Systematic Literature Review Help',
                'og_description' => 'Expert literature review writing service with professional academic writers. Get comprehensive systematic literature review help with meta-analysis and thematic synthesis. Trusted by 5,000+ researchers worldwide.',
                'og_image' => 'https://academicscribe.com/images/literature-review-writing-service-og.jpg',
                'og_type' => 'website',
            ]
        );

        // Content blocks
        $blocks = [
            [
                'block_type' => 'service_hero',
                'block_key' => 'literature_review_hero',
                'order' => 1,
                'content' => [
                    'title' => 'Professional Literature Review Writing Service | Expert Systematic Literature Review Help',
                    'description' => 'Get expert literature review writing service from professional writers with advanced degrees. Our comprehensive systematic literature review help covers all research methodologies with meta-analysis, thematic synthesis, and critical evaluation. Trusted by 5,000+ researchers worldwide with 24/7 support and guaranteed results.',
                    'keyPoints' => [
                        'Professional literature review writers with expertise in systematic review methodology',
                        'Systematic literature review help with 100% original analysis and plagiarism-free content',
                        'Literature review assistance covering all disciplines with PRISMA guidelines and meta-analysis',
                        'Literature review writing service with perfect academic formatting and citation standards',
                    ],
                    'ctaText' => 'Get Professional Literature Review Help Now',
                    'ctaLink' => '/contact',
                    'trustSignals' => [
                        '5,000+ successful literature reviews',
                        '4.8/5 average rating',
                        '24/7 expert support',
                        'Money-back guarantee',
                    ],
                ],
            ],
            [
                'block_type' => 'service_overview',
                'block_key' => 'literature_review_overview',
                'order' => 2,
                'content' => [
                    'title' => 'Our Comprehensive Literature Review Writing Service',
                    'description' => 'We provide professional literature review writing across all research methodologies with expert systematic review techniques and comprehensive analysis. Our literature review assistance includes original synthesis, critical evaluation, and advanced methodology support for researchers at all academic levels.',
                    'features' => [
                        [
                            'title' => 'Expert Literature Review Writers',
                            'description' => 'Professional literature review writers with advanced degrees (Master\'s & PhD) who understand systematic review methodology, research synthesis techniques, and discipline-specific requirements for quality literature review writing service',
                            'icon' => 'Award',
                        ],
                        [
                            'title' => 'Systematic Review Methodology',
                            'description' => 'Systematic literature review help with rigorous methodology, comprehensive search strategies, PRISMA guidelines, and statistical analysis tailored to your specific research objectives and academic requirements',
                            'icon' => 'Search',
                        ],
                        [
                            'title' => 'All Review Types Covered',
                            'description' => 'Literature review assistance covering systematic reviews, narrative reviews, meta-analysis, scoping reviews, integrative reviews, and critical reviews with subject-matter experts and research specialists',
                            'icon' => 'BookOpen',
                        ],
                        [
                            'title' => 'Perfect Academic Standards',
                            'description' => 'Professional literature review writing with proper citations, references, and formatting in APA, MLA, Chicago, Harvard, and other academic styles with attention to journal-specific guidelines',
                            'icon' => 'FileText',
                        ],
                    ],
                ],
            ],
            [
                'block_type' => 'service_process',
                'block_key' => 'literature_review_process',
                'order' => 3,
                'content' => [
                    'title' => 'Our Professional Literature Review Writing Process',
                    'description' => 'We follow a systematic, evidence-based approach to deliver exceptional literature review writing service with rigorous methodology, comprehensive analysis, and academic excellence that meets your research objectives.',
                    'steps' => [
                        ['number' => 1, 'title' => 'Research Question & Scope Definition', 'description' => 'Comprehensive analysis of your literature review requirements, research questions, academic objectives, and review scope to establish clear direction for your systematic literature review help project with methodology planning.', 'icon' => 'Target'],
                        ['number' => 2, 'title' => 'Search Strategy Development', 'description' => 'Develop comprehensive search strategy with relevant databases, keywords, inclusion/exclusion criteria, and search protocols for professional literature review writing service excellence.', 'icon' => 'Search'],
                        ['number' => 3, 'title' => 'Literature Search & Screening', 'description' => 'Conduct systematic literature search using academic databases, peer-reviewed journals, and credible sources with rigorous screening process for your literature review assistance with quality assessment.', 'icon' => 'Database'],
                        ['number' => 4, 'title' => 'Data Extraction & Quality Assessment', 'description' => 'Extract relevant data from selected studies and perform quality assessment using standardized tools for your literature review writing with methodological rigor and evidence evaluation.', 'icon' => 'BarChart'],
                        ['number' => 5, 'title' => 'Synthesis & Analysis', 'description' => 'Professional literature review writers conduct thematic synthesis, meta-analysis, or narrative analysis with clear organization, critical evaluation, and evidence-based conclusions for systematic literature review help excellence.', 'icon' => 'Edit'],
                        ['number' => 6, 'title' => 'Citation & Academic Formatting', 'description' => 'Apply proper academic citations, reference lists, and formatting according to your required style guide with attention to journal guidelines for literature review writing service compliance.', 'icon' => 'FileText'],
                        ['number' => 7, 'title' => 'Quality Review & Expert Editing', 'description' => 'Comprehensive review, editing, and proofreading to ensure your literature review meets academic standards, research requirements, and excellence expectations for professional literature review writing service delivery.', 'icon' => 'CheckCircle'],
                    ],
                ],
            ],
            [
                'block_type' => 'service_benefits',
                'block_key' => 'literature_review_benefits',
                'order' => 4,
                'content' => [
                    'title' => 'Why Choose Our Professional Literature Review Writing Service',
                    'description' => 'Discover why over 5,000 researchers and students choose our expert literature review writing assistance for their academic success and research excellence',
                    'benefits' => [
                        ['title' => 'Systematic Review Methodology Experts', 'description' => 'Professional literature review writers with advanced degrees (Master\'s & PhD) in your specific field who understand systematic review methodology, PRISMA guidelines, and research synthesis techniques for exceptional results', 'icon' => 'Award'],
                        ['title' => '100% Original Analysis & Custom Reviews', 'description' => 'Literature review assistance with completely original analysis, unique synthesis, plagiarism-free content, and comprehensive originality reports tailored to your specific research requirements and academic objectives', 'icon' => 'Shield'],
                        ['title' => 'Comprehensive Research Methodology', 'description' => 'Systematic literature review help with rigorous search strategies, advanced screening techniques, quality assessment methods, and evidence-based synthesis that meets highest academic standards', 'icon' => 'Search'],
                        ['title' => 'Perfect Academic Standards Compliance', 'description' => 'Literature review writing service that meets all academic standards with proper structure, perfect citations, journal formatting requirements, and adherence to research integrity guidelines', 'icon' => 'CheckCircle'],
                        ['title' => 'Guaranteed Delivery & Expert Support', 'description' => 'Professional literature review writing service with reliable delivery schedules, 24/7 expert support, unlimited revisions, and money-back guarantee for your complete satisfaction and research success', 'icon' => 'Clock'],
                    ],
                    'imageAlt' => 'Professional Literature Review Writing Service Benefits - Systematic Literature Review Help with Expert Writers and Original Analysis',
                    'ctaText' => 'Order Professional Literature Review Now',
                    'ctaLink' => '/place-order',
                    'guarantees' => [
                        '100% Original Analysis',
                        'On-Time Delivery',
                        '24/7 Expert Support',
                        'Money-Back Guarantee',
                        'Unlimited Revisions',
                    ],
                ],
            ],
            [
                'block_type' => 'service_testimonials',
                'block_key' => 'literature_review_testimonials',
                'order' => 5,
                'content' => [
                    'title' => 'What Our Literature Review Writing Service Clients Say',
                    'description' => 'Read authentic testimonials from researchers and students who\'ve achieved academic success with our professional literature review writing assistance',
                    'testimonials' => [
                        ['text' => 'The literature review writing service I received was outstanding. The writer conducted systematic methodology and provided comprehensive synthesis that impressed my supervisor. The systematic literature review help included perfect citations and met all my university\'s research requirements. I received excellent feedback on my dissertation!', 'author' => 'Dr. Sarah Johnson', 'role' => 'PhD Student, Public Health', 'rating' => 5, 'verified' => true, 'university' => 'Johns Hopkins University'],
                        ['text' => 'I needed professional literature review writing for my master\'s thesis, and the quality exceeded my expectations. The literature review assistance provided rigorous methodology with comprehensive analysis and clear presentation of findings. The writer was knowledgeable about systematic review techniques and very responsive.', 'author' => 'Michael Chen', 'role' => 'Master\'s Student, Psychology', 'rating' => 5, 'verified' => true, 'university' => 'Stanford University'],
                        ['text' => 'As an international researcher, I struggled with systematic review methodology. The literature review writing service helped me understand proper search strategies and synthesis techniques while delivering a high-quality review that met journal standards. The expert support was available 24/7 and incredibly helpful.', 'author' => 'Dr. Priya Patel', 'role' => 'Postdoctoral Researcher, Medicine', 'rating' => 5, 'verified' => true, 'university' => 'Harvard Medical School'],
                    ],
                    'trustMetrics' => [
                        'totalReviews' => '1,456',
                        'averageRating' => '4.8',
                        'satisfactionRate' => '97%',
                        'repeatCustomers' => '88%',
                    ],
                ],
            ],
            [
                'block_type' => 'service_faqs',
                'block_key' => 'literature_review_faqs',
                'order' => 6,
                'content' => [
                    'title' => 'Literature Review Writing Service FAQs',
                    'description' => 'Get answers to common questions about our professional literature review writing assistance and systematic literature review help services',
                    'faqs' => [
                        ['question' => 'What types of literature reviews does your literature review writing service handle?', 'answer' => 'Our professional literature review writing service handles all types of literature reviews including systematic literature reviews, narrative literature reviews, meta-analysis reviews, scoping reviews, integrative literature reviews, critical literature reviews, rapid reviews, and umbrella reviews. Our systematic literature review help covers all academic disciplines including healthcare, education, psychology, business, social sciences, engineering, and specialized research fields. We also handle specific methodologies like PRISMA systematic reviews, Cochrane reviews, and journal-specific review formats.'],
                        ['question' => 'What qualifications do your literature review writers have?', 'answer' => 'Our literature review writers hold advanced degrees (Master\'s and PhD) in their respective fields with extensive literature review writing and systematic review experience. Each professional literature review writer undergoes rigorous screening including academic credential verification, literature review portfolio assessment, systematic review methodology expertise testing, and critical analysis skills evaluation. Our literature review assistance team includes published researchers, systematic review specialists, meta-analysis experts, and academic professionals with proven track records in comprehensive literature synthesis and research methodology.'],
                        ['question' => 'How do you ensure quality in your systematic literature review help?', 'answer' => 'Our systematic literature review help ensures quality through rigorous methodology including comprehensive search strategies, systematic screening processes, quality assessment using standardized tools (PRISMA, AMSTAR, JBI), and evidence-based synthesis techniques. Our professional literature review writers follow established guidelines, use multiple databases, conduct duplicate screening, and perform quality checks at every stage. We provide detailed methodology documentation, PRISMA flow diagrams, and comprehensive quality assessment reports with every literature review writing service order.'],
                        ['question' => 'Can you help with meta-analysis and statistical synthesis?', 'answer' => 'Yes, our literature review assistance includes comprehensive meta-analysis and statistical synthesis support. Our professional literature review writers are experienced in quantitative synthesis techniques, forest plot creation, heterogeneity assessment, subgroup analysis, and sensitivity analysis. We use advanced statistical software (RevMan, R, STATA) for meta-analysis and provide detailed statistical reports. Our systematic literature review help covers both fixed-effect and random-effects models, publication bias assessment, and GRADE evidence evaluation for comprehensive meta-analysis literature reviews.'],
                        ['question' => 'Do you follow PRISMA guidelines for systematic reviews?', 'answer' => 'Yes, our literature review writing service strictly follows PRISMA (Preferred Reporting Items for Systematic Reviews and Meta-Analyses) guidelines for all systematic literature reviews. Our systematic literature review help includes PRISMA flow diagrams, comprehensive search strategies, systematic screening processes, quality assessment, and structured reporting. We also follow other relevant guidelines including AMSTAR for systematic review quality, JBI methodology for different review types, and journal-specific requirements. Our professional literature review writers are trained in evidence-based methodology and systematic review best practices.'],
                        ['question' => 'What is your revision policy for literature review writing service?', 'answer' => 'Our literature review writing service includes unlimited free revisions within 30 days of delivery to ensure complete satisfaction. If your literature review assistance doesn\'t meet your requirements, needs adjustments, or requires modifications based on supervisor feedback, our professional literature review writers will revise it at no additional cost. We\'re committed to your research success and will work with you until your literature review meets all your academic requirements, research standards, and personal expectations. Major scope changes may require additional discussion, but standard revisions are always free.'],
                        ['question' => 'How do you handle different citation styles in literature reviews?', 'answer' => 'Our professional literature review writing service includes expert formatting in all major academic citation styles including APA 7th edition, MLA 9th edition, Chicago/Turabian, Harvard, Vancouver, IEEE, and journal-specific formats. Our literature review writers are trained in academic citation standards and use professional reference management tools (Zotero, Mendeley, EndNote) to ensure accuracy. Every systematic literature review help order includes properly formatted in-text citations, reference lists, and adherence to your institution\'s or journal\'s specific formatting requirements including margins, spacing, and style guidelines.'],
                        ['question' => 'Can you help with literature reviews for different academic levels?', 'answer' => 'Yes, our literature review assistance covers all academic levels including undergraduate literature reviews, master\'s thesis literature reviews, PhD dissertation literature reviews, postdoctoral research reviews, and professional journal submissions. Our professional literature review writers adapt methodology and complexity to match your academic level and requirements. Whether you need a basic narrative review for coursework or a comprehensive systematic review for publication, our literature review writing service provides appropriate depth, rigor, and academic standards for your specific educational level and research objectives.'],
                    ],
                ],
            ],
            [
                'block_type' => 'service_cta',
                'block_key' => 'literature_review_cta',
                'order' => 7,
                'content' => [
                    'title' => 'Ready to Get Professional Literature Review Help That Guarantees Research Success?',
                    'description' => 'Join over 5,000 successful researchers who\'ve achieved academic excellence with our expert literature review writing service. Get systematic literature review help from professional writers with advanced degrees today.',
                    'benefits' => [
                        ['text' => 'Expert literature review writers with systematic review expertise', 'icon' => '🏆'],
                        ['text' => '100% original analysis with rigorous methodology', 'icon' => '🔍'],
                        ['text' => 'Perfect citations and academic formatting guaranteed', 'icon' => '📄'],
                        ['text' => '24/7 expert support with money-back guarantee', 'icon' => '🛡️'],
                    ],
                    'primaryButtonText' => 'Get Professional Literature Review Help Now',
                    'primaryButtonLink' => '/place-order',
                    'secondaryButtonText' => 'Free Literature Review Consultation',
                    'secondaryButtonLink' => '/contact',
                    'bgColor' => 'blue',
                    'urgencyText' => 'Limited Time: 15% OFF for new research projects',
                    'guarantees' => [
                        '100% Original Analysis',
                        'On-Time Delivery',
                        'Money-Back Guarantee',
                        'Unlimited Revisions',
                        '24/7 Expert Support',
                    ],
                    'trustBadges' => [
                        'SSL Secured',
                        'Privacy Protected',
                        'Expert Writers',
                        'Research Excellence',
                    ],
                ],
            ],
            [
                'block_type' => 'internal_links',
                'block_key' => 'literature_review_related_services',
                'order' => 8,
                'content' => [
                    'title' => 'Related Academic Writing Services',
                    'description' => 'Explore our comprehensive range of academic writing services',
                    'links' => [
                        ['title' => 'Dissertation Writing Service', 'url' => '/services/dissertation-writing', 'description' => 'PhD dissertation help and doctoral thesis writing'],
                        ['title' => 'Research Paper Writing Service', 'url' => '/services/research-papers', 'description' => 'Professional research paper help and academic research writing'],
                        ['title' => 'Essay Writing Service', 'url' => '/services/essay-writing', 'description' => 'Professional essay writing for all academic levels'],
                        ['title' => 'Thesis Writing Help', 'url' => '/services/thesis-proposals', 'description' => 'Master\'s thesis writing assistance and support'],
                    ],
                ],
            ],
            [
                'block_type' => 'internal_links',
                'block_key' => 'literature_review_support_pages',
                'order' => 9,
                'content' => [
                    'title' => 'Learn More About Our Literature Review Writing Service',
                    'description' => 'Get detailed information about our literature review specialists and pricing',
                    'links' => [
                        ['title' => 'Our Writers', 'url' => '/writers', 'description' => 'Meet our literature review writing specialists'],
                        ['title' => 'Pricing', 'url' => '/pricing-page', 'description' => 'Transparent literature review writing pricing'],
                        ['title' => 'Samples', 'url' => '/samples', 'description' => 'View literature review writing samples'],
                        ['title' => 'Guarantees', 'url' => '/guarantees', 'description' => 'Our quality and satisfaction guarantees'],
                        ['title' => 'Reviews', 'url' => '/reviews', 'description' => 'Read customer reviews and testimonials'],
                    ],
                ],
            ],
            [
                'block_type' => 'internal_links',
                'block_key' => 'literature_review_resource_links',
                'order' => 10,
                'content' => [
                    'title' => 'Professional Literature Review Writing Service Resources',
                    'links' => [
                        ['title' => 'Literature Review Writing Guide', 'url' => '/blog/literature-review-writing-guide', 'description' => 'Complete guide to writing literature reviews'],
                        ['title' => 'Systematic Review Methodology', 'url' => '/blog/systematic-review-methodology', 'description' => 'Master systematic review techniques and PRISMA guidelines'],
                        ['title' => 'Meta-Analysis Guide', 'url' => '/blog/meta-analysis-guide', 'description' => 'Learn statistical synthesis and meta-analysis techniques'],
                        ['title' => 'Literature Search Strategies', 'url' => '/blog/literature-search-strategies', 'description' => 'Effective database searching and source identification'],
                    ],
                ],
            ],
        ];

        // Upsert content blocks
        foreach ($blocks as $i => $block) {
            PageContent::updateOrCreate(
                [
                    'page_id' => $page->id,
                    'block_key' => $block['block_key'],
                ],
                [
                    'block_type' => $block['block_type'],
                    'content' => $block['content'],
                    'order' => $block['order'] ?? ($i + 1),
                    'is_active' => true,
                ]
            );
        }

        $this->command->info('Literature Reviews service content seeded successfully!');
    }
}
