import React, { useState } from 'react';
import { Button } from '@/Components/ui/Button';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Textarea } from '@/Components/ui/Textarea';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Save, Eye } from 'lucide-react';

/**
 * PriceCalculatorBlockEditor - Admin editor for Price Calculator section
 * 
 * This component allows admins to customize the price calculator section
 * that can be added to any page (homepage, service pages, blog posts)
 */
export default function PriceCalculatorBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {
        eyebrow: '',
        title: 'Get Instant Pricing',
        subtitle: 'Calculate the cost of your assignment in seconds. No commitment required.',
        description: '',
        buttonText: 'Calculate Your Price',
        buttonVariant: 'primary', // 'primary', 'secondary', 'outline'
        showEyebrow: true,
        showDescription: false,
        backgroundColor: 'gradient', // 'gradient', 'white', 'gray', 'blue'
        alignment: 'center', // 'center', 'left'
    });

    const [activeTab, setActiveTab] = useState('content');

    const handleSave = () => {
        onUpdate(block.id, content);
    };

    const tabs = [
        { id: 'content', label: 'Content', icon: '📝' },
        { id: 'button', label: 'Button', icon: '🔘' },
        { id: 'style', label: 'Style', icon: '🎨' },
    ];

    const backgroundOptions = [
        { value: 'gradient', label: 'Gradient (Blue to Indigo)', preview: 'bg-gradient-to-br from-blue-50 to-indigo-50' },
        { value: 'white', label: 'White/Light', preview: 'bg-white' },
        { value: 'gray', label: 'Gray', preview: 'bg-gray-50' },
        { value: 'blue', label: 'Blue', preview: 'bg-blue-50' },
    ];

    const buttonVariants = [
        { value: 'primary', label: 'Primary (Blue)', preview: 'bg-blue-600 text-white' },
        { value: 'secondary', label: 'Secondary (Gray)', preview: 'bg-gray-600 text-white' },
        { value: 'outline', label: 'Outline', preview: 'border-2 border-blue-600 text-blue-600' },
    ];

    return (
        <div className="space-y-6">
            {/* Tab Navigation */}
            <div className="flex flex-wrap gap-2 border-b border-gray-200 dark:border-gray-700 pb-3">
                {tabs.map((tab) => (
                    <button
                        key={tab.id}
                        type="button"
                        onClick={() => setActiveTab(tab.id)}
                        className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                            activeTab === tab.id
                                ? 'bg-blue-600 text-white'
                                : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                        }`}
                    >
                        {tab.icon} {tab.label}
                    </button>
                ))}
            </div>

            {/* Content Tab */}
            {activeTab === 'content' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white flex items-center gap-2">
                        📝 Section Content
                    </h3>

                    <div className="space-y-4">
                        {/* Eyebrow */}
                        <div>
                            <div className="flex items-center justify-between mb-2">
                                <Label htmlFor="eyebrow">Eyebrow Text (Optional)</Label>
                                <label className="flex items-center gap-2 text-sm">
                                    <input
                                        type="checkbox"
                                        checked={content.showEyebrow}
                                        onChange={(e) => setContent({ ...content, showEyebrow: e.target.checked })}
                                        className="rounded"
                                    />
                                    <span className="text-gray-600 dark:text-gray-400">Show eyebrow</span>
                                </label>
                            </div>
                            <Input
                                id="eyebrow"
                                value={content.eyebrow || ''}
                                onChange={(e) => setContent({ ...content, eyebrow: e.target.value })}
                                placeholder="e.g., PRICING CALCULATOR"
                                disabled={!content.showEyebrow}
                            />
                            <p className="text-xs text-gray-500 mt-1">Small text above the main title (uppercase, typically)</p>
                        </div>

                        {/* Title */}
                        <div>
                            <Label htmlFor="title">Main Title *</Label>
                            <Input
                                id="title"
                                value={content.title || ''}
                                onChange={(e) => setContent({ ...content, title: e.target.value })}
                                placeholder="e.g., Get Instant Pricing for Your Assignment"
                                required
                            />
                            <p className="text-xs text-gray-500 mt-1">Main heading for the section</p>
                        </div>

                        {/* Subtitle */}
                        <div>
                            <Label htmlFor="subtitle">Subtitle</Label>
                            <Textarea
                                id="subtitle"
                                value={content.subtitle || ''}
                                onChange={(e) => setContent({ ...content, subtitle: e.target.value })}
                                placeholder="e.g., Calculate the cost of your academic paper in seconds. No commitment required."
                                rows={3}
                            />
                            <p className="text-xs text-gray-500 mt-1">Supporting text below the title</p>
                        </div>

                        {/* Optional Description */}
                        <div>
                            <div className="flex items-center justify-between mb-2">
                                <Label>Extended Description (Optional)</Label>
                                <label className="flex items-center gap-2 text-sm">
                                    <input
                                        type="checkbox"
                                        checked={content.showDescription}
                                        onChange={(e) => setContent({ ...content, showDescription: e.target.checked })}
                                        className="rounded"
                                    />
                                    <span className="text-gray-600 dark:text-gray-400">Show description</span>
                                </label>
                            </div>
                            {content.showDescription && (
                                <>
                                    <RichTextEditor
                                        value={content.description || ''}
                                        onChange={(value) => setContent({ ...content, description: value })}
                                        placeholder="Additional information about the price calculator..."
                                        height={150}
                                    />
                                    <p className="text-xs text-gray-500 mt-1">Optional rich text description (supports formatting)</p>
                                </>
                            )}
                        </div>
                    </div>
                </div>
            )}

            {/* Button Tab */}
            {activeTab === 'button' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white flex items-center gap-2">
                        🔘 Button Configuration
                    </h3>

                    <div className="space-y-4">
                        {/* Button Text */}
                        <div>
                            <Label htmlFor="buttonText">Button Text</Label>
                            <Input
                                id="buttonText"
                                value={content.buttonText || ''}
                                onChange={(e) => setContent({ ...content, buttonText: e.target.value })}
                                placeholder="e.g., Calculate Your Price"
                            />
                            <p className="text-xs text-gray-500 mt-1">Text displayed on the calculator trigger button</p>
                        </div>

                        {/* Button Variant */}
                        <div>
                            <Label>Button Style</Label>
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-3 mt-2">
                                {buttonVariants.map((variant) => (
                                    <button
                                        key={variant.value}
                                        type="button"
                                        onClick={() => setContent({ ...content, buttonVariant: variant.value })}
                                        className={`p-4 rounded-lg border-2 transition-all ${
                                            content.buttonVariant === variant.value
                                                ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20'
                                                : 'border-gray-200 dark:border-gray-700 hover:border-gray-300'
                                        }`}
                                    >
                                        <div className="text-sm font-medium text-gray-900 dark:text-white mb-2">
                                            {variant.label}
                                        </div>
                                        <div className={`px-4 py-2 rounded-lg text-sm font-medium ${variant.preview}`}>
                                            Preview
                                        </div>
                                    </button>
                                ))}
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Style Tab */}
            {activeTab === 'style' && (
                <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white flex items-center gap-2">
                        🎨 Section Styling
                    </h3>

                    <div className="space-y-4">
                        {/* Background Color */}
                        <div>
                            <Label>Background Style</Label>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-3 mt-2">
                                {backgroundOptions.map((bg) => (
                                    <button
                                        key={bg.value}
                                        type="button"
                                        onClick={() => setContent({ ...content, backgroundColor: bg.value })}
                                        className={`p-4 rounded-lg border-2 transition-all ${
                                            content.backgroundColor === bg.value
                                                ? 'border-blue-500 ring-2 ring-blue-200'
                                                : 'border-gray-200 dark:border-gray-700 hover:border-gray-300'
                                        }`}
                                    >
                                        <div className="text-sm font-medium text-gray-900 dark:text-white mb-2">
                                            {bg.label}
                                        </div>
                                        <div className={`h-12 rounded ${bg.preview} border border-gray-300`}></div>
                                    </button>
                                ))}
                            </div>
                        </div>

                        {/* Text Alignment */}
                        <div>
                            <Label>Text Alignment</Label>
                            <div className="flex gap-3 mt-2">
                                <button
                                    type="button"
                                    onClick={() => setContent({ ...content, alignment: 'center' })}
                                    className={`flex-1 px-4 py-3 rounded-lg border-2 text-sm font-medium transition-all ${
                                        content.alignment === 'center'
                                            ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20 text-blue-700 dark:text-blue-300'
                                            : 'border-gray-200 dark:border-gray-700 text-gray-700 dark:text-gray-300 hover:border-gray-300'
                                    }`}
                                >
                                    Center Aligned
                                </button>
                                <button
                                    type="button"
                                    onClick={() => setContent({ ...content, alignment: 'left' })}
                                    className={`flex-1 px-4 py-3 rounded-lg border-2 text-sm font-medium transition-all ${
                                        content.alignment === 'left'
                                            ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20 text-blue-700 dark:text-blue-300'
                                            : 'border-gray-200 dark:border-gray-700 text-gray-700 dark:text-gray-300 hover:border-gray-300'
                                    }`}
                                >
                                    Left Aligned
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Preview Section */}
            <div className="border-t pt-6">
                <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-6">
                    <div className="flex items-center gap-2 mb-4">
                        <Eye className="h-5 w-5 text-gray-600 dark:text-gray-400" />
                        <h4 className="font-semibold text-gray-900 dark:text-white">Preview</h4>
                    </div>
                    <div className={`p-8 rounded-lg ${
                        content.backgroundColor === 'gradient' ? 'bg-gradient-to-br from-blue-50 to-indigo-50' :
                        content.backgroundColor === 'white' ? 'bg-white' :
                        content.backgroundColor === 'gray' ? 'bg-gray-50' :
                        'bg-blue-50'
                    }`}>
                        <div className={`max-w-4xl mx-auto ${content.alignment === 'center' ? 'text-center' : 'text-left'}`}>
                            {content.showEyebrow && content.eyebrow && (
                                <div className="text-sm font-semibold text-blue-600 uppercase tracking-wide mb-2">
                                    {content.eyebrow}
                                </div>
                            )}
                            <h2 className="text-3xl font-bold text-gray-900 mb-4">
                                {content.title || 'Get Instant Pricing'}
                            </h2>
                            {content.subtitle && (
                                <p className="text-lg text-gray-600 mb-6">
                                    {content.subtitle}
                                </p>
                            )}
                            {content.showDescription && content.description && (
                                <div 
                                    className="text-gray-600 mb-6 prose prose-sm max-w-none"
                                    dangerouslySetInnerHTML={{ __html: content.description }}
                                />
                            )}
                            <button
                                type="button"
                                className={`px-8 py-4 rounded-lg text-lg font-semibold transition-all ${
                                    content.buttonVariant === 'primary' ? 'bg-blue-600 text-white hover:bg-blue-700' :
                                    content.buttonVariant === 'secondary' ? 'bg-gray-600 text-white hover:bg-gray-700' :
                                    'border-2 border-blue-600 text-blue-600 hover:bg-blue-50'
                                }`}
                            >
                                {content.buttonText || 'Calculate Your Price'}
                            </button>
                        </div>
                    </div>
                    <p className="text-xs text-gray-500 mt-4 text-center">
                        💡 This preview shows how the section will appear on the page. The actual calculator will open in a popup.
                    </p>
                </div>
            </div>

            {/* Save Button */}
            <div className="flex justify-end pt-4 border-t">
                <Button onClick={handleSave} className="gap-2">
                    <Save className="h-4 w-4" />
                    Save Price Calculator Section
                </Button>
            </div>
        </div>
    );
}
