import React, { useState } from 'react';
import { Input } from '@/Components/ui/Input';
import { Label } from '@/Components/ui/Label';
import { Button } from '@/Components/ui/Button';
import RichTextEditor from '@/Components/Admin/RichTextEditor';
import { Plus, X, ChevronUp, ChevronDown, Save } from 'lucide-react';

export default function PricingReasonsBlockEditor({ block, onUpdate }) {
    const [content, setContent] = useState(block.content || {});
    const [reasons, setReasons] = useState(content.reasons || []);

    const handleSave = () => {
        onUpdate(block.id, { ...content, reasons });
    };

    const addReason = () => {
        setReasons([...reasons, { 
            title: '', 
            description: '',
            link: '/services/essay-writing',
            linkText: 'Learn More'
        }]);
    };

    const removeReason = (index) => {
        setReasons(reasons.filter((_, i) => i !== index));
    };

    const updateReason = (index, field, value) => {
        const updated = [...reasons];
        updated[index] = { ...updated[index], [field]: value };
        setReasons(updated);
    };

    const moveReasonUp = (index) => {
        if (index === 0) return;
        const updated = [...reasons];
        [updated[index - 1], updated[index]] = [updated[index], updated[index - 1]];
        setReasons(updated);
    };

    const moveReasonDown = (index) => {
        if (index === reasons.length - 1) return;
        const updated = [...reasons];
        [updated[index], updated[index + 1]] = [updated[index + 1], updated[index]];
        setReasons(updated);
    };

    return (
        <div className="space-y-6">
            <div>
                <Label htmlFor="title">Section Title</Label>
                <Input
                    id="title"
                    value={content.title || ''}
                    onChange={(e) => setContent({...content, title: e.target.value})}
                />
            </div>

            <div>
                <Label htmlFor="description">Section Description</Label>
                <RichTextEditor
                    value={content.description || ''}
                    onChange={(value) => setContent({...content, description: value})}
                    height={150}
                />
            </div>

            <div className="border-t pt-4">
                <div className="flex justify-between items-center mb-4">
                    <Label>Reasons ({reasons.length})</Label>
                    <Button
                        type="button"
                        onClick={addReason}
                        className="bg-green-600 hover:bg-green-700 text-white"
                    >
                        <Plus className="h-4 w-4 mr-1" />
                        Add Reason
                    </Button>
                </div>

                <div className="space-y-6">
                    {reasons.map((reason, index) => (
                        <div key={index} className="border border-gray-200 rounded-lg p-4 bg-gray-50">
                            <div className="flex justify-between items-center mb-3">
                                <span className="text-sm font-semibold text-gray-700">Reason #{index + 1}</span>
                                <div className="flex gap-2">
                                    <button
                                        type="button"
                                        onClick={() => moveReasonUp(index)}
                                        disabled={index === 0}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move up"
                                    >
                                        <ChevronUp className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => moveReasonDown(index)}
                                        disabled={index === reasons.length - 1}
                                        className="p-1 text-gray-500 hover:text-gray-700 disabled:opacity-30 disabled:cursor-not-allowed"
                                        title="Move down"
                                    >
                                        <ChevronDown className="h-4 w-4" />
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => removeReason(index)}
                                        className="p-1 text-red-500 hover:text-red-700"
                                        title="Remove reason"
                                    >
                                        <X className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>

                            <div className="space-y-3">
                                <div>
                                    <Label htmlFor={`title-${index}`}>Reason Title</Label>
                                    <Input
                                        id={`title-${index}`}
                                        value={reason.title || ''}
                                        onChange={(e) => updateReason(index, 'title', e.target.value)}
                                        placeholder="e.g., Cheapest Essays Online with Quality Guarantee"
                                    />
                                </div>

                                <div>
                                    <Label htmlFor={`description-${index}`}>Description</Label>
                                    <RichTextEditor
                                        value={reason.description || ''}
                                        onChange={(value) => updateReason(index, 'description', value)}
                                        height={120}
                                    />
                                </div>

                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <Label htmlFor={`link-${index}`}>Link URL</Label>
                                        <Input
                                            id={`link-${index}`}
                                            value={reason.link || ''}
                                            onChange={(e) => updateReason(index, 'link', e.target.value)}
                                            placeholder="/services/essay-writing"
                                        />
                                    </div>
                                    <div>
                                        <Label htmlFor={`linkText-${index}`}>Link Text</Label>
                                        <Input
                                            id={`linkText-${index}`}
                                            value={reason.linkText || ''}
                                            onChange={(e) => updateReason(index, 'linkText', e.target.value)}
                                            placeholder="View Cheapest Essay Services"
                                        />
                                    </div>
                                </div>
                            </div>
                        </div>
                    ))}

                    {reasons.length === 0 && (
                        <div className="text-center py-8 text-gray-500 border-2 border-dashed border-gray-300 rounded-lg">
                            No reasons yet. Click "Add Reason" to get started.
                        </div>
                    )}
                </div>
            </div>
            
            <Button
                type="button"
                onClick={handleSave}
                className="bg-blue-600 hover:bg-blue-700 text-white"
            >
                <Save className="h-4 w-4 mr-2" />
                Save Changes
            </Button>
        </div>
    );
}



