import React, { useState } from 'react';
import { router } from '@inertiajs/react';
import {
    StarIcon,
    CheckCircleIcon,
    XCircleIcon,
    ExclamationTriangleIcon
} from '@heroicons/react/24/outline';

export default function QualityAssessmentForm({ revision, onClose, onSuccess }) {
    const [formData, setFormData] = useState({
        quality_rating: 0,
        quality_notes: '',
    });
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [hoveredRating, setHoveredRating] = useState(0);

    const handleRatingChange = (rating) => {
        setFormData(prev => ({ ...prev, quality_rating: rating }));
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        setIsSubmitting(true);

        router.post(route('admin.payment-impacts.assess-quality', revision.id), formData, {
            onSuccess: () => {
                setIsSubmitting(false);
                onSuccess && onSuccess();
            },
            onError: () => {
                setIsSubmitting(false);
            }
        });
    };

    const getRatingDescription = (rating) => {
        if (rating >= 9) return 'Excellent - Full bonus applied';
        if (rating >= 7) return 'Good - No impact';
        if (rating >= 5) return 'Fair - 60% penalty applied';
        if (rating >= 1) return 'Poor - Full penalty applied';
        return 'Select a rating';
    };

    const getRatingColor = (rating) => {
        if (rating >= 9) return 'text-green-600';
        if (rating >= 7) return 'text-blue-600';
        if (rating >= 5) return 'text-yellow-600';
        if (rating >= 1) return 'text-red-600';
        return 'text-gray-400';
    };

    return (
        <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
            <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
                <div className="mt-3">
                    {/* Header */}
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="text-lg font-medium text-gray-900">
                            Quality Assessment
                        </h3>
                        <button
                            onClick={onClose}
                            className="text-gray-400 hover:text-gray-600"
                        >
                            <XCircleIcon className="h-6 w-6" />
                        </button>
                    </div>

                    {/* Revision Info */}
                    <div className="bg-gray-50 rounded-lg p-3 mb-4">
                        <p className="text-sm text-gray-600">
                            <span className="font-medium">Revision:</span> #{revision.id}
                        </p>
                        <p className="text-sm text-gray-600">
                            <span className="font-medium">Type:</span> {revision.revision_type}
                        </p>
                        <p className="text-sm text-gray-600">
                            <span className="font-medium">Writer:</span> {revision.assigned_to?.name || 'N/A'}
                        </p>
                    </div>

                    <form onSubmit={handleSubmit} className="space-y-4">
                        {/* Quality Rating */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Quality Rating (1-10)
                            </label>
                            <div className="flex items-center space-x-1">
                                {[1, 2, 3, 4, 5, 6, 7, 8, 9, 10].map((rating) => (
                                    <button
                                        key={rating}
                                        type="button"
                                        onClick={() => handleRatingChange(rating)}
                                        onMouseEnter={() => setHoveredRating(rating)}
                                        onMouseLeave={() => setHoveredRating(0)}
                                        className={`p-1 transition-colors ${
                                            (hoveredRating >= rating || formData.quality_rating >= rating)
                                                ? 'text-yellow-400'
                                                : 'text-gray-300'
                                        }`}
                                    >
                                        <StarIcon className="h-8 w-8" />
                                    </button>
                                ))}
                            </div>
                            <p className={`text-sm mt-2 ${getRatingColor(formData.quality_rating)}`}>
                                {getRatingDescription(formData.quality_rating)}
                            </p>
                        </div>

                        {/* Quality Notes */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Assessment Notes (Optional)
                            </label>
                            <textarea
                                value={formData.quality_notes}
                                onChange={(e) => setFormData(prev => ({ ...prev, quality_notes: e.target.value }))}
                                rows={3}
                                className="block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                                placeholder="Provide feedback on the revision quality..."
                            />
                        </div>

                        {/* Payment Impact Preview */}
                        {formData.quality_rating > 0 && (
                            <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
                                <h4 className="text-sm font-medium text-blue-900 mb-2">
                                    Payment Impact Preview
                                </h4>
                                <div className="space-y-1 text-sm text-blue-800">
                                    {formData.quality_rating >= 9 && (
                                        <div className="flex items-center space-x-2">
                                            <CheckCircleIcon className="h-4 w-4 text-green-600" />
                                            <span>Quality Bonus: +$5.00</span>
                                        </div>
                                    )}
                                    {formData.quality_rating <= 6 && (
                                        <div className="flex items-center space-x-2">
                                            <ExclamationTriangleIcon className="h-4 w-4 text-red-600" />
                                            <span>
                                                Quality Penalty: 
                                                {formData.quality_rating <= 4 ? ' -$10.00' : ' -$6.00 (60%)'}
                                            </span>
                                        </div>
                                    )}
                                    {formData.quality_rating >= 7 && formData.quality_rating <= 8 && (
                                        <div className="flex items-center space-x-2">
                                            <CheckCircleIcon className="h-4 w-4 text-blue-600" />
                                            <span>No payment impact</span>
                                        </div>
                                    )}
                                </div>
                            </div>
                        )}

                        {/* Submit Button */}
                        <div className="flex space-x-3 pt-4">
                            <button
                                type="button"
                                onClick={onClose}
                                className="flex-1 px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                            >
                                Cancel
                            </button>
                            <button
                                type="submit"
                                disabled={!formData.quality_rating || isSubmitting}
                                className="flex-1 inline-flex items-center justify-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {isSubmitting ? (
                                    <>
                                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                        Assessing...
                                    </>
                                ) : (
                                    <>
                                        <CheckCircleIcon className="h-4 w-4 mr-2" />
                                        Assess Quality
                                    </>
                                )}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
} 