import React, { useState } from 'react';
import { router } from '@inertiajs/react';
import {
    XCircleIcon,
    ExclamationTriangleIcon,
    CheckCircleIcon
} from '@heroicons/react/24/outline';

export default function RevisionRejectionForm({ revision, onClose, onSuccess }) {
    const [formData, setFormData] = useState({
        rejection_reason: '',
        admin_notes: '',
    });
    const [isSubmitting, setIsSubmitting] = useState(false);

    const handleSubmit = (e) => {
        e.preventDefault();
        setIsSubmitting(true);

        router.post(route('admin.payment-impacts.reject', revision.id), formData, {
            onSuccess: () => {
                setIsSubmitting(false);
                onSuccess && onSuccess();
            },
            onError: () => {
                setIsSubmitting(false);
            }
        });
    };

    return (
        <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
            <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
                <div className="mt-3">
                    {/* Header */}
                    <div className="flex items-center justify-between mb-4">
                        <h3 className="text-lg font-medium text-gray-900">
                            Reject Revision
                        </h3>
                        <button
                            onClick={onClose}
                            className="text-gray-400 hover:text-gray-600"
                        >
                            <XCircleIcon className="h-6 w-6" />
                        </button>
                    </div>

                    {/* Warning */}
                    <div className="bg-red-50 border border-red-200 rounded-lg p-3 mb-4">
                        <div className="flex items-center space-x-2">
                            <ExclamationTriangleIcon className="h-5 w-5 text-red-600" />
                            <span className="text-sm font-medium text-red-800">
                                This action will reject the revision and apply penalties
                            </span>
                        </div>
                    </div>

                    {/* Revision Info */}
                    <div className="bg-gray-50 rounded-lg p-3 mb-4">
                        <p className="text-sm text-gray-600">
                            <span className="font-medium">Revision:</span> #{revision.id}
                        </p>
                        <p className="text-sm text-gray-600">
                            <span className="font-medium">Type:</span> {revision.revision_type}
                        </p>
                        <p className="text-sm text-gray-600">
                            <span className="font-medium">Writer:</span> {revision.assigned_to?.name || 'N/A'}
                        </p>
                        <p className="text-sm text-gray-600">
                            <span className="font-medium">Status:</span> {revision.status}
                        </p>
                    </div>

                    <form onSubmit={handleSubmit} className="space-y-4">
                        {/* Rejection Reason */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Rejection Reason <span className="text-red-500">*</span>
                            </label>
                            <textarea
                                value={formData.rejection_reason}
                                onChange={(e) => setFormData(prev => ({ ...prev, rejection_reason: e.target.value }))}
                                rows={3}
                                required
                                className="block w-full border-gray-300 rounded-md shadow-sm focus:ring-red-500 focus:border-red-500"
                                placeholder="Explain why this revision is being rejected..."
                            />
                        </div>

                        {/* Admin Notes */}
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                Admin Notes (Optional)
                            </label>
                            <textarea
                                value={formData.admin_notes}
                                onChange={(e) => setFormData(prev => ({ ...prev, admin_notes: e.target.value }))}
                                rows={2}
                                className="block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                                placeholder="Internal notes for admin reference..."
                            />
                        </div>

                        {/* Payment Impact Warning */}
                        <div className="bg-orange-50 border border-orange-200 rounded-lg p-3">
                            <h4 className="text-sm font-medium text-orange-900 mb-2">
                                Payment Impact
                            </h4>
                            <div className="text-sm text-orange-800">
                                <p>• Quality rating will be set to 1/10</p>
                                <p>• Full quality penalty will be applied (-$10.00)</p>
                                <p>• Revision status will be changed to "rejected"</p>
                            </div>
                        </div>

                        {/* Submit Button */}
                        <div className="flex space-x-3 pt-4">
                            <button
                                type="button"
                                onClick={onClose}
                                className="flex-1 px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                            >
                                Cancel
                            </button>
                            <button
                                type="submit"
                                disabled={!formData.rejection_reason.trim() || isSubmitting}
                                className="flex-1 inline-flex items-center justify-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {isSubmitting ? (
                                    <>
                                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                        Rejecting...
                                    </>
                                ) : (
                                    <>
                                        <XCircleIcon className="h-4 w-4 mr-2" />
                                        Reject Revision
                                    </>
                                )}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
} 