import React, { useCallback, useEffect, useMemo, useRef } from "react";
import { EditorContent, useEditor } from "@tiptap/react";
import StarterKit from "@tiptap/starter-kit";
import Placeholder from "@tiptap/extension-placeholder";
import Link from "@tiptap/extension-link";
import Underline from "@tiptap/extension-underline";
import TextAlign from "@tiptap/extension-text-align";
import Image from "@tiptap/extension-image";
import TextStyle from "@tiptap/extension-text-style";
import Color from "@tiptap/extension-color";
import Table from "@tiptap/extension-table";
import TableRow from "@tiptap/extension-table-row";
import TableCell from "@tiptap/extension-table-cell";
import TableHeader from "@tiptap/extension-table-header";
import TaskList from "@tiptap/extension-task-list";
import TaskItem from "@tiptap/extension-task-item";
import Highlight from "@tiptap/extension-highlight";
import Subscript from "@tiptap/extension-subscript";
import Superscript from "@tiptap/extension-superscript";
import { Node } from "@tiptap/core";
import clsx from "clsx";

// Custom Div extension to preserve div elements with classes
const Div = Node.create({
    name: "div",
    group: "block",
    content: "block+",
    parseHTML() {
        return [{ tag: "div" }];
    },
    renderHTML({ HTMLAttributes }) {
        return ["div", HTMLAttributes, 0];
    },
    addAttributes() {
        return {
            class: {
                default: null,
                parseHTML: element => element.getAttribute("class"),
                renderHTML: attributes => {
                    if (!attributes.class) {
                        return {};
                    }
                    return {
                        class: attributes.class,
                    };
                },
            },
            style: {
                default: null,
                parseHTML: element => element.getAttribute("style"),
                renderHTML: attributes => {
                    if (!attributes.style) {
                        return {};
                    }
                    return {
                        style: attributes.style,
                    };
                },
            },
        };
    },
});

const TOOLBAR_BUTTON_BASE =
    "inline-flex items-center justify-center px-2 py-1 text-sm rounded border border-transparent transition-colors";

const ToolbarButton = ({ active, disabled, onClick, children, title }) => (
    <button
        type="button"
        className={clsx(
            TOOLBAR_BUTTON_BASE,
            "text-gray-700 dark:text-gray-200",
            active
                ? "bg-blue-100 text-blue-700 dark:bg-blue-500/20 dark:text-blue-200"
                : "bg-gray-100 hover:bg-gray-200 dark:bg-gray-800 dark:hover:bg-gray-700",
            disabled && "opacity-50 cursor-not-allowed hover:bg-gray-100"
        )}
        onMouseDown={(event) => event.preventDefault()}
        onClick={() => onClick?.()}
        disabled={disabled}
        title={title}
    >
        {children}
    </button>
);

const headingOptions = [
    { level: 1, label: "H1" },
    { level: 2, label: "H2" },
    { level: 3, label: "H3" },
    { level: 4, label: "H4" },
];

export default function RichTextEditor({
    value = "",
    onChange,
    height = 400,
    placeholder = "Enter content...",
    disabled = false,
}) {
    const fileInputRef = useRef(null);
    const [showHtmlView, setShowHtmlView] = React.useState(false);
    const [htmlContent, setHtmlContent] = React.useState("");

    const extensions = useMemo(
        () => [
            StarterKit.configure({
                heading: {
                    levels: headingOptions.map((option) => option.level),
                    HTMLAttributes: {
                        class: null,
                    },
                    addAttributes() {
                        return {
                            ...this.parent?.(),
                            style: {
                                default: null,
                                parseHTML: element => element.getAttribute('style'),
                                renderHTML: attributes => {
                                    if (!attributes.style) {
                                        return {};
                                    }
                                    return {
                                        style: attributes.style,
                                    };
                                },
                            },
                        };
                    },
                },
                paragraph: {
                    HTMLAttributes: {
                        class: null,
                    },
                    addAttributes() {
                        return {
                            ...this.parent?.(),
                            style: {
                                default: null,
                                parseHTML: element => element.getAttribute('style'),
                                renderHTML: attributes => {
                                    if (!attributes.style) {
                                        return {};
                                    }
                                    return {
                                        style: attributes.style,
                                    };
                                },
                            },
                        };
                    },
                },
                bold: {
                    // Ensure bold/strong tags are preserved
                    parseHTML() {
                        return [
                            { tag: 'strong' },
                            { tag: 'b', getAttrs: () => ({ style: 'font-weight: bold' }) },
                            { style: 'font-weight', getAttrs: value => /^(bold(er)?|[5-9]\d{2,})$/.test(value) && null },
                        ];
                    },
                    renderHTML() {
                        return ['strong', 0];
                    },
                },
                italic: {
                    // Ensure italic/em tags are preserved
                    parseHTML() {
                        return [
                            { tag: 'i' },
                            { tag: 'em' },
                            { style: 'font-style=italic' },
                        ];
                    },
                    renderHTML() {
                        return ['em', 0];
                    },
                },
            }),
            Div,
            Underline,
            TextStyle,
            Color,
            Highlight.configure({
                multicolor: true,
            }),
            Subscript,
            Superscript,
            TaskList,
            TaskItem.configure({
                nested: true,
            }),
            Table.extend({
                addAttributes() {
                    return {
                        ...this.parent?.(),
                        class: {
                            default: null,
                            parseHTML: element => element.getAttribute('class'),
                            renderHTML: attributes => {
                                if (!attributes.class) {
                                    return {};
                                }
                                return {
                                    class: attributes.class,
                                };
                            },
                        },
                    };
                },
            }).configure({
                resizable: true,
                allowTableNodeSelection: true,
            }),
            TableRow.extend({
                addAttributes() {
                    return {
                        ...this.parent?.(),
                        class: {
                            default: null,
                            parseHTML: element => element.getAttribute('class'),
                            renderHTML: attributes => {
                                if (!attributes.class) {
                                    return {};
                                }
                                return {
                                    class: attributes.class,
                                };
                            },
                        },
                    };
                },
            }),
            TableHeader.extend({
                addAttributes() {
                    return {
                        ...this.parent?.(),
                        class: {
                            default: null,
                            parseHTML: element => element.getAttribute('class'),
                            renderHTML: attributes => {
                                if (!attributes.class) {
                                    return {};
                                }
                                return {
                                    class: attributes.class,
                                };
                            },
                        },
                    };
                },
            }),
            TableCell.extend({
                addAttributes() {
                    return {
                        ...this.parent?.(),
                        class: {
                            default: null,
                            parseHTML: element => element.getAttribute('class'),
                            renderHTML: attributes => {
                                if (!attributes.class) {
                                    return {};
                                }
                                return {
                                    class: attributes.class,
                                };
                            },
                        },
                        style: {
                            default: null,
                            parseHTML: element => element.getAttribute('style'),
                            renderHTML: attributes => {
                                if (!attributes.style) {
                                    return {};
                                }
                                return {
                                    style: attributes.style,
                                };
                            },
                        },
                    };
                },
            }),
            Placeholder.configure({
                placeholder,
            }),
            Link.extend({
                inclusive: false,
                addAttributes() {
                    return {
                        ...this.parent?.(),
                        href: {
                            default: null,
                            parseHTML: element => {
                                const href = element.getAttribute('href');
                                // Always preserve href, even if empty or invalid
                                return href || null;
                            },
                            renderHTML: attributes => {
                                // Always render href if it exists, even if empty
                                if (attributes.href === null || attributes.href === undefined) {
                                    return {};
                                }
                                return {
                                    href: attributes.href || '',
                                };
                            },
                        },
                        target: {
                            default: null,
                            parseHTML: element => element.getAttribute('target') || null,
                            renderHTML: attributes => {
                                if (!attributes.target) {
                                    return {};
                                }
                                return {
                                    target: attributes.target,
                                };
                            },
                        },
                        rel: {
                            default: null,
                            parseHTML: element => element.getAttribute('rel') || null,
                            renderHTML: attributes => {
                                if (!attributes.rel) {
                                    return {};
                                }
                                return {
                                    rel: attributes.rel,
                                };
                            },
                        },
                    };
                },
                parseHTML() {
                    return [
                        {
                            tag: 'a[href]',
                            getAttrs: element => {
                                const href = element.getAttribute('href');
                                // Accept any link with an href attribute, even if empty
                                return href !== null ? {} : false;
                            },
                        },
                    ];
                },
            }).configure({
                HTMLAttributes: {
                    rel: "noopener noreferrer",
                },
                openOnClick: false,
                autolink: false, // Disable autolink to prevent conflicts
                defaultProtocol: "https",
            }),
            TextAlign.configure({
                types: ["heading", "paragraph", "tableCell"],
            }),
            Image.configure({
                allowBase64: true,
            }),
        ],
        [placeholder]
    );

    const editor = useEditor(
        {
            extensions,
            content: value || "",
            editable: !disabled,
            parseOptions: {
                preserveWhitespace: "full",
            },
            onUpdate: ({ editor }) => {
                if (!onChange) return;
                // Only update from editor if not in HTML view
                if (!showHtmlView) {
                    // Get HTML from editor - this goes through TipTap's schema
                    let html = editor.getHTML();
                    
                    // CRITICAL FIX: Check if links were stripped and restore them
                    // TipTap might strip links during getHTML(), so we need to preserve them
                    // by checking the editor's JSON representation
                    try {
                        const json = editor.getJSON();
                        // If we have links in the JSON but they're missing from HTML,
                        // we need to manually reconstruct them
                        // For now, we'll rely on the Link extension configuration to preserve them
                    } catch (e) {
                        // If JSON parsing fails, continue with HTML
                    }
                    
                    const normalized = html === "<p></p>" ? "" : html;
                    onChange(normalized);
                }
            },
        },
        [extensions, showHtmlView]
    );

    // Initialize htmlContent when value changes
    useEffect(() => {
        if (value !== undefined) {
            setHtmlContent(value || "");
        }
    }, [value]);

    useEffect(() => {
        if (!editor) {
            return;
        }

        // If in HTML view, don't sync with editor - preserve raw HTML
        if (showHtmlView) {
            return;
        }

        const currentHTML = editor.getHTML();
        const nextHTML = value || "";
        if (nextHTML !== currentHTML && nextHTML !== htmlContent) {
            editor.commands.setContent(nextHTML || "<p></p>", false);
        }
    }, [editor, value, showHtmlView, htmlContent]);

    useEffect(() => {
        if (!editor) {
            return;
        }
        editor.setEditable(!disabled);
    }, [editor, disabled]);

    const setLink = useCallback(() => {
        if (!editor) return;
        const previousUrl = editor.getAttributes("link").href;
        const url = window.prompt("Enter URL", previousUrl || "https://");

        if (url === null) {
            return;
        }

        if (url === "") {
            editor.chain().focus().extendMarkRange("link").unsetLink().run();
            return;
        }

        editor
            .chain()
            .focus()
            .extendMarkRange("link")
            .setLink({ href: url })
            .run();
    }, [editor]);

    const addImage = useCallback(() => {
        if (!editor) return;
        // Trigger hidden file input first for uploads, fallback to URL prompt
        if (fileInputRef.current) {
            fileInputRef.current.click();
        } else {
            const url = window.prompt("Image URL");
            if (url) {
                editor.chain().focus().setImage({ src: url }).run();
            }
        }
    }, [editor]);

    const handleImageFile = useCallback(
        async (event) => {
            if (!editor) return;
            const file = event.target.files?.[0];
            event.target.value = "";
            if (!file) return;

            const reader = new FileReader();
            reader.onload = () => {
                const src = reader.result;
                if (typeof src === "string") {
                    editor.chain().focus().setImage({ src }).run();
                }
            };
            reader.readAsDataURL(file);
        },
        [editor]
    );

    if (!editor) {
        return (
            <div className="flex items-center justify-center h-32 rounded-md border border-dashed border-gray-300 dark:border-gray-700 text-sm text-gray-500 dark:text-gray-400">
                Loading editor…
            </div>
        );
    }

    return (
        <div className="space-y-2">
            {!disabled && (
                <div className="flex flex-wrap gap-2 rounded-md border border-gray-200 bg-gray-50 p-2 dark:border-gray-700 dark:bg-gray-900/60">
                    <div className="flex flex-wrap items-center gap-2">
                        <ToolbarButton
                            onClick={() => editor.chain().focus().undo().run()}
                            disabled={!editor.can().chain().focus().undo().run()}
                            title="Undo"
                        >
                            ↺
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() => editor.chain().focus().redo().run()}
                            disabled={!editor.can().chain().focus().redo().run()}
                            title="Redo"
                        >
                            ↻
                        </ToolbarButton>
                    </div>
                    <div className="flex flex-wrap items-center gap-2 border-l border-gray-200 pl-2 dark:border-gray-700">
                        <select
                            className="rounded border border-gray-200 bg-white px-2 py-1 text-sm dark:border-gray-700 dark:bg-gray-800 dark:text-gray-200"
                            value={
                                headingOptions.find((option) =>
                                    editor.isActive("heading", {
                                        level: option.level,
                                    })
                                )?.level || "paragraph"
                            }
                            onChange={(event) => {
                                const level = event.target.value;
                                if (level === "paragraph") {
                                    editor.chain().focus().setParagraph().run();
                                } else {
                                    editor
                                        .chain()
                                        .focus()
                                        .setHeading({ level: Number(level) })
                                        .run();
                                }
                            }}
                        >
                            <option value="paragraph">Paragraph</option>
                            {headingOptions.map((option) => (
                                <option key={option.level} value={option.level}>
                                    {option.label}
                                </option>
                            ))}
                        </select>
                    </div>

                    <div className="flex flex-wrap items-center gap-2 border-l border-gray-200 pl-2 dark:border-gray-700">
                        <ToolbarButton
                            onClick={() => editor.chain().focus().toggleBold().run()}
                            active={editor.isActive("bold")}
                            title="Bold"
                        >
                            <span className="font-semibold">B</span>
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() => editor.chain().focus().toggleItalic().run()}
                            active={editor.isActive("italic")}
                            title="Italic"
                        >
                            <span className="italic">I</span>
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().toggleUnderline().run()
                            }
                            active={editor.isActive("underline")}
                            title="Underline"
                        >
                            <span className="underline">U</span>
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().toggleStrike().run()
                            }
                            active={editor.isActive("strike")}
                            title="Strikethrough"
                        >
                            <span className="line-through">S</span>
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() => {
                                const color = window.prompt("Text color hex", "#2563eb");
                                if (color) {
                                    editor.chain().focus().setColor(color).run();
                                }
                            }}
                            title="Text color"
                        >
                            🎨
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() => {
                                if (editor.isActive("highlight")) {
                                    editor.chain().focus().unsetHighlight().run();
                                } else {
                                    const color = window.prompt("Highlight color hex", "#fef08a");
                                    if (color) {
                                        editor.chain().focus().toggleHighlight({ color }).run();
                                    }
                                }
                            }}
                            active={editor.isActive("highlight")}
                            title="Highlight"
                        >
                            🖍️
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() => editor.chain().focus().toggleSubscript().run()}
                            active={editor.isActive("subscript")}
                            title="Subscript"
                        >
                            <span className="text-xs">x₂</span>
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() => editor.chain().focus().toggleSuperscript().run()}
                            active={editor.isActive("superscript")}
                            title="Superscript"
                        >
                            <span className="text-xs">x²</span>
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() => editor.chain().focus().unsetAllMarks().run()}
                            title="Clear formatting"
                        >
                            ⌫
                        </ToolbarButton>
                    </div>

                    <div className="flex flex-wrap items-center gap-2 border-l border-gray-200 pl-2 dark:border-gray-700">
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().setTextAlign("left").run()
                            }
                            active={editor.isActive({ textAlign: "left" })}
                            title="Align left"
                        >
                            ☰
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().setTextAlign("center").run()
                            }
                            active={editor.isActive({ textAlign: "center" })}
                            title="Align center"
                        >
                            ≡
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().setTextAlign("right").run()
                            }
                            active={editor.isActive({ textAlign: "right" })}
                            title="Align right"
                        >
                            ☰
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().setTextAlign("justify").run()
                            }
                            active={editor.isActive({ textAlign: "justify" })}
                            title="Justify"
                        >
                            ≣
                        </ToolbarButton>
                    </div>

                    <div className="flex flex-wrap items-center gap-2 border-l border-gray-200 pl-2 dark:border-gray-700">
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().toggleBulletList().run()
                            }
                            active={editor.isActive("bulletList")}
                            title="Bullet list"
                        >
                            ••
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().toggleOrderedList().run()
                            }
                            active={editor.isActive("orderedList")}
                            title="Numbered list"
                        >
                            1.
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().toggleTaskList().run()
                            }
                            active={editor.isActive("taskList")}
                            title="Task list"
                        >
                            ☑
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().toggleBlockquote().run()
                            }
                            active={editor.isActive("blockquote")}
                            title="Blockquote"
                        >
                            ❝
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().toggleCodeBlock().run()
                            }
                            active={editor.isActive("codeBlock")}
                            title="Code block"
                        >
                            {"</>"}
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().setHorizontalRule().run()
                            }
                            title="Horizontal rule"
                        >
                            ─
                        </ToolbarButton>
                    </div>

                    <div className="flex flex-wrap items-center gap-2 border-l border-gray-200 pl-2 dark:border-gray-700">
                        <ToolbarButton
                            onClick={setLink}
                            active={editor.isActive("link")}
                            title="Insert/edit link"
                        >
                            🔗
                        </ToolbarButton>
                        <ToolbarButton
                            onClick={() =>
                                editor.chain().focus().unsetLink().run()
                            }
                            title="Remove link"
                        >
                            ✖️
                        </ToolbarButton>
                        <ToolbarButton onClick={addImage} title="Insert image">
                            🖼️
                        </ToolbarButton>
                    </div>

                    <div className="flex flex-wrap items-center gap-2 border-l border-gray-200 pl-2 dark:border-gray-700">
                        <ToolbarButton
                            onClick={() => {
                                const rows = parseInt(window.prompt("Number of rows", "3") || "3");
                                const cols = parseInt(window.prompt("Number of columns", "3") || "3");
                                if (rows > 0 && cols > 0) {
                                    editor.chain().focus().insertTable({ rows, cols, withHeaderRow: true }).run();
                                }
                            }}
                            title="Insert table"
                        >
                            ⧉
                        </ToolbarButton>
                        {editor.isActive("table") && (
                            <>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().addColumnBefore().run()}
                                    disabled={!editor.can().addColumnBefore()}
                                    title="Add column before"
                                >
                                    ⬅️
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().addColumnAfter().run()}
                                    disabled={!editor.can().addColumnAfter()}
                                    title="Add column after"
                                >
                                    ➡️
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().deleteColumn().run()}
                                    disabled={!editor.can().deleteColumn()}
                                    title="Delete column"
                                >
                                    🗑️
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().addRowBefore().run()}
                                    disabled={!editor.can().addRowBefore()}
                                    title="Add row before"
                                >
                                    ⬆️
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().addRowAfter().run()}
                                    disabled={!editor.can().addRowAfter()}
                                    title="Add row after"
                                >
                                    ⬇️
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().deleteRow().run()}
                                    disabled={!editor.can().deleteRow()}
                                    title="Delete row"
                                >
                                    🗑️
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().mergeCells().run()}
                                    disabled={!editor.can().mergeCells()}
                                    title="Merge cells"
                                >
                                    ⬌
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().splitCell().run()}
                                    disabled={!editor.can().splitCell()}
                                    title="Split cell"
                                >
                                    ⬍
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().toggleHeaderRow().run()}
                                    title="Toggle header row"
                                >
                                    H
                                </ToolbarButton>
                                <ToolbarButton
                                    onClick={() => editor.chain().focus().deleteTable().run()}
                                    disabled={!editor.can().deleteTable()}
                                    title="Delete table"
                                >
                                    ⧈
                                </ToolbarButton>
                            </>
                        )}
                    </div>

                    <div className="flex flex-wrap items-center gap-2 border-l border-gray-200 pl-2 dark:border-gray-700">
                        <ToolbarButton
                            onClick={() => {
                                if (showHtmlView) {
                                    // Switching from HTML view to visual view
                                    // Update editor with HTML content
                                    if (editor) {
                                        editor.commands.setContent(htmlContent || "<p></p>", false);
                                    }
                                    setShowHtmlView(false);
                                } else {
                                    // Switching from visual view to HTML view
                                    // Save current HTML content
                                    if (editor) {
                                        const currentHtml = editor.getHTML();
                                        setHtmlContent(currentHtml);
                                    }
                                    setShowHtmlView(true);
                                }
                            }}
                            active={showHtmlView}
                            title="Toggle HTML source view"
                        >
                            {"<>"}
                        </ToolbarButton>
                    </div>
                </div>
            )}

            {showHtmlView ? (
                <div
                    className={clsx(
                        "rounded-md border bg-white shadow-inner transition-colors",
                        "border-gray-200 dark:border-gray-700 dark:bg-gray-900"
                    )}
                >
                    <textarea
                        value={htmlContent}
                        onChange={(e) => {
                            const newHtml = e.target.value;
                            setHtmlContent(newHtml);
                            // Directly update the parent component with raw HTML
                            if (onChange) {
                                onChange(newHtml || "");
                            }
                            // Optionally update editor for preview (but don't parse fully to preserve attributes)
                            if (editor) {
                                try {
                                    editor.commands.setContent(newHtml || "<p></p>", false);
                                } catch (e) {
                                    // If parsing fails, that's okay - we're preserving raw HTML
                                }
                            }
                        }}
                        className={clsx(
                            "w-full px-4 py-3 font-mono text-sm leading-relaxed",
                            "bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-100",
                            "border-0 resize-none",
                            "focus:outline-none focus:ring-0 focus:border-0",
                            "focus-visible:outline-none focus-visible:ring-0",
                            disabled && "pointer-events-none opacity-70"
                        )}
                        style={{
                            minHeight: `${height}px`,
                            maxHeight: `${Math.max(height, 500)}px`,
                            overflowY: "auto",
                            border: "none",
                            outline: "none",
                            whiteSpace: "pre-wrap",
                            lineHeight: "1.75",
                            tabSize: "2",
                        }}
                        onFocus={(e) => {
                            e.target.style.border = "none";
                            e.target.style.outline = "none";
                            e.target.style.boxShadow = "none";
                        }}
                        placeholder="Enter HTML content..."
                    />
                </div>
            ) : (
                <div
                    className={clsx(
                        "rounded-md border bg-white shadow-inner transition-colors",
                        "border-gray-200 dark:border-gray-700 dark:bg-gray-900"
                    )}
                >
                    <EditorContent
                        editor={editor}
                        className={clsx(
                            "tiptap prose prose-sm max-w-none px-4 py-3 focus:outline-none sm:prose",
                            "dark:prose-invert dark:text-gray-100",
                            disabled && "pointer-events-none opacity-70"
                        )}
                        style={{
                            minHeight: `${height}px`,
                            maxHeight: `${Math.max(height, 500)}px`,
                            overflowY: "auto",
                        }}
                    />
                </div>
            )}

            <input
                ref={fileInputRef}
                type="file"
                accept="image/*"
                className="hidden"
                onChange={handleImageFile}
            />
        </div>
    );
}
