import React, { useState, useEffect } from 'react';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
    faFileWord,
    faFileExcel,
    faFilePowerpoint,
    faFilePdf,
    faFileArchive,
    faDownload,
    faTrash,
    faEye,
    faUser,
    faPenNib,
    faShieldAlt,
    faClock,
    faCheckCircle,
    faExclamationTriangle
} from '@fortawesome/free-solid-svg-icons';
import axios from 'axios';

// File type configurations with icons
const ALLOWED_TYPES = {
    "application/msword": { ext: ".doc", icon: faFileWord },
    "application/vnd.openxmlformats-officedocument.wordprocessingml.document": { ext: ".docx", icon: faFileWord },
    "application/vnd.ms-excel": { ext: ".xls", icon: faFileExcel },
    "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet": { ext: ".xlsx", icon: faFileExcel },
    "application/vnd.ms-powerpoint": { ext: ".ppt", icon: faFilePowerpoint },
    "application/vnd.openxmlformats-officedocument.presentationml.presentation": { ext: ".pptx", icon: faFilePowerpoint },
    "application/pdf": { ext: ".pdf", icon: faFilePdf },
    "application/zip": { ext: ".zip", icon: faFileArchive },
    "application/x-zip-compressed": { ext: ".zip", icon: faFileArchive },
};

export default function AdminFileManagement({ orderId, files = [], onFileDeleted, isAdmin = true, orderStatus, onStatusChange }) {
    const [fileList, setFileList] = useState(files);
    const [isDeleting, setIsDeleting] = useState({});
    const [deleteError, setDeleteError] = useState("");
    const [activeFilter, setActiveFilter] = useState('all');
    const [isMarkingUnderReview, setIsMarkingUnderReview] = useState(false);

    useEffect(() => {
        setFileList(files);
    }, [files]);

    const formatFileSize = (bytes) => {
        if (bytes === 0) return "0 Bytes";
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
    };

    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const getFileIcon = (fileType) => {
        const fileConfig = ALLOWED_TYPES[fileType];
        return fileConfig ? fileConfig.icon : faFilePdf;
    };

    const getFileTypeLabel = (fileType) => {
        const fileConfig = ALLOWED_TYPES[fileType];
        return fileConfig ? fileConfig.ext.toUpperCase() : 'Unknown';
    };

    const getUploaderIcon = (userType) => {
        switch (userType) {
            case 'client':
                return faUser;
            case 'writer':
                return faPenNib;
            case 'admin':
                return faShieldAlt;
            default:
                return faUser;
        }
    };

    const getUploaderColor = (userType) => {
        switch (userType) {
            case 'client':
                return 'text-blue-600 dark:text-blue-400';
            case 'writer':
                return 'text-green-600 dark:text-green-400';
            case 'admin':
                return 'text-purple-600 dark:text-purple-400';
            default:
                return 'text-gray-600 dark:text-gray-400';
        }
    };

    const getCategoryIcon = (category) => {
        switch (category) {
            case 'draft':
                return faClock;
            case 'complete':
                return faCheckCircle;
            case 'admin':
                return faShieldAlt;
            case 'sample':
                return faEye;
            case 'instructions':
                return faExclamationTriangle;
            default:
                return faFilePdf;
        }
    };

    const getCategoryColor = (category) => {
        switch (category) {
            case 'draft':
                return 'text-yellow-600 dark:text-yellow-400';
            case 'complete':
                return 'text-green-600 dark:text-green-400';
            case 'admin':
                return 'text-purple-600 dark:text-purple-400';
            case 'sample':
                return 'text-blue-600 dark:text-blue-400';
            case 'instructions':
                return 'text-orange-600 dark:text-orange-400';
            default:
                return 'text-gray-600 dark:text-gray-400';
        }
    };

    const getCategoryLabel = (category) => {
        const labels = {
            'draft': 'Draft',
            'complete': 'Complete',
            'admin': 'Admin',
            'sample': 'Sample',
            'instructions': 'Instructions',
            'general': 'General'
        };
        return labels[category] || category;
    };

    const canDeleteFile = (file) => {
        // Admins can delete all files
        if (isAdmin) return true;
        
        // Users can only delete their own files
        // This would need to be passed from the parent component
        return false;
    };

    const handleDownload = async (file) => {
        try {
            const response = await axios.get(`/files/${file.id}/download`, {
                responseType: 'blob'
            });
            
            // Create a download link
            const url = window.URL.createObjectURL(new Blob([response.data]));
            const link = document.createElement('a');
            link.href = url;
            link.setAttribute('download', file.file_name);
            document.body.appendChild(link);
            link.click();
            link.remove();
            window.URL.revokeObjectURL(url);
        } catch (error) {
            console.error('Download failed:', error);
            alert('Failed to download file. Please try again.');
        }
    };

    const handleDelete = async (file) => {
        if (!confirm('Are you sure you want to delete this file?')) {
            return;
        }

        setIsDeleting(prev => ({ ...prev, [file.id]: true }));
        setDeleteError("");

        try {
            await axios.delete(`/files/${file.id}`);
            setFileList(prev => prev.filter(f => f.id !== file.id));
            onFileDeleted(file);
        } catch (error) {
            console.error('Delete failed:', error);
            setDeleteError(error.response?.data?.message || 'Failed to delete file');
        } finally {
            setIsDeleting(prev => ({ ...prev, [file.id]: false }));
        }
    };

    const handleFileAccess = async (file) => {
        // Check if this is a writer-submitted file and order is in submitted status
        if (file.uploader_user_type === 'writer' && orderStatus === 'submitted' && isAdmin) {
            try {
                setIsMarkingUnderReview(true);
                
                // Mark order as under review
                const response = await axios.post(`/admin/orders/${orderId}/mark-under-review`);
                
                if (response.data.success !== false) {
                    // Notify parent component about status change
                    if (onStatusChange) {
                        onStatusChange('under_review');
                    }
                    
                    // Show success message
                    alert('Order status changed to "Under Review" as you accessed writer-submitted files.');
                }
                
            } catch (error) {
                console.error('Failed to mark order as under review:', error);
                const errorMessage = error.response?.data?.message || 'Failed to update order status. Please try again.';
                alert(errorMessage);
            } finally {
                setIsMarkingUnderReview(false);
            }
        }
    };

    const getFilteredFiles = () => {
        if (activeFilter === 'all') return fileList;
        return fileList.filter(file => file.file_category === activeFilter);
    };

    const getFilterOptions = () => {
        const categories = [...new Set(fileList.map(file => file.file_category))];
        return [
            { value: 'all', label: 'All Files' },
            ...categories.map(cat => ({
                value: cat,
                label: getCategoryLabel(cat)
            }))
        ];
    };

    const groupedFiles = getFilteredFiles().reduce((groups, file) => {
        // Group files by uploader type: writer files go to writer group, others to client group
        const target = file.uploader_user_type === 'writer' ? 'writer' : 'client';
        if (!groups[target]) groups[target] = [];
        groups[target].push(file);
        return groups;
    }, {});

    if (fileList.length === 0) {
        return (
            <div className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-8 text-center">
                <FontAwesomeIcon icon={faFilePdf} className="w-12 h-12 text-gray-400 dark:text-gray-500 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">No Files Uploaded</h3>
                <p className="text-gray-500 dark:text-gray-400">
                    No files have been uploaded for this order yet.
                </p>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header and Filters */}
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center space-y-4 sm:space-y-0">
                <div>
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                        File Management
                    </h3>
                    <p className="text-sm text-gray-500 dark:text-gray-400">
                        {fileList.length} file(s) uploaded
                    </p>
                </div>

                {/* Filter Dropdown */}
                <div className="flex items-center space-x-3">
                    <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                        Filter by:
                    </label>
                    <select
                        value={activeFilter}
                        onChange={(e) => setActiveFilter(e.target.value)}
                        className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white text-sm"
                    >
                        {getFilterOptions().map(option => (
                            <option key={option.value} value={option.value}>
                                {option.label}
                            </option>
                        ))}
                    </select>
                </div>
            </div>

            {/* Error Message */}
            {deleteError && (
                <div className="p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                    <div className="flex items-center space-x-2 text-red-800 dark:text-red-200">
                        <FontAwesomeIcon icon={faExclamationTriangle} className="w-4 h-4" />
                        <span>{deleteError}</span>
                    </div>
                </div>
            )}

            {/* Files by Target Type */}
            {Object.entries(groupedFiles).map(([targetType, files]) => (
                <div key={targetType} className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700">
                    <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                        <div className="flex items-center space-x-2">
                            <FontAwesomeIcon 
                                icon={targetType === 'writer' ? faPenNib : faUser} 
                                className={`w-5 h-5 ${targetType === 'writer' ? 'text-green-600' : 'text-blue-600'}`} 
                            />
                            <h4 className="text-lg font-medium text-gray-900 dark:text-white">
                                Files for {targetType === 'writer' ? 'Writer' : 'Client'}
                            </h4>
                            <span className="px-2 py-1 bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-400 text-xs rounded-full">
                                {files.length}
                            </span>
                        </div>
                    </div>

                    <div className="divide-y divide-gray-200 dark:divide-gray-700">
                        {files.map((file) => (
                            <div key={file.id} className="px-6 py-4 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center space-x-4 flex-1">
                                        {/* File Icon and Info */}
                                        <div className="flex items-center space-x-3">
                                            <FontAwesomeIcon
                                                icon={getFileIcon(file.file_type)}
                                                className="w-8 h-8 text-gray-500 dark:text-gray-400"
                                            />
                                            <div>
                                                <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                    {file.file_name}
                                                </p>
                                                <div className="flex items-center space-x-4 text-xs text-gray-500 dark:text-gray-400">
                                                    <span>{getFileTypeLabel(file.file_type)}</span>
                                                    <span>{formatFileSize(file.file_size)}</span>
                                                    <span>{formatDate(file.date_uploaded)}</span>
                                                </div>
                                            </div>
                                        </div>

                                        {/* File Category */}
                                        <div className="flex items-center space-x-2">
                                            <FontAwesomeIcon
                                                icon={getCategoryIcon(file.file_category)}
                                                className={`w-4 h-4 ${getCategoryColor(file.file_category)}`}
                                            />
                                            <span className={`text-sm font-medium ${getCategoryColor(file.file_category)}`}>
                                                {getCategoryLabel(file.file_category)}
                                            </span>
                                        </div>

                                        {/* Uploader Info */}
                                        <div className="flex items-center space-x-2">
                                            <FontAwesomeIcon
                                                icon={getUploaderIcon(file.uploader_user_type)}
                                                className={`w-4 h-4 ${getUploaderColor(file.uploader_user_type)}`}
                                            />
                                            <span className={`text-sm ${getUploaderColor(file.uploader_user_type)}`}>
                                                {file.uploader_name}
                                            </span>
                                        </div>

                                        {/* Description */}
                                        {file.description && (
                                            <div className="text-sm text-gray-600 dark:text-gray-400 max-w-xs truncate">
                                                {file.description}
                                            </div>
                                        )}
                                    </div>

                                    {/* Actions */}
                                    <div className="flex items-center space-x-2">
                                        {/* Status Change Loading Indicator */}
                                        {isMarkingUnderReview && (
                                            <div className="flex items-center space-x-2 text-orange-600">
                                                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-orange-600"></div>
                                                <span className="text-xs">Updating...</span>
                                            </div>
                                        )}
                                        
                                        {/* View/Preview Button */}
                                        <button
                                            onClick={() => handleFileAccess(file)}
                                            disabled={isMarkingUnderReview}
                                            className="p-2 text-green-600 hover:text-green-800 dark:text-green-400 dark:hover:text-green-300 transition-colors disabled:opacity-50"
                                            title="View file"
                                        >
                                            <FontAwesomeIcon icon={faEye} className="w-4 h-4" />
                                        </button>
                                        
                                        {/* Download Button */}
                                        <button
                                            onClick={() => {
                                                handleFileAccess(file);
                                                handleDownload(file);
                                            }}
                                            disabled={isMarkingUnderReview}
                                            className="p-2 text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 transition-colors disabled:opacity-500"
                                            title="Download file"
                                        >
                                            <FontAwesomeIcon icon={faDownload} className="w-4 h-4" />
                                        </button>

                                        {canDeleteFile(file) && (
                                            <button
                                                onClick={() => handleDelete(file)}
                                                disabled={isDeleting[file.id]}
                                                className="p-2 text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300 transition-colors disabled:opacity-50"
                                                title="Delete file"
                                            >
                                                {isDeleting[file.id] ? (
                                                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-red-600"></div>
                                                ) : (
                                                    <FontAwesomeIcon icon={faTrash} className="w-4 h-4" />
                                                )}
                                            </button>
                                        )}
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            ))}
        </div>
    );
} 