import React from "react";
import { Link } from "@inertiajs/react";
import { Clock, DollarSign, Calendar, MessageSquare, User } from "lucide-react";
import { formatDistance } from "date-fns";

export default function BidCard({
    bid,
    onAccept,
    showActions = true,
    showOrderInfo = false,
    isClientView = true,
}) {
    // Format date for display
    const formatDate = (dateString) => {
        if (!dateString) return "N/A";

        const date = new Date(dateString);
        if (isNaN(date.getTime())) return "Invalid Date";

        return date.toLocaleDateString("en-US", {
            year: "numeric",
            month: "short",
            day: "numeric",
            hour: "2-digit",
            minute: "2-digit",
        });
    };

    // Calculate time remaining until completion
    const timeUntilCompletion = (completionTime) => {
        if (!completionTime) return "No deadline set";

        const now = new Date();
        const completion = new Date(completionTime);

        if (isNaN(completion.getTime())) return "Invalid deadline";

        if (now > completion) return "Deadline passed";

        return formatDistance(completion, now, { addSuffix: true });
    };

    // Format currency with the correct symbol
    const formatCurrency = (amount, currencyCode = "USD") => {
        // Get the appropriate currency symbol
        const currencySymbol = getCurrencySymbol(currencyCode);

        // Format the number with 2 decimal places
        const formattedAmount = parseFloat(amount).toFixed(2);

        // Return the formatted currency string
        return `${currencySymbol}${formattedAmount}`;
    };

    // Helper function to get currency symbol from currency code
    const getCurrencySymbol = (currencyCode) => {
        switch (currencyCode?.toUpperCase()) {
            case "EUR":
                return "€";
            case "GBP":
                return "£";
            case "CAD":
                return "C$";
            case "AUD":
                return "A$";
            default:
                return "$";
        }
    };

    // Function to get status badge class
    const getStatusBadgeClass = (status) => {
        switch (status) {
            case "accepted":
                return "bg-green-500 text-white";
            case "rejected":
                return "bg-red-500 text-white";
            case "pending":
                return "bg-yellow-500 text-yellow-100";
            default:
                return "bg-gray-500 text-white";
        }
    };

    // Get writer name/nickname (for privacy)
    const getWriterName = () => {
        if (!isClientView) {
            // If not client view, show real name
            return bid.writer?.name || "Unknown Writer";
        }

        // For client view, show nickname for privacy
        return bid.writer_nickname || `Writer #${bid.writer_id}`;
    };

    return (
        <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none  dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden hover:shadow-md transition-shadow duration-200">
            {/* Header with order info or status */}
            <div className="flex justify-between items-center p-4 border-b border-gray-100 dark:border-gray-700 bg-gray-50 dark:bg-gray-700/50">
                {showOrderInfo ? (
                    <div>
                        <Link
                            href={`/orders/${bid.order.id}`}
                            className="text-indigo-600 dark:text-indigo-400 font-medium hover:text-indigo-800 dark:hover:text-indigo-300"
                        >
                            Order #{bid.order.order_number}
                        </Link>
                        <p className="text-xs text-gray-500 dark:text-gray-400 truncate max-w-[200px]">
                            {bid.order.title}
                        </p>
                    </div>
                ) : (
                    <div className="flex items-center">
                        <User className="w-4 h-4 mr-2 text-gray-500 dark:text-gray-400" />
                        <span className="font-medium text-gray-900 dark:text-gray-100">
                            {getWriterName()}
                        </span>
                    </div>
                )}

                <span
                    className={`px-3 py-1 rounded-full text-xs font-medium ${getStatusBadgeClass(
                        bid.status
                    )}`}
                >
                    {bid.status.charAt(0).toUpperCase() + bid.status.slice(1)}
                </span>
            </div>

            {/* Bid Details */}
            <div className="p-4 grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="flex items-start">
                    <DollarSign className="w-5 h-5 mr-2 text-green-500 dark:text-green-400 mt-0.5" />
                    <div>
                        <p className="text-sm text-gray-500 dark:text-gray-400">
                            Bid Amount
                        </p>
                        <p className="font-medium text-green-600 dark:text-green-400">
                            {formatCurrency(bid.bid_amount)}
                        </p>
                    </div>
                </div>

                <div className="flex items-start">
                    <Clock className="w-5 h-5 mr-2 text-blue-500 dark:text-blue-400 mt-0.5" />
                    <div>
                        <p className="text-sm text-gray-500 dark:text-gray-400">
                            Completion Time
                        </p>
                        <p className="font-medium text-gray-900 dark:text-gray-100">
                            {formatDate(bid.completion_time)}
                        </p>
                        <p className="text-xs text-gray-500 dark:text-gray-400">
                            {timeUntilCompletion(bid.completion_time)}
                        </p>
                    </div>
                </div>

                {!showOrderInfo && (
                    <div className="flex items-start">
                        <User className="w-5 h-5 mr-2 text-purple-500 dark:text-purple-400 mt-0.5" />
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Writer
                            </p>
                            <p className="font-medium text-gray-900 dark:text-gray-100">
                                {getWriterName()}
                            </p>
                            {bid.writer_experience && (
                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                    {bid.writer_experience} years experience
                                </p>
                            )}
                        </div>
                    </div>
                )}

                <div className="flex items-start">
                    <Calendar className="w-5 h-5 mr-2 text-indigo-500 dark:text-indigo-400 mt-0.5" />
                    <div>
                        <p className="text-sm text-gray-500 dark:text-gray-400">
                            Submitted On
                        </p>
                        <p className="font-medium text-gray-900 dark:text-gray-100">
                            {formatDate(bid.created_at)}
                        </p>
                    </div>
                </div>
            </div>

            {/* Comments Section */}
            {bid.comments && (
                <div className="px-4 pb-4">
                    <div className="flex items-start">
                        <MessageSquare className="w-5 h-5 mr-2 text-gray-500 dark:text-gray-400 mt-0.5" />
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Comments
                            </p>
                            <p className="text-sm text-gray-700 dark:text-gray-300">
                                {bid.comments}
                            </p>
                        </div>
                    </div>
                </div>
            )}

            {/* Actions */}
            {showActions && bid.status === "pending" && onAccept && (
                <div className="px-4 py-3 bg-gray-50 dark:bg-gray-700 border-t border-gray-100 dark:border-gray-600 flex justify-end space-x-2">
                    {/* <button
                        onClick={() => onAccept(bid)}
                        className="px-4 py-2 bg-green-600 hover:bg-green-700 text-white rounded-md text-sm font-medium transition-colors duration-200"
                    >
                        Accept Bid
                    </button> */}
                </div>
            )}
        </div>
    );
}
