import React, { useState } from "react";
import { Gavel, PlusCircle } from "lucide-react";
import BidCard from "@/Components/Bids/BidCard";
import { Link } from "@inertiajs/react";

export default function BidComparisonSection({ bids, order, onAcceptBid }) {
    const [isLoading, setIsLoading] = useState(false);

    const handleAcceptBid = (bid) => {
        if (
            window.confirm(
                `Are you sure you want to accept this bid from ${
                    bid.writer_nickname || "this writer"
                }?`
            )
        ) {
            setIsLoading(true);
            onAcceptBid(bid);
        }
    };

    const pendingBids = bids.filter((bid) => bid.status === "pending");
    const acceptedBid = bids.find((bid) => bid.status === "accepted");

    if (!bids || bids.length === 0) {
        return (
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6 mb-6">
                <div className="text-center py-6">
                    <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-indigo-100 text-indigo-600 dark:bg-indigo-900/30 dark:text-indigo-400 mb-4">
                        <Gavel className="w-8 h-8" />
                    </div>
                    <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-2">
                        No bids yet
                    </h3>
                    <p className="text-gray-500 dark:text-gray-400 max-w-lg mx-auto mb-6">
                        Writers are reviewing your order. Bids will appear here
                        when writers propose to work on your project.
                    </p>
                    <Link
                        href={route("orders.index")}
                        className="inline-flex items-center px-4 py-2 bg-indigo-600 hover:bg-indigo-700 border border-transparent rounded-md font-medium text-sm text-white"
                    >
                        <PlusCircle className="w-4 h-4 mr-2" />
                        View All Orders
                    </Link>
                </div>
            </div>
        );
    }

    return (
        <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6 mb-6">
            <div className="flex justify-between items-center mb-4">
                <h2 className="text-xl font-semibold text-gray-800 dark:text-gray-200 flex items-center">
                    <Gavel className="w-6 h-6 mr-2" />
                    Bids ({bids.length})
                </h2>
                <Link
                    href={route("client.bids.index")}
                    className="text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-300 text-sm font-medium"
                >
                    View All Bids
                </Link>
            </div>

            {/* If there's an accepted bid, show it at the top */}
            {acceptedBid && (
                <div className="mb-6">
                    <h3 className="text-lg font-medium text-gray-800 dark:text-gray-200 mb-3">
                        Accepted Bid
                    </h3>
                    <BidCard
                        bid={acceptedBid}
                        showActions={false}
                        isClientView={true}
                    />
                </div>
            )}

            {/* Show pending bids */}
            {pendingBids.length > 0 && (
                <div>
                    <h3 className="text-lg font-medium text-gray-800 dark:text-gray-200 mb-3">
                        {acceptedBid ? "Other Bids" : "Available Bids"}
                    </h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        {pendingBids.map((bid) => (
                            <BidCard
                                key={bid.id}
                                bid={bid}
                                onAccept={handleAcceptBid}
                                showActions={
                                    !acceptedBid &&
                                    order.order_status === "bidding"
                                }
                                isClientView={true}
                            />
                        ))}
                    </div>
                </div>
            )}
        </div>
    );
}
