import React, { useState, useEffect, useCallback } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Eye,
    Code,
    FileText,
    AlertCircle,
    CheckCircle,
    X,
    ChevronDown,
    ChevronUp,
    Calendar,
    User,
    Tag,
    Clock,
    Hash,
    BookOpen,
    Copy,
    Download,
    ExternalLink,
} from "lucide-react";

const BlogFilePreview = ({ file, onClose, className = "" }) => {
    // State management
    const [content, setContent] = useState("");
    const [parsedData, setParsedData] = useState(null);
    const [validationErrors, setValidationErrors] = useState([]);
    const [validationWarnings, setValidationWarnings] = useState([]);
    const [loading, setLoading] = useState(true);
    const [activeTab, setActiveTab] = useState("preview"); // preview, code, validation
    const [expandedSections, setExpandedSections] = useState({
        metadata: true,
        content: true,
        seo: false,
        tags: false,
    });

    // Load file content
    useEffect(() => {
        if (file) {
            loadFileContent();
        }
    }, [file]);

    // Load and parse file content
    const loadFileContent = useCallback(async () => {
        setLoading(true);
        setValidationErrors([]);
        setValidationWarnings([]);

        try {
            const reader = new FileReader();
            reader.onload = (e) => {
                const fileContent = e.target.result;
                setContent(fileContent);

                try {
                    // Parse the JavaScript file content
                    const parsed = parseJavaScriptFile(fileContent);
                    setParsedData(parsed);
                    validateBlogPost(parsed);
                } catch (parseError) {
                    setValidationErrors([`Parse error: ${parseError.message}`]);
                }

                setLoading(false);
            };

            reader.onerror = () => {
                setValidationErrors(["Failed to read file"]);
                setLoading(false);
            };

            reader.readAsText(file);
        } catch (error) {
            setValidationErrors([`Error loading file: ${error.message}`]);
            setLoading(false);
        }
    }, [file]);

    // Parse JavaScript file (simplified parser)
    const parseJavaScriptFile = (content) => {
        // Remove comments and extract the export default object
        const cleanContent = content
            .replace(/\/\*[\s\S]*?\*\//g, "")
            .replace(/\/\/.*$/gm, "");

        // Look for export default
        const exportMatch = cleanContent.match(
            /export\s+default\s+({[\s\S]*})/
        );
        if (!exportMatch) {
            throw new Error("No export default found");
        }

        // Basic extraction of object properties (simplified)
        const objectContent = exportMatch[1];

        // This is a simplified parser - in production, you'd use a proper AST parser
        const data = {
            title: extractStringValue(objectContent, "title"),
            slug: extractStringValue(objectContent, "slug"),
            excerpt: extractStringValue(objectContent, "excerpt"),
            content: extractStringValue(objectContent, "content"),
            author: extractStringValue(objectContent, "author"),
            category: extractStringValue(objectContent, "category"),
            publishedAt: extractStringValue(objectContent, "publishedAt"),
            readTime: extractNumberValue(objectContent, "readTime"),
            featured: extractBooleanValue(objectContent, "featured"),
            tags: extractArrayValue(objectContent, "tags"),
            keywords: extractArrayValue(objectContent, "keywords"),
            meta: extractObjectValue(objectContent, "meta"),
            seo: extractObjectValue(objectContent, "seo"),
        };

        return data;
    };

    // Helper functions for parsing
    const extractStringValue = (content, key) => {
        const regex = new RegExp(`${key}:\\s*['"\`]([^'"\`]*?)['"\`]`, "i");
        const match = content.match(regex);
        return match ? match[1] : "";
    };

    const extractNumberValue = (content, key) => {
        const regex = new RegExp(`${key}:\\s*([0-9]+)`, "i");
        const match = content.match(regex);
        return match ? parseInt(match[1]) : 0;
    };

    const extractBooleanValue = (content, key) => {
        const regex = new RegExp(`${key}:\\s*(true|false)`, "i");
        const match = content.match(regex);
        return match ? match[1] === "true" : false;
    };

    const extractArrayValue = (content, key) => {
        const regex = new RegExp(`${key}:\\s*\\[([^\\]]*?)\\]`, "i");
        const match = content.match(regex);
        if (!match) return [];

        const arrayContent = match[1];
        return arrayContent
            .split(",")
            .map((item) => item.trim().replace(/['"`]/g, ""))
            .filter((item) => item);
    };

    const extractObjectValue = (content, key) => {
        const regex = new RegExp(`${key}:\\s*{([^}]*?)}`, "i");
        const match = content.match(regex);
        if (!match) return {};

        // Simplified object parsing
        const obj = {};
        const pairs = match[1].split(",");
        pairs.forEach((pair) => {
            const [k, v] = pair.split(":").map((s) => s.trim());
            if (k && v) {
                obj[k.replace(/['"`]/g, "")] = v.replace(/['"`]/g, "");
            }
        });
        return obj;
    };

    // Validate blog post data
    const validateBlogPost = (data) => {
        const errors = [];
        const warnings = [];

        // Required fields
        if (!data.title) errors.push("Title is required");
        if (!data.slug) errors.push("Slug is required");
        if (!data.excerpt) errors.push("Excerpt is required");
        if (!data.content) errors.push("Content is required");
        if (!data.author) errors.push("Author is required");
        if (!data.category) errors.push("Category is required");
        if (!data.publishedAt) errors.push("Published date is required");

        // Format validation
        if (data.title && data.title.length < 10) {
            warnings.push("Title is quite short (less than 10 characters)");
        }
        if (data.title && data.title.length > 100) {
            warnings.push("Title is quite long (more than 100 characters)");
        }
        if (data.excerpt && data.excerpt.length < 50) {
            warnings.push("Excerpt is quite short (less than 50 characters)");
        }
        if (data.excerpt && data.excerpt.length > 300) {
            warnings.push("Excerpt is quite long (more than 300 characters)");
        }
        if (data.slug && !/^[a-z0-9-]+$/.test(data.slug)) {
            errors.push(
                "Slug must contain only lowercase letters, numbers, and hyphens"
            );
        }

        // Date validation
        if (data.publishedAt && isNaN(Date.parse(data.publishedAt))) {
            errors.push("Invalid published date format");
        }

        // SEO validation
        if (data.meta) {
            if (!data.meta.title) warnings.push("SEO title is recommended");
            if (!data.meta.description)
                warnings.push("SEO description is recommended");
            if (!data.meta.keywords)
                warnings.push("SEO keywords are recommended");
        }

        setValidationErrors(errors);
        setValidationWarnings(warnings);
    };

    // Toggle section expansion
    const toggleSection = (section) => {
        setExpandedSections((prev) => ({
            ...prev,
            [section]: !prev[section],
        }));
    };

    // Copy content to clipboard
    const copyToClipboard = (text) => {
        navigator.clipboard.writeText(text);
    };

    // Format date
    const formatDate = (dateString) => {
        if (!dateString) return "Not set";
        const date = new Date(dateString);
        return isNaN(date.getTime())
            ? "Invalid date"
            : date.toLocaleDateString();
    };

    // Get validation status
    const getValidationStatus = () => {
        if (validationErrors.length > 0) return "error";
        if (validationWarnings.length > 0) return "warning";
        return "success";
    };

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.3 } },
    };

    if (!file) return null;

    return (
        <div className={`blog-file-preview ${className}`}>
            <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                <motion.div
                    initial={{ opacity: 0, scale: 0.95 }}
                    animate={{ opacity: 1, scale: 1 }}
                    exit={{ opacity: 0, scale: 0.95 }}
                    className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-4xl w-full max-h-[90vh] overflow-hidden"
                >
                    {/* Header */}
                    <div className="flex items-center justify-between p-6 border-b border-gray-200 dark:border-gray-700">
                        <div className="flex items-center space-x-3">
                            <div className="w-10 h-10 bg-blue-100 dark:bg-blue-900/30 rounded-lg flex items-center justify-center">
                                <FileText className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                            </div>
                            <div>
                                <h2 className="text-lg font-semibold text-gray-900 dark:text-white">
                                    {file.name}
                                </h2>
                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                    {(file.size / 1024).toFixed(1)} KB •
                                    JavaScript Blog Post
                                </p>
                            </div>
                        </div>
                        <button
                            onClick={onClose}
                            className="p-2 text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 rounded-lg transition-colors"
                        >
                            <X className="w-5 h-5" />
                        </button>
                    </div>

                    {/* Tab Navigation */}
                    <div className="flex border-b border-gray-200 dark:border-gray-700">
                        <button
                            onClick={() => setActiveTab("preview")}
                            className={`px-6 py-3 text-sm font-medium transition-colors ${
                                activeTab === "preview"
                                    ? "text-blue-600 dark:text-blue-400 border-b-2 border-blue-600 dark:border-blue-400"
                                    : "text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300"
                            }`}
                        >
                            <Eye className="w-4 h-4 mr-2 inline" />
                            Preview
                        </button>
                        <button
                            onClick={() => setActiveTab("code")}
                            className={`px-6 py-3 text-sm font-medium transition-colors ${
                                activeTab === "code"
                                    ? "text-blue-600 dark:text-blue-400 border-b-2 border-blue-600 dark:border-blue-400"
                                    : "text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300"
                            }`}
                        >
                            <Code className="w-4 h-4 mr-2 inline" />
                            Code
                        </button>
                        <button
                            onClick={() => setActiveTab("validation")}
                            className={`px-6 py-3 text-sm font-medium transition-colors ${
                                activeTab === "validation"
                                    ? "text-blue-600 dark:text-blue-400 border-b-2 border-blue-600 dark:border-blue-400"
                                    : "text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300"
                            }`}
                        >
                            {getValidationStatus() === "error" ? (
                                <AlertCircle className="w-4 h-4 mr-2 inline text-red-500" />
                            ) : getValidationStatus() === "warning" ? (
                                <AlertCircle className="w-4 h-4 mr-2 inline text-yellow-500" />
                            ) : (
                                <CheckCircle className="w-4 h-4 mr-2 inline text-green-500" />
                            )}
                            Validation
                        </button>
                    </div>

                    {/* Content */}
                    <div className="flex-1 overflow-y-auto max-h-[calc(90vh-200px)]">
                        {loading ? (
                            <div className="flex items-center justify-center h-64">
                                <div className="text-center">
                                    <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 dark:border-blue-400 mx-auto mb-4"></div>
                                    <p className="text-gray-500 dark:text-gray-400">
                                        Loading file content...
                                    </p>
                                </div>
                            </div>
                        ) : (
                            <div className="p-6">
                                {/* Preview Tab */}
                                {activeTab === "preview" && parsedData && (
                                    <motion.div
                                        initial="hidden"
                                        animate="visible"
                                        variants={fadeIn}
                                        className="space-y-6"
                                    >
                                        {/* Metadata Section */}
                                        <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-4">
                                            <button
                                                onClick={() =>
                                                    toggleSection("metadata")
                                                }
                                                className="flex items-center justify-between w-full text-left"
                                            >
                                                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                                    Post Metadata
                                                </h3>
                                                {expandedSections.metadata ? (
                                                    <ChevronUp className="w-5 h-5 text-gray-500" />
                                                ) : (
                                                    <ChevronDown className="w-5 h-5 text-gray-500" />
                                                )}
                                            </button>

                                            {expandedSections.metadata && (
                                                <div className="mt-4 grid grid-cols-1 md:grid-cols-2 gap-4">
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                            Title
                                                        </label>
                                                        <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-2 rounded border">
                                                            {parsedData.title ||
                                                                "Not set"}
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                            Slug
                                                        </label>
                                                        <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-2 rounded border">
                                                            {parsedData.slug ||
                                                                "Not set"}
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                            Author
                                                        </label>
                                                        <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-2 rounded border flex items-center">
                                                            <User className="w-4 h-4 mr-2 text-gray-400" />
                                                            {parsedData.author ||
                                                                "Not set"}
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                            Category
                                                        </label>
                                                        <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-2 rounded border flex items-center">
                                                            <BookOpen className="w-4 h-4 mr-2 text-gray-400" />
                                                            {parsedData.category ||
                                                                "Not set"}
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                            Published Date
                                                        </label>
                                                        <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-2 rounded border flex items-center">
                                                            <Calendar className="w-4 h-4 mr-2 text-gray-400" />
                                                            {formatDate(
                                                                parsedData.publishedAt
                                                            )}
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                            Read Time
                                                        </label>
                                                        <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-2 rounded border flex items-center">
                                                            <Clock className="w-4 h-4 mr-2 text-gray-400" />
                                                            {parsedData.readTime ||
                                                                0}{" "}
                                                            minutes
                                                        </div>
                                                    </div>
                                                </div>
                                            )}
                                        </div>

                                        {/* Content Section */}
                                        <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-4">
                                            <button
                                                onClick={() =>
                                                    toggleSection("content")
                                                }
                                                className="flex items-center justify-between w-full text-left"
                                            >
                                                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                                    Content Preview
                                                </h3>
                                                {expandedSections.content ? (
                                                    <ChevronUp className="w-5 h-5 text-gray-500" />
                                                ) : (
                                                    <ChevronDown className="w-5 h-5 text-gray-500" />
                                                )}
                                            </button>

                                            {expandedSections.content && (
                                                <div className="mt-4 space-y-4">
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                            Excerpt
                                                        </label>
                                                        <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-3 rounded border">
                                                            {parsedData.excerpt ||
                                                                "Not set"}
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                            Content (First 500
                                                            characters)
                                                        </label>
                                                        <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-3 rounded border max-h-32 overflow-y-auto">
                                                            {parsedData.content
                                                                ? parsedData.content.substring(
                                                                      0,
                                                                      500
                                                                  ) +
                                                                  (parsedData
                                                                      .content
                                                                      .length >
                                                                  500
                                                                      ? "..."
                                                                      : "")
                                                                : "Not set"}
                                                        </div>
                                                    </div>
                                                </div>
                                            )}
                                        </div>

                                        {/* Tags Section */}
                                        <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-4">
                                            <button
                                                onClick={() =>
                                                    toggleSection("tags")
                                                }
                                                className="flex items-center justify-between w-full text-left"
                                            >
                                                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                                    Tags & Keywords
                                                </h3>
                                                {expandedSections.tags ? (
                                                    <ChevronUp className="w-5 h-5 text-gray-500" />
                                                ) : (
                                                    <ChevronDown className="w-5 h-5 text-gray-500" />
                                                )}
                                            </button>

                                            {expandedSections.tags && (
                                                <div className="mt-4 space-y-4">
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                            Tags
                                                        </label>
                                                        <div className="flex flex-wrap gap-2">
                                                            {parsedData.tags &&
                                                            parsedData.tags
                                                                .length > 0 ? (
                                                                parsedData.tags.map(
                                                                    (
                                                                        tag,
                                                                        index
                                                                    ) => (
                                                                        <span
                                                                            key={
                                                                                index
                                                                            }
                                                                            className="inline-flex items-center px-2 py-1 rounded-full text-xs bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-200"
                                                                        >
                                                                            <Tag className="w-3 h-3 mr-1" />
                                                                            {
                                                                                tag
                                                                            }
                                                                        </span>
                                                                    )
                                                                )
                                                            ) : (
                                                                <span className="text-gray-500 dark:text-gray-400">
                                                                    No tags
                                                                </span>
                                                            )}
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                            Keywords
                                                        </label>
                                                        <div className="flex flex-wrap gap-2">
                                                            {parsedData.keywords &&
                                                            parsedData.keywords
                                                                .length > 0 ? (
                                                                parsedData.keywords.map(
                                                                    (
                                                                        keyword,
                                                                        index
                                                                    ) => (
                                                                        <span
                                                                            key={
                                                                                index
                                                                            }
                                                                            className="inline-flex items-center px-2 py-1 rounded-full text-xs bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-200"
                                                                        >
                                                                            <Hash className="w-3 h-3 mr-1" />
                                                                            {
                                                                                keyword
                                                                            }
                                                                        </span>
                                                                    )
                                                                )
                                                            ) : (
                                                                <span className="text-gray-500 dark:text-gray-400">
                                                                    No keywords
                                                                </span>
                                                            )}
                                                        </div>
                                                    </div>
                                                </div>
                                            )}
                                        </div>

                                        {/* SEO Section */}
                                        <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-4">
                                            <button
                                                onClick={() =>
                                                    toggleSection("seo")
                                                }
                                                className="flex items-center justify-between w-full text-left"
                                            >
                                                <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                                    SEO Metadata
                                                </h3>
                                                {expandedSections.seo ? (
                                                    <ChevronUp className="w-5 h-5 text-gray-500" />
                                                ) : (
                                                    <ChevronDown className="w-5 h-5 text-gray-500" />
                                                )}
                                            </button>

                                            {expandedSections.seo && (
                                                <div className="mt-4 space-y-4">
                                                    {parsedData.meta &&
                                                    Object.keys(parsedData.meta)
                                                        .length > 0 ? (
                                                        Object.entries(
                                                            parsedData.meta
                                                        ).map(
                                                            ([key, value]) => (
                                                                <div key={key}>
                                                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                                        {key
                                                                            .charAt(
                                                                                0
                                                                            )
                                                                            .toUpperCase() +
                                                                            key.slice(
                                                                                1
                                                                            )}
                                                                    </label>
                                                                    <div className="text-sm text-gray-900 dark:text-white bg-white dark:bg-gray-700 p-2 rounded border">
                                                                        {value ||
                                                                            "Not set"}
                                                                    </div>
                                                                </div>
                                                            )
                                                        )
                                                    ) : (
                                                        <p className="text-gray-500 dark:text-gray-400">
                                                            No SEO metadata
                                                            found
                                                        </p>
                                                    )}
                                                </div>
                                            )}
                                        </div>
                                    </motion.div>
                                )}

                                {/* Code Tab */}
                                {activeTab === "code" && (
                                    <motion.div
                                        initial="hidden"
                                        animate="visible"
                                        variants={fadeIn}
                                    >
                                        <div className="bg-gray-900 dark:bg-gray-950 rounded-lg p-4 overflow-x-auto">
                                            <div className="flex items-center justify-between mb-4">
                                                <span className="text-sm text-gray-400">
                                                    JavaScript
                                                </span>
                                                <button
                                                    onClick={() =>
                                                        copyToClipboard(content)
                                                    }
                                                    className="flex items-center text-sm text-gray-400 hover:text-white transition-colors"
                                                >
                                                    <Copy className="w-4 h-4 mr-1" />
                                                    Copy
                                                </button>
                                            </div>
                                            <pre className="text-sm text-gray-300 whitespace-pre-wrap">
                                                <code>{content}</code>
                                            </pre>
                                        </div>
                                    </motion.div>
                                )}

                                {/* Validation Tab */}
                                {activeTab === "validation" && (
                                    <motion.div
                                        initial="hidden"
                                        animate="visible"
                                        variants={fadeIn}
                                        className="space-y-6"
                                    >
                                        {/* Validation Status */}
                                        <div
                                            className={`p-4 rounded-lg border ${
                                                getValidationStatus() ===
                                                "error"
                                                    ? "bg-red-50 dark:bg-red-900/20 border-red-200 dark:border-red-800"
                                                    : getValidationStatus() ===
                                                      "warning"
                                                    ? "bg-yellow-50 dark:bg-yellow-900/20 border-yellow-200 dark:border-yellow-800"
                                                    : "bg-green-50 dark:bg-green-900/20 border-green-200 dark:border-green-800"
                                            }`}
                                        >
                                            <div className="flex items-center">
                                                {getValidationStatus() ===
                                                "error" ? (
                                                    <AlertCircle className="w-5 h-5 text-red-500 mr-3" />
                                                ) : getValidationStatus() ===
                                                  "warning" ? (
                                                    <AlertCircle className="w-5 h-5 text-yellow-500 mr-3" />
                                                ) : (
                                                    <CheckCircle className="w-5 h-5 text-green-500 mr-3" />
                                                )}
                                                <div>
                                                    <h3
                                                        className={`font-medium ${
                                                            getValidationStatus() ===
                                                            "error"
                                                                ? "text-red-900 dark:text-red-100"
                                                                : getValidationStatus() ===
                                                                  "warning"
                                                                ? "text-yellow-900 dark:text-yellow-100"
                                                                : "text-green-900 dark:text-green-100"
                                                        }`}
                                                    >
                                                        {getValidationStatus() ===
                                                        "error"
                                                            ? "Validation Failed"
                                                            : getValidationStatus() ===
                                                              "warning"
                                                            ? "Validation Passed with Warnings"
                                                            : "Validation Passed"}
                                                    </h3>
                                                    <p
                                                        className={`text-sm ${
                                                            getValidationStatus() ===
                                                            "error"
                                                                ? "text-red-700 dark:text-red-300"
                                                                : getValidationStatus() ===
                                                                  "warning"
                                                                ? "text-yellow-700 dark:text-yellow-300"
                                                                : "text-green-700 dark:text-green-300"
                                                        }`}
                                                    >
                                                        {getValidationStatus() ===
                                                        "error"
                                                            ? "This file has validation errors that must be fixed before upload."
                                                            : getValidationStatus() ===
                                                              "warning"
                                                            ? "This file has warnings but can still be uploaded."
                                                            : "This file meets all validation requirements."}
                                                    </p>
                                                </div>
                                            </div>
                                        </div>

                                        {/* Validation Errors */}
                                        {validationErrors.length > 0 && (
                                            <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-4">
                                                <h4 className="font-medium text-red-900 dark:text-red-100 mb-2">
                                                    Errors (
                                                    {validationErrors.length})
                                                </h4>
                                                <ul className="space-y-1">
                                                    {validationErrors.map(
                                                        (error, index) => (
                                                            <li
                                                                key={index}
                                                                className="flex items-start text-sm text-red-700 dark:text-red-300"
                                                            >
                                                                <AlertCircle className="w-4 h-4 mr-2 mt-0.5 flex-shrink-0" />
                                                                {error}
                                                            </li>
                                                        )
                                                    )}
                                                </ul>
                                            </div>
                                        )}

                                        {/* Validation Warnings */}
                                        {validationWarnings.length > 0 && (
                                            <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
                                                <h4 className="font-medium text-yellow-900 dark:text-yellow-100 mb-2">
                                                    Warnings (
                                                    {validationWarnings.length})
                                                </h4>
                                                <ul className="space-y-1">
                                                    {validationWarnings.map(
                                                        (warning, index) => (
                                                            <li
                                                                key={index}
                                                                className="flex items-start text-sm text-yellow-700 dark:text-yellow-300"
                                                            >
                                                                <AlertCircle className="w-4 h-4 mr-2 mt-0.5 flex-shrink-0" />
                                                                {warning}
                                                            </li>
                                                        )
                                                    )}
                                                </ul>
                                            </div>
                                        )}

                                        {/* Validation Rules */}
                                        <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-4">
                                            <h4 className="font-medium text-gray-900 dark:text-white mb-3">
                                                Validation Rules
                                            </h4>
                                            <div className="space-y-2 text-sm">
                                                <div className="flex items-center">
                                                    <CheckCircle className="w-4 h-4 text-green-500 mr-2" />
                                                    <span className="text-gray-700 dark:text-gray-300">
                                                        Must export default
                                                        object
                                                    </span>
                                                </div>
                                                <div className="flex items-center">
                                                    <CheckCircle className="w-4 h-4 text-green-500 mr-2" />
                                                    <span className="text-gray-700 dark:text-gray-300">
                                                        Required fields: title,
                                                        slug, excerpt, content,
                                                        author, category,
                                                        publishedAt
                                                    </span>
                                                </div>
                                                <div className="flex items-center">
                                                    <CheckCircle className="w-4 h-4 text-green-500 mr-2" />
                                                    <span className="text-gray-700 dark:text-gray-300">
                                                        Slug must be lowercase
                                                        with hyphens only
                                                    </span>
                                                </div>
                                                <div className="flex items-center">
                                                    <CheckCircle className="w-4 h-4 text-green-500 mr-2" />
                                                    <span className="text-gray-700 dark:text-gray-300">
                                                        Published date must be
                                                        valid ISO format
                                                    </span>
                                                </div>
                                                <div className="flex items-center">
                                                    <AlertCircle className="w-4 h-4 text-yellow-500 mr-2" />
                                                    <span className="text-gray-700 dark:text-gray-300">
                                                        SEO metadata (title,
                                                        description, keywords)
                                                        recommended
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    </motion.div>
                                )}
                            </div>
                        )}
                    </div>

                    {/* Footer */}
                    <div className="flex items-center justify-between p-6 border-t border-gray-200 dark:border-gray-700">
                        <div className="flex items-center space-x-4">
                            <div
                                className={`flex items-center text-sm ${
                                    getValidationStatus() === "error"
                                        ? "text-red-600 dark:text-red-400"
                                        : getValidationStatus() === "warning"
                                        ? "text-yellow-600 dark:text-yellow-400"
                                        : "text-green-600 dark:text-green-400"
                                }`}
                            >
                                {getValidationStatus() === "error" ? (
                                    <AlertCircle className="w-4 h-4 mr-1" />
                                ) : getValidationStatus() === "warning" ? (
                                    <AlertCircle className="w-4 h-4 mr-1" />
                                ) : (
                                    <CheckCircle className="w-4 h-4 mr-1" />
                                )}
                                {validationErrors.length > 0
                                    ? `${validationErrors.length} errors`
                                    : validationWarnings.length > 0
                                    ? `${validationWarnings.length} warnings`
                                    : "Valid"}
                            </div>

                            {parsedData && (
                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                    {parsedData.title
                                        ? `"${parsedData.title.substring(
                                              0,
                                              30
                                          )}${
                                              parsedData.title.length > 30
                                                  ? "..."
                                                  : ""
                                          }"`
                                        : "Untitled"}
                                </div>
                            )}
                        </div>

                        <div className="flex space-x-3">
                            <button
                                onClick={() => copyToClipboard(content)}
                                className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                            >
                                <Copy className="w-4 h-4 mr-2 inline" />
                                Copy Code
                            </button>

                            <button
                                onClick={() => {
                                    const blob = new Blob([content], {
                                        type: "text/javascript",
                                    });
                                    const url = URL.createObjectURL(blob);
                                    const a = document.createElement("a");
                                    a.href = url;
                                    a.download = file.name;
                                    a.click();
                                    URL.revokeObjectURL(url);
                                }}
                                className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-600 transition-colors"
                            >
                                <Download className="w-4 h-4 mr-2 inline" />
                                Download
                            </button>

                            <button
                                onClick={onClose}
                                className="px-4 py-2 text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 rounded-lg transition-colors"
                            >
                                Close
                            </button>
                        </div>
                    </div>
                </motion.div>
            </div>
        </div>
    );
};

export default BlogFilePreview;
