import React from "react";
import { motion } from "framer-motion";
import { Calendar, Clock, ArrowRight, User } from "lucide-react";
import OptimizedImage from "../OptimizedImage";

/**
 * PostCardGrid Component - Grid layout for blog posts
 */
export const PostCardGrid = ({ post, categories = {}, className = "" }) => {
    // Format date helper
    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString("en-US", {
            year: "numeric",
            month: "long",
            day: "numeric",
        });
    };

    // Get category info
    const getCategoryInfo = (categorySlug) => {
        return (
            categories[categorySlug] || {
                name: categorySlug,
                color: "blue",
            }
        );
    };

    const categoryInfo = getCategoryInfo(post.category);

    return (
        <article
            className={`bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 rounded-lg shadow-md overflow-hidden hover:shadow-lg transition-shadow ${className}`}
        >
            {post.featuredImage && (
                <OptimizedImage
                    src={post.featuredImage}
                    alt={post.featuredImageAlt || post.title}
                    className="w-full h-48 object-cover"
                    loading="lazy"
                    sizes="(max-width: 768px) 100vw, (max-width: 1024px) 50vw, 400px"
                />
            )}
            <div className="p-6">
                <div className="flex flex-col sm:flex-row sm:items-center mb-3 gap-2">
                    <span
                        className={`inline-block px-3 py-1 text-xs font-semibold text-white bg-${categoryInfo.color}-600 rounded-full`}
                    >
                        {categoryInfo.name}
                    </span>
                    <span className="text-sm text-gray-600 dark:text-gray-400 flex items-center">
                        <Calendar className="h-4 w-4 mr-1" />
                        {formatDate(post.publishedAt)}
                    </span>
                </div>

                <h3 className="text-lg sm:text-xl font-bold text-gray-900 dark:text-white mb-3">
                    <a
                        href={`/blog/${post.slug}`}
                        className="hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                    >
                        {post.title}
                    </a>
                </h3>

                <p className="text-gray-700 dark:text-gray-300 mb-4 text-sm sm:text-base">
                    {post.excerpt}
                </p>

                <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4">
                        <span className="text-sm text-gray-600 dark:text-gray-400 flex items-center">
                            <Clock className="h-4 w-4 mr-1" />
                            {post.readTime} min read
                        </span>

                        {post.author && (
                            <span className="text-sm text-gray-600 dark:text-gray-400 flex items-center">
                                <User className="h-4 w-4 mr-1" />
                                {typeof post.author === "string"
                                    ? post.author
                                    : post.author.name}
                            </span>
                        )}
                    </div>

                    <a
                        href={`/blog/${post.slug}`}
                        className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 font-medium inline-flex items-center text-sm"
                    >
                        Read More <ArrowRight className="ml-1 h-4 w-4" />
                    </a>
                </div>
            </div>
        </article>
    );
};

/**
 * PostCardList Component - List layout for blog posts
 */
export const PostCardList = ({ post, categories = {}, className = "" }) => {
    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString("en-US", {
            year: "numeric",
            month: "long",
            day: "numeric",
        });
    };

    const getCategoryInfo = (categorySlug) => {
        return (
            categories[categorySlug] || {
                name: categorySlug,
                color: "blue",
            }
        );
    };

    const categoryInfo = getCategoryInfo(post.category);

    return (
        <article
            className={`bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 rounded-lg shadow-md overflow-hidden hover:shadow-lg transition-shadow ${className}`}
        >
            <div className="sm:flex">
                {post.featuredImage && (
                    <div className="sm:w-1/3 lg:w-1/4">
                        <OptimizedImage
                            src={post.featuredImage}
                            alt={post.featuredImageAlt || post.title}
                            className="w-full h-48 sm:h-full object-cover"
                            loading="lazy"
                            sizes="(max-width: 768px) 100vw, 300px"
                        />
                    </div>
                )}
                <div
                    className={`p-6 ${
                        post.featuredImage ? "sm:w-2/3 lg:w-3/4" : "w-full"
                    }`}
                >
                    <div className="flex flex-col sm:flex-row sm:items-center justify-between mb-3 gap-2">
                        <div className="flex items-center gap-3">
                            <span
                                className={`inline-block px-3 py-1 text-xs font-semibold text-white bg-${categoryInfo.color}-600 rounded-full`}
                            >
                                {categoryInfo.name}
                            </span>
                            <span className="text-sm text-gray-600 dark:text-gray-400 flex items-center">
                                <Calendar className="h-4 w-4 mr-1" />
                                {formatDate(post.publishedAt)}
                            </span>
                        </div>
                        <span className="text-sm text-gray-600 dark:text-gray-400 flex items-center">
                            <Clock className="h-4 w-4 mr-1" />
                            {post.readTime} min read
                        </span>
                    </div>

                    <h3 className="text-xl sm:text-2xl font-bold text-gray-900 dark:text-white mb-3">
                        <a
                            href={`/blog/${post.slug}`}
                            className="hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                        >
                            {post.title}
                        </a>
                    </h3>

                    <p className="text-gray-700 dark:text-gray-300 mb-4">
                        {post.excerpt}
                    </p>

                    <div className="flex items-center justify-between">
                        <div className="flex flex-wrap gap-2">
                            {post.keywords &&
                                post.keywords
                                    .slice(0, 3)
                                    .map((keyword, index) => (
                                        <span
                                            key={index}
                                            className="px-2 py-1 text-xs bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-full"
                                        >
                                            {keyword}
                                        </span>
                                    ))}
                        </div>
                        <a
                            href={`/blog/${post.slug}`}
                            className="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 font-medium inline-flex items-center"
                        >
                            Read More <ArrowRight className="ml-1 h-4 w-4" />
                        </a>
                    </div>
                </div>
            </div>
        </article>
    );
};

/**
 * PostCardFeatured Component - Large featured post layout
 */
export const PostCardFeatured = ({ post, categories = {}, className = "" }) => {
    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString("en-US", {
            year: "numeric",
            month: "long",
            day: "numeric",
        });
    };

    const getCategoryInfo = (categorySlug) => {
        return (
            categories[categorySlug] || {
                name: categorySlug,
                color: "blue",
            }
        );
    };

    const categoryInfo = getCategoryInfo(post.category);

    return (
        <article
            className={`bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 rounded-lg shadow-lg overflow-hidden hover:shadow-xl transition-shadow ${className}`}
        >
            {post.featuredImage && (
                <OptimizedImage
                    src={post.featuredImage}
                    alt={post.featuredImageAlt || post.title}
                    className="w-full h-64 sm:h-80 object-cover"
                    priority={true}
                    sizes="(max-width: 768px) 100vw, (max-width: 1024px) 80vw, 1000px"
                />
            )}
            <div className="p-8">
                <div className="flex flex-col sm:flex-row sm:items-center justify-between mb-4 gap-3">
                    <div className="flex items-center gap-4">
                        <span
                            className={`inline-block px-4 py-2 text-sm font-semibold text-white bg-${categoryInfo.color}-600 rounded-full`}
                        >
                            {categoryInfo.name}
                        </span>
                        <span className="text-sm text-gray-600 dark:text-gray-400 flex items-center">
                            <Calendar className="h-4 w-4 mr-1" />
                            {formatDate(post.publishedAt)}
                        </span>
                    </div>
                    <div className="flex items-center space-x-4 text-sm text-gray-600 dark:text-gray-400">
                        <span className="flex items-center">
                            <Clock className="h-4 w-4 mr-1" />
                            {post.readTime} min read
                        </span>
                        {post.author && (
                            <span className="flex items-center">
                                <User className="h-4 w-4 mr-1" />
                                {typeof post.author === "string"
                                    ? post.author
                                    : post.author.name}
                            </span>
                        )}
                    </div>
                </div>

                <h2 className="text-2xl sm:text-3xl font-bold text-gray-900 dark:text-white mb-4">
                    <a
                        href={`/blog/${post.slug}`}
                        className="hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                    >
                        {post.title}
                    </a>
                </h2>

                {post.subtitle && (
                    <p className="text-lg text-gray-600 dark:text-gray-300 mb-4 font-medium">
                        {post.subtitle}
                    </p>
                )}

                <p className="text-gray-700 dark:text-gray-300 mb-6 text-base leading-relaxed">
                    {post.excerpt}
                </p>

                <div className="flex items-center justify-between">
                    <div className="flex flex-wrap gap-2">
                        {post.tags &&
                            post.tags.slice(0, 4).map((tag, index) => (
                                <span
                                    key={index}
                                    className="px-3 py-1 text-xs bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-full"
                                >
                                    {tag}
                                </span>
                            ))}
                    </div>
                    <a
                        href={`/blog/${post.slug}`}
                        className="bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white px-6 py-3 rounded-lg font-medium transition-colors inline-flex items-center"
                    >
                        Read Full Article{" "}
                        <ArrowRight className="ml-2 h-5 w-5" />
                    </a>
                </div>
            </div>
        </article>
    );
};

/**
 * PostCardSkeleton Component - Loading placeholder
 */
export const PostCardSkeleton = ({ layout = "grid", className = "" }) => {
    if (layout === "list") {
        return (
            <article
                className={`bg-white dark:bg-gray-800 rounded-lg shadow-md overflow-hidden ${className}`}
            >
                <div className="sm:flex">
                    <div className="sm:w-1/3 lg:w-1/4">
                        <div className="w-full h-48 sm:h-full bg-gray-200 dark:bg-gray-700 animate-pulse"></div>
                    </div>
                    <div className="p-6 sm:w-2/3 lg:w-3/4">
                        <div className="flex items-center mb-3 gap-3">
                            <div className="h-6 w-20 bg-gray-200 dark:bg-gray-700 rounded-full animate-pulse"></div>
                            <div className="h-4 w-24 bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                        </div>
                        <div className="h-8 w-3/4 bg-gray-200 dark:bg-gray-700 rounded animate-pulse mb-3"></div>
                        <div className="space-y-2 mb-4">
                            <div className="h-4 w-full bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                            <div className="h-4 w-5/6 bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                        </div>
                        <div className="flex justify-between items-center">
                            <div className="flex gap-2">
                                <div className="h-6 w-16 bg-gray-200 dark:bg-gray-700 rounded-full animate-pulse"></div>
                                <div className="h-6 w-16 bg-gray-200 dark:bg-gray-700 rounded-full animate-pulse"></div>
                                <div className="h-6 w-16 bg-gray-200 dark:bg-gray-700 rounded-full animate-pulse"></div>
                            </div>
                            <div className="h-4 w-20 bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                        </div>
                    </div>
                </div>
            </article>
        );
    }

    return (
        <article
            className={`bg-white dark:bg-gray-800 rounded-lg shadow-md overflow-hidden ${className}`}
        >
            <div className="w-full h-48 bg-gray-200 dark:bg-gray-700 animate-pulse"></div>
            <div className="p-6">
                <div className="flex items-center mb-3 gap-2">
                    <div className="h-6 w-20 bg-gray-200 dark:bg-gray-700 rounded-full animate-pulse"></div>
                    <div className="h-4 w-24 bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                </div>
                <div className="h-6 w-3/4 bg-gray-200 dark:bg-gray-700 rounded animate-pulse mb-3"></div>
                <div className="space-y-2 mb-4">
                    <div className="h-4 w-full bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                    <div className="h-4 w-5/6 bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                    <div className="h-4 w-4/5 bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                </div>
                <div className="flex justify-between items-center">
                    <div className="h-4 w-20 bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                    <div className="h-4 w-20 bg-gray-200 dark:bg-gray-700 rounded animate-pulse"></div>
                </div>
            </div>
        </article>
    );
};

// Default export for backward compatibility
export default PostCardGrid;
