import React from 'react';
import { motion } from 'framer-motion';
import { Link } from '@inertiajs/react';
import { Mail, Phone, MessageSquare, Linkedin } from 'lucide-react';

export default function ContactBlock({ content }) {
    if (!content) {
        return null;
    }

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    return (
        <>
            {/* Contact Section */}
            <section
                id="contact"
                className="py-16 shadow-lg bg-gray-100 dark:bg-gray-800"
            >
                <div className="w-full mx-auto sm:px-6 lg:px-8">
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center mb-12"
                    >
                        <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                            {content.title}
                        </h2>
                        <div 
                            className="text-lg text-gray-600 dark:text-gray-300 max-w-2xl mx-auto"
                            dangerouslySetInnerHTML={{ __html: content.subtitle }}
                        />
                    </motion.div>

                    <div className="grid md:grid-cols-2 gap-8">
                        {/* Contact Form */}
                        <motion.div
                            initial="hidden"
                            whileInView="visible"
                            viewport={{ once: true }}
                            variants={fadeIn}
                            className="bg-gray-200 dark:bg-gray-900 rounded-lg shadow-md p-6"
                        >
                            <h3 className="text-xl font-semibold mb-6 text-blue-800 dark:text-blue-400">
                                Send Us a Message
                            </h3>
                            <form className="space-y-4">
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Full Name
                                        </label>
                                        <input
                                            type="text"
                                            className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-800 dark:text-white"
                                            placeholder="Your name"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                            Email Address
                                        </label>
                                        <input
                                            type="email"
                                            className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-800 dark:text-white"
                                            placeholder="your@email.com"
                                        />
                                    </div>
                                </div>
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                        Subject
                                    </label>
                                    <input
                                        type="text"
                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-800 dark:text-white"
                                        placeholder="How can we help you?"
                                    />
                                </div>
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                        Message
                                    </label>
                                    <textarea
                                        rows="4"
                                        className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-800 dark:text-white"
                                        placeholder="Please describe your academic needs..."
                                    ></textarea>
                                </div>
                                <button
                                    type="submit"
                                    className="w-full bg-blue-600 hover:bg-blue-700 text-white font-medium py-2 px-4 rounded-md transition-colors duration-200"
                                >
                                    Send Message
                                </button>
                            </form>
                        </motion.div>

                        {/* Contact Information */}
                        <motion.div
                            initial="hidden"
                            whileInView="visible"
                            viewport={{ once: true }}
                            variants={fadeIn}
                            className="hidden md:flex md:flex-col md:justify-between bg-gray-200 dark:bg-gray-900 rounded-lg shadow-md p-6"
                        >
                            <div>
                                <h3 className="text-xl font-semibold mb-6 text-blue-800 dark:text-gray-100">
                                    Contact Information
                                </h3>
                                <p className="text-gray-600 dark:text-white mb-8">
                                    Our academic consultants are available 24/7 to answer your questions and provide guidance on your academic journey.
                                </p>

                                {/* Contact Info - Database Driven Only */}
                                {content.contact_info && content.contact_info.length > 0 && (
                                    <div className="space-y-4">
                                        {content.contact_info.map((info, index) => {
                                            const IconComponent = info.icon === 'mail' ? Mail : 
                                                                 info.icon === 'phone' ? Phone : 
                                                                 MessageSquare;
                                            
                                            return (
                                                <div key={index} className="flex items-start">
                                                    <IconComponent className="h-6 w-6 text-blue-600 mr-3 mt-1" />
                                                    <div>
                                                        <h4 className="font-medium text-gray-900 dark:text-white">
                                                            {info.title}
                                                        </h4>
                                                        {info.description && (
                                                            <p className="text-gray-600 dark:text-gray-300">
                                                                {info.description}
                                                            </p>
                                                        )}
                                                        {info.value && info.action ? (
                                                            info.action === '/chat' || info.action.includes('tawk') || info.action.includes('javascript:') ? (
                                                                <button
                                                                    onClick={(e) => {
                                                                        e.preventDefault();
                                                                        if (window.Tawk_API && window.Tawk_API.maximize) {
                                                                            window.Tawk_API.maximize();
                                                                        }
                                                                    }}
                                                                    className="text-blue-600 dark:text-blue-400 hover:underline cursor-pointer"
                                                                >
                                                                    {info.value}
                                                                </button>
                                                            ) : (
                                                                <Link 
                                                                    href={info.action}
                                                                    className="text-blue-600 dark:text-blue-400 hover:underline"
                                                                >
                                                                    {info.value}
                                                                </Link>
                                                            )
                                                        ) : info.value ? (
                                                            <p className="text-gray-600 dark:text-gray-300">
                                                                {info.value}
                                                            </p>
                                                        ) : null}
                                                    </div>
                                                </div>
                                            );
                                        })}
                                    </div>
                                )}
                            </div>

                            {/* Social Links - Only show if provided */}
                            {content.social_links && content.social_links.length > 0 && (
                                <div className="mt-8">
                                    <h4 className="font-medium text-gray-900 dark:text-white mb-3">
                                        Follow Us
                                    </h4>
                                    <div className="flex space-x-4">
                                        {content.social_links.map((link, index) => {
                                            // Only render if URL is provided
                                            if (!link.url) return null;
                                            
                                            return (
                                                <Link
                                                    key={index}
                                                    href={link.url}
                                                    className="text-blue-600 dark:text-gray-400 hover:text-blue-700 dark:hover:text-blue-400 transition-colors"
                                                    aria-label={link.platform ? `Visit Our ${link.platform} Page` : 'Visit Our Social Media'}
                                                >
                                                    {link.icon === 'linkedin' ? (
                                                        <Linkedin className="h-6 w-6" />
                                                    ) : link.icon === 'facebook' ? (
                                                        <svg className="h-6 w-6" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true">
                                                            <path fillRule="evenodd" d="M22 12c0-5.523-4.477-10-10-10S2 6.477 2 12c0 4.991 3.657 9.128 8.438 9.878v-6.987h-2.54V12h2.54V9.797c0-2.506 1.492-3.89 3.777-3.89 1.094 0 2.238.195 2.238.195v2.46h-1.26c-1.243 0-1.63.771-1.63 1.562V12h2.773l-.443 2.89h-2.33v6.988C18.343 21.128 22 16.991 22 12z" clipRule="evenodd" />
                                                        </svg>
                                                    ) : link.icon === 'twitter' ? (
                                                        <svg className="h-6 w-6" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true">
                                                            <path d="M8.29 20.251c7.547 0 11.675-6.253 11.675-11.675 0-.178 0-.355-.012-.53A8.348 8.348 0 0022 5.92a8.19 8.19 0 01-2.357.646 4.118 4.118 0 001.804-2.27 8.224 8.224 0 01-2.605.996 4.107 4.107 0 00-6.993 3.743 11.65 11.65 0 01-8.457-4.287 4.106 4.106 0 001.27 5.477A4.072 4.072 0 012.8 9.713v.052a4.105 4.105 0 003.292 4.022 4.095 4.095 0 01-1.853.07 4.108 4.108 0 003.834 2.85A8.233 8.233 0 012 18.407a11.616 11.616 0 006.29 1.84" />
                                                        </svg>
                                                    ) : link.icon === 'instagram' ? (
                                                        <svg className="h-6 w-6" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true">
                                                            <path fillRule="evenodd" d="M12.315 2c2.43 0 2.784.013 3.808.06 1.064.049 1.791.218 2.427.465a4.902 4.902 0 011.772 1.153 4.902 4.902 0 011.153 1.772c.247.636.416 1.363.465 2.427.048 1.067.06 1.407.06 4.123v.08c0 2.643-.012 2.987-.06 4.043-.049 1.064-.218 1.791-.465 2.427a4.902 4.902 0 01-1.153 1.772 4.902 4.902 0 01-1.772 1.153c-.636.247-1.363.416-2.427.465-1.067.048-1.407.06-4.123.06h-.08c-2.643 0-2.987-.012-4.043-.06-1.064-.049-1.791-.218-2.427-.465a4.902 4.902 0 01-1.772-1.153 4.902 4.902 0 01-1.153-1.772c-.247-.636-.416-1.363-.465-2.427-.047-1.024-.06-1.379-.06-3.808v-.63c0-2.43.013-2.784.06-3.808.049-1.064.218-1.791.465-2.427a4.902 4.902 0 011.153-1.772A4.902 4.902 0 015.45 2.525c.636-.247 1.363-.416 2.427-.465C8.901 2.013 9.256 2 11.685 2h.63zm-.081 1.802h-.468c-2.456 0-2.784.011-3.807.058-.975.045-1.504.207-1.857.344-.467.182-.8.398-1.15.748-.35.35-.566.683-.748 1.15-.137.353-.3.882-.344 1.857-.047 1.023-.058 1.351-.058 3.807v.468c0 2.456.011 2.784.058 3.807.045.975.207 1.504.344 1.857.182.466.399.8.748 1.15.35.35.683.566 1.15.748.353.137.882.3 1.857.344 1.054.048 1.37.058 4.041.058h.08c2.597 0 2.917-.01 3.96-.058.976-.045 1.505-.207 1.858-.344.466-.182.8-.398 1.15-.748.35-.35.566-.683.748-1.15.137-.353.3-.882.344-1.857.048-1.055.058-1.37.058-4.041v-.08c0-2.597-.01-2.917-.058-3.96-.045-.976-.207-1.505-.344-1.858a3.097 3.097 0 00-.748-1.15 3.098 3.098 0 00-1.15-.748c-.353-.137-.882-.3-1.857-.344-1.023-.047-1.351-.058-3.807-.058zM12 6.865a5.135 5.135 0 110 10.27 5.135 5.135 0 010-10.27zm0 1.802a3.333 3.333 0 100 6.666 3.333 3.333 0 000-6.666zm5.338-3.205a1.2 1.2 0 110 2.4 1.2 1.2 0 010-2.4z" clipRule="evenodd" />
                                                        </svg>
                                                    ) : (
                                                        // Fallback for unknown icons
                                                        <span className="h-6 w-6 flex items-center justify-center text-xs font-bold">
                                                            {link.platform?.charAt(0).toUpperCase() || '?'}
                                                        </span>
                                                    )}
                                                </Link>
                                            );
                                        })}
                                    </div>
                                </div>
                            )}
                        </motion.div>
                    </div>
                </div>
            </section>
        </>
    );
}

