import React, { useMemo, useState } from 'react';
import { ChevronDown, Search } from 'lucide-react';

export default function FAQsBlock({ content }) {
    if (!content) return null;

    const title = content.title || '';
    const subtitle = content.subtitle || '';
    
    // Support both simple structure (faqs array) and complex structure (categories array)
    const hasCategories = content.categories && Array.isArray(content.categories) && content.categories.length > 0;
    const hasFaqs = content.faqs && Array.isArray(content.faqs) && content.faqs.length > 0;
    
    // Use categories if available, otherwise create a default category from faqs
    const categories = hasCategories 
        ? content.categories 
        : hasFaqs 
            ? [{ name: 'All FAQs', faqs: content.faqs }]
            : [];

    const [activeCategory, setActiveCategory] = useState(0);
    const [openFAQ, setOpenFAQ] = useState(null);
    const [searchTerm, setSearchTerm] = useState('');

    const allFaqs = useMemo(() => {
        return categories.flatMap((cat) =>
            (cat.faqs || []).map((f) => ({ ...f, categoryName: cat.name || '' }))
        );
    }, [categories]);

    const filteredFaqs = useMemo(() => {
        if (!searchTerm) return (categories[activeCategory]?.faqs) || [];
        const s = searchTerm.toLowerCase();
        return allFaqs.filter((f) =>
            (f.question || '').toLowerCase().includes(s) ||
            (f.answer || '').toLowerCase().includes(s) ||
            (f.keywords || []).some((k) => (k || '').toLowerCase().includes(s))
        );
    }, [searchTerm, allFaqs, categories, activeCategory]);

    const toggleFAQ = (index) => setOpenFAQ(openFAQ === index ? null : index);

    // Return null if no FAQs at all
    if (categories.length === 0) {
        return null;
    }

    return (
        <section className="py-20 bg-gradient-to-br from-gray-50 to-blue-50 dark:from-gray-900 dark:to-gray-800 w-full">
            <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                {/* Header */}
                <div className="text-center mb-10">
                    {title && (
                        <h2 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white mb-6">{title}</h2>
                    )}
                    {subtitle && (
                        <p className="mt-2 max-w-3xl mx-auto text-lg text-gray-600 dark:text-gray-300 leading-relaxed">{subtitle}</p>
                    )}
                </div>

                {/* Search */}
                <div className="mb-10 max-w-2xl mx-auto">
                    <div className="relative">
                        <Search className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
                        <input
                            type="text"
                            placeholder="Search FAQs..."
                            className="w-full pl-12 pr-4 py-4 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-800 dark:text-white text-lg"
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                        />
                    </div>
                    {searchTerm && (
                        <div className="mt-3 text-center text-gray-600 dark:text-gray-300">
                            Found {filteredFaqs.length} results for "{searchTerm}"{' '}
                            <button className="ml-2 text-blue-600 dark:text-blue-400 hover:underline" onClick={() => setSearchTerm('')}>Clear</button>
                        </div>
                    )}
                </div>

                {/* Categories - Only show if there are multiple categories */}
                {!searchTerm && categories.length > 1 && (
                    <div className="mb-10">
                        <div className="flex flex-wrap justify-center gap-3">
                            {categories.map((cat, idx) => (
                                <button
                                    key={idx}
                                    className={`px-5 py-3 rounded-xl font-medium transition-all duration-300 ${
                                        activeCategory === idx
                                            ? 'bg-gradient-to-r from-blue-600 to-purple-600 text-white shadow-lg'
                                            : 'bg-white dark:bg-gray-800 text-gray-700 dark:text-gray-300 hover:shadow-md border border-gray-200 dark:border-gray-700'
                                    }`}
                                    onClick={() => { setActiveCategory(idx); setOpenFAQ(null); }}
                                >
                                    {cat.name}
                                    {cat.faqs && (
                                        <span className="ml-2 text-sm opacity-75">({cat.faqs.length})</span>
                                    )}
                                </button>
                            ))}
                        </div>
                        {categories[activeCategory]?.description && (
                            <div className="text-center mt-4 text-gray-600 dark:text-gray-300">
                                {categories[activeCategory].description}
                            </div>
                        )}
                    </div>
                )}

                {/* FAQ Items */}
                <div className="space-y-4 mb-12">
                    {filteredFaqs.map((faq, index) => (
                        <div key={index} className="bg-white dark:bg-gray-800 rounded-xl shadow border border-gray-200 dark:border-gray-700 overflow-hidden">
                            <button
                                className="flex justify-between items-center w-full p-5 text-left hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors duration-200"
                                onClick={() => toggleFAQ(index)}
                            >
                                <div className="flex-1">
                                    <h3 className="font-semibold text-lg text-gray-900 dark:text-white pr-4">{faq.question}</h3>
                                    {searchTerm && faq.categoryName && (
                                        <div className="mt-1 text-sm text-blue-600 dark:text-blue-400">{faq.categoryName}</div>
                                    )}
                                </div>
                                <ChevronDown className={`w-5 h-5 text-gray-500 dark:text-gray-400 transition-transform ${openFAQ === index ? 'rotate-180' : ''}`} />
                            </button>
                            {openFAQ === index && (
                                <div className="px-5 pb-5">
                                    <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
                                        <div 
                                            className="text-gray-700 dark:text-gray-300 leading-relaxed prose prose-sm dark:prose-invert max-w-none"
                                            dangerouslySetInnerHTML={{ __html: faq.answer }}
                                        />
                                        {Array.isArray(faq.keywords) && faq.keywords.length > 0 && (
                                            <div className="mt-3 flex flex-wrap gap-2">
                                                {faq.keywords.map((k, i) => (
                                                    <span key={i} className="text-xs bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300 px-2 py-1 rounded-full">{k}</span>
                                                ))}
                                            </div>
                                        )}
                                    </div>
                                </div>
                            )}
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
}
