import React, { useState } from 'react';

export default function GalleryBlock({ content }) {
    if (!content) return null;

    const {
        title = '',
        images = []
    } = content;

    const [selectedImage, setSelectedImage] = useState(null);

    if (!images || images.length === 0) {
        return null;
    }

    return (
        <section className="gallery-section py-16 bg-white">
            <div className="container mx-auto px-4">
                {title && (
                    <div className="text-center mb-12">
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900">
                            {title}
                        </h2>
                    </div>
                )}
                
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                    {images.map((image, index) => (
                        <div 
                            key={index} 
                            className="relative aspect-square overflow-hidden rounded-lg cursor-pointer group"
                            onClick={() => setSelectedImage(image)}
                        >
                            <img
                                src={image.url || image.src}
                                alt={image.alt || image.caption || `Gallery image ${index + 1}`}
                                className="w-full h-full object-cover transition-transform group-hover:scale-105"
                            />
                            
                            {/* Overlay */}
                            <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-30 transition-all duration-300 flex items-center justify-center">
                                <div className="opacity-0 group-hover:opacity-100 transition-opacity">
                                    <svg className="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0zM10 7v3m0 0v3m0-3h3m-3 0H7" />
                                    </svg>
                                </div>
                            </div>
                            
                            {/* Caption */}
                            {image.caption && (
                                <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black to-transparent p-4">
                                    <p className="text-white text-sm">{image.caption}</p>
                                </div>
                            )}
                        </div>
                    ))}
                </div>
            </div>
            
            {/* Modal for enlarged image */}
            {selectedImage && (
                <div 
                    className="fixed inset-0 bg-black bg-opacity-90 z-50 flex items-center justify-center p-4"
                    onClick={() => setSelectedImage(null)}
                >
                    <div className="relative max-w-4xl max-h-full">
                        <button
                            onClick={() => setSelectedImage(null)}
                            className="absolute -top-12 right-0 text-white hover:text-gray-300 transition-colors"
                        >
                            <svg className="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                            </svg>
                        </button>
                        
                        <img
                            src={selectedImage.url || selectedImage.src}
                            alt={selectedImage.alt || selectedImage.caption || 'Enlarged gallery image'}
                            className="max-w-full max-h-full object-contain rounded-lg"
                        />
                        
                        {selectedImage.caption && (
                            <p className="text-white text-center mt-4">{selectedImage.caption}</p>
                        )}
                    </div>
                </div>
            )}
        </section>
    );
}
