import React from 'react';
import { motion } from 'framer-motion';
import {
    Users,
    BookOpen,
    Award,
    Clock,
    Star,
    Globe,
    TrendingUp,
    CheckCircle,
    GraduationCap,
    FileText,
    Heart,
    Shield,
    Target,
    Zap,
} from 'lucide-react';

const iconMap = {
    'heart': Heart,
    'book-open': BookOpen,
    'users': Users,
    'target': Target,
    'clock': Clock,
    'globe': Globe,
    'shield': Shield,
    'award': Award,
    'star': Star,
    'zap': Zap,
    'graduation-cap': GraduationCap,
    'file-text': FileText,
};

export default function StatsBlock({ content }) {
    if (!content) {
        return null;
    }

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
            },
        },
    };

    const getIconComponent = (iconName, className = "w-12 h-12") => {
        const IconComponent = iconMap[iconName];
        return IconComponent ? <IconComponent className={className} /> : <BookOpen className={className} />;
    };

    const getColorClasses = (color) => {
        const colorMap = {
            red: 'text-red-500',
            blue: 'text-blue-500',
            green: 'text-green-500',
            purple: 'text-purple-500',
            yellow: 'text-yellow-500',
            orange: 'text-orange-500',
        };
        return colorMap[color] || 'text-blue-500';
    };

    return (
        <>
            {/* Writer Statistics Section */}
            <section className="w-full bg-white dark:bg-gray-800 py-16">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center mb-12"
                    >
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                            {content.title}
                        </h3>
                        <div 
                            className="text-lg text-gray-600 dark:text-gray-300 max-w-3xl mx-auto prose prose-lg dark:prose-invert"
                            dangerouslySetInnerHTML={{ __html: content.subtitle }}
                        />
                    </motion.div>

                    {/* Main Stats Grid */}
                    {content.main_stats && content.main_stats.length > 0 && (
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8 mb-16">
                            {content.main_stats.map((stat, index) => (
                                <motion.div
                                    key={index}
                                    initial="hidden"
                                    whileInView="visible"
                                    viewport={{ once: true }}
                                    variants={fadeIn}
                                    className="text-center"
                                >
                                    <div className="bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-700 dark:to-gray-600 rounded-xl p-6">
                                        <div className="flex justify-center mb-4">
                                            <div className={getColorClasses(stat.color)}>
                                                {getIconComponent(stat.icon)}
                                            </div>
                                        </div>
                                        <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                            {stat.number}
                                        </div>
                                        <h4 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                                            {stat.label}
                                        </h4>
                                        <div 
                                            className="text-sm text-gray-600 dark:text-gray-300 mb-2 prose prose-sm dark:prose-invert max-w-none"
                                            dangerouslySetInnerHTML={{ __html: stat.description }}
                                        />
                                        {stat.detail && (
                                            <div 
                                                className="text-xs text-gray-500 dark:text-gray-400 prose prose-xs dark:prose-invert max-w-none"
                                                dangerouslySetInnerHTML={{ __html: stat.detail }}
                                            />
                                        )}
                                    </div>
                                </motion.div>
                            ))}
                        </div>
                    )}

                    {/* Additional Stats */}
                    {content.additional_stats && content.additional_stats.length > 0 && (
                        <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-6 mb-16">
                            {content.additional_stats.map((stat, index) => (
                                <motion.div
                                    key={index}
                                    initial="hidden"
                                    whileInView="visible"
                                    viewport={{ once: true }}
                                    variants={fadeIn}
                                    className="text-center"
                                >
                                    <div className="bg-white dark:bg-gray-700 rounded-lg p-4 shadow-sm">
                                        <div className="flex justify-center mb-2">
                                            <div className={getColorClasses(stat.color || 'blue')}>
                                                {getIconComponent(stat.icon, "w-8 h-8")}
                                            </div>
                                        </div>
                                        <div className="text-2xl font-bold text-gray-900 dark:text-white mb-1">
                                            {stat.number}
                                        </div>
                                        <h5 className="text-sm font-semibold text-gray-900 dark:text-white mb-1">
                                            {stat.label}
                                        </h5>
                                        <div 
                                            className="text-xs text-gray-600 dark:text-gray-300 prose prose-xs dark:prose-invert max-w-none"
                                            dangerouslySetInnerHTML={{ __html: stat.description }}
                                        />
                                    </div>
                                </motion.div>
                            ))}
                        </div>
                    )}

                    {/* Writer Qualifications */}
                    {content.writer_stats && content.writer_stats.length > 0 && (
                        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
                            <motion.div
                                initial="hidden"
                                whileInView="visible"
                                viewport={{ once: true }}
                                variants={staggerContainer}
                                className="space-y-6"
                            >
                                {content.writer_stats.map((stat, index) => (
                                    <motion.div
                                        key={index}
                                        variants={fadeIn}
                                        className="flex items-center space-x-4"
                                    >
                                        <div className="flex-shrink-0 w-20 text-right">
                                            <div className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                                                {stat.count}
                                            </div>
                                        </div>
                                        <div className="flex-grow">
                                            <div className="flex items-center justify-between mb-1">
                                                <h4 className="font-semibold text-gray-900 dark:text-white">
                                                    {stat.label}
                                                </h4>
                                            </div>
                                            <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2 mb-2">
                                                <div
                                                    className="bg-gradient-to-r from-blue-500 to-purple-600 h-2 rounded-full transition-all duration-1000"
                                                    style={{
                                                        width: `${stat.percentage}%`,
                                                    }}
                                                ></div>
                                            </div>
                                            <div 
                                                className="text-sm text-gray-600 dark:text-gray-300 prose prose-sm dark:prose-invert max-w-none"
                                                dangerouslySetInnerHTML={{ __html: stat.description }}
                                            />
                                        </div>
                                    </motion.div>
                                ))}
                            </motion.div>

                            {/* Expertise Distribution */}
                            {content.expertise_distribution && content.expertise_distribution.length > 0 && (
                                <motion.div
                                    initial="hidden"
                                    whileInView="visible"
                                    viewport={{ once: true }}
                                    variants={fadeIn}
                                    className="bg-gradient-to-br from-blue-50 to-purple-50 dark:from-gray-700 dark:to-gray-600 rounded-2xl p-8"
                                >
                                    <h4 className="text-2xl font-bold text-gray-900 dark:text-white mb-6 text-center">
                                        Writer Expertise Distribution
                                    </h4>
                                    <div className="grid grid-cols-2 gap-4">
                                        {content.expertise_distribution.map((field, index) => (
                                            <div
                                                key={index}
                                                className="text-center p-4 bg-white dark:bg-gray-800 rounded-lg shadow-sm"
                                            >
                                                <div
                                                    className={`w-12 h-12 bg-${field.color}-500 rounded-full flex items-center justify-center mx-auto mb-2`}
                                                >
                                                    <span className="text-white font-bold">
                                                        {field.count}
                                                    </span>
                                                </div>
                                                <h5 className="font-semibold text-gray-900 dark:text-white text-sm">
                                                    {field.field}
                                                </h5>
                                            </div>
                                        ))}
                                    </div>
                                </motion.div>
                            )}
                        </div>
                    )}
                </div>
            </section>
        </>
    );
}

