import React, { useState, useEffect } from 'react';
import { 
    Download, 
    Eye, 
    CheckCircle, 
    RotateCcw, 
    Clock, 
    AlertCircle,
    FileText,
    MessageSquare
} from 'lucide-react';
import { router } from '@inertiajs/react';

const ClientApprovalInterface = ({ order, className = '' }) => {
    const [showRevisionModal, setShowRevisionModal] = useState(false);
    const [revisionNotes, setRevisionNotes] = useState('');
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [timeRemaining, setTimeRemaining] = useState(null);
    const [error, setError] = useState('');

    // Calculate time remaining until auto-approval
    useEffect(() => {
        if (order.auto_approval_at) {
            const updateTimer = () => {
                const now = new Date().getTime();
                const autoApprovalTime = new Date(order.auto_approval_at).getTime();
                const remaining = autoApprovalTime - now;

                if (remaining > 0) {
                    setTimeRemaining(remaining);
                } else {
                    setTimeRemaining(0);
                }
            };

            updateTimer();
            const interval = setInterval(updateTimer, 1000);

            return () => clearInterval(interval);
        }
    }, [order.auto_approval_at]);

    const formatTimeRemaining = (milliseconds) => {
        if (milliseconds <= 0) return 'Auto-approval due';
        
        const hours = Math.floor(milliseconds / (1000 * 60 * 60));
        const minutes = Math.floor((milliseconds % (1000 * 60 * 60)) / (1000 * 60));
        const seconds = Math.floor((milliseconds % (1000 * 60)) / 1000);
        
        return `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    };

    const handleApprove = async () => {
        setIsSubmitting(true);
        setError('');

        try {
            await router.post(`/orders/${order.id}/approve`, {}, {
                onSuccess: () => {
                    // Success handled by page refresh
                },
                onError: (errors) => {
                    setError(errors.message || 'Failed to approve order');
                }
            });
        } catch (err) {
            setError('An error occurred while approving the order');
        } finally {
            setIsSubmitting(false);
        }
    };

    const handleRequestRevision = async () => {
        if (!revisionNotes.trim()) {
            setError('Please provide revision notes');
            return;
        }

        setIsSubmitting(true);
        setError('');

        try {
            await router.post(`/orders/${order.id}/revision`, {
                revision_notes: revisionNotes
            }, {
                onSuccess: () => {
                    setShowRevisionModal(false);
                    setRevisionNotes('');
                },
                onError: (errors) => {
                    setError(errors.revision_notes || errors.message || 'Failed to request revision');
                }
            });
        } catch (err) {
            setError('An error occurred while requesting revision');
        } finally {
            setIsSubmitting(false);
        }
    };

    const getOrderFiles = () => {
        if (!order.files || !Array.isArray(order.files)) {
            return [];
        }
        return order.files.filter(file => file.upload_type === 'admin_upload');
    };

    const formatDate = (dateString) => {
        if (!dateString) return null;
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const files = getOrderFiles();

    return (
        <div className={`bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-lg p-6 ${className}`}>
            <div className="flex items-center mb-6">
                <Eye className="w-5 h-5 text-blue-600 dark:text-blue-400 mr-2" />
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                    Review & Approve Work
                </h3>
            </div>

            {/* Auto-approval countdown */}
            {order.auto_approval_at && (
                <div className="mb-6 p-4 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-700 rounded-md">
                    <div className="flex items-center">
                        <Clock className="w-5 h-5 text-yellow-600 dark:text-yellow-400 mr-2" />
                        <div>
                            <p className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                                Auto-approval in progress
                            </p>
                            <p className="text-xs text-yellow-700 dark:text-yellow-300 mt-1">
                                If you don't take action, this order will be automatically approved in:
                            </p>
                        </div>
                    </div>
                    <div className="mt-3">
                        <div className="text-2xl font-mono font-bold text-yellow-800 dark:text-yellow-200">
                            {formatTimeRemaining(timeRemaining)}
                        </div>
                        <p className="text-xs text-yellow-600 dark:text-yellow-400 mt-1">
                            Auto-approval scheduled for: {formatDate(order.auto_approval_at)}
                        </p>
                    </div>
                </div>
            )}

            {/* Work files */}
            <div className="mb-6">
                <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                    Submitted Work Files
                </h4>
                {files.length > 0 ? (
                    <div className="space-y-3">
                        {files.map((file, index) => (
                            <div key={index} className="flex items-center justify-between p-3 bg-white dark:bg-gray-800 rounded-md border border-gray-200 dark:border-gray-700">
                                <div className="flex items-center">
                                    <FileText className="w-4 h-4 text-gray-500 dark:text-gray-400 mr-3" />
                                    <div>
                                        <p className="text-sm font-medium text-gray-900 dark:text-white">
                                            {file.file_name}
                                        </p>
                                        {file.description && (
                                            <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                {file.description}
                                            </p>
                                        )}
                                    </div>
                                </div>
                                <div className="flex items-center space-x-2">
                                    <a
                                        href={`/storage/${file.file_path}`}
                                        download={file.file_name}
                                        className="inline-flex items-center px-3 py-1 text-xs font-medium text-blue-600 dark:text-blue-400 hover:text-blue-700 dark:hover:text-blue-300 transition-colors"
                                    >
                                        <Download className="w-3 h-3 mr-1" />
                                        Download
                                    </a>
                                    <a
                                        href={`/storage/${file.file_path}`}
                                        target="_blank"
                                        rel="noopener noreferrer"
                                        className="inline-flex items-center px-3 py-1 text-xs font-medium text-gray-600 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300 transition-colors"
                                    >
                                        <Eye className="w-3 h-3 mr-1" />
                                        Preview
                                    </a>
                                </div>
                            </div>
                        ))}
                    </div>
                ) : (
                    <div className="text-center py-8">
                        <FileText className="w-12 h-12 text-gray-400 mx-auto mb-3" />
                        <p className="text-sm text-gray-500 dark:text-gray-400">
                            No work files have been uploaded yet.
                        </p>
                    </div>
                )}
            </div>

            {/* Action buttons */}
            <div className="flex flex-col sm:flex-row gap-3">
                <button
                    onClick={handleApprove}
                    disabled={isSubmitting}
                    className="flex-1 flex items-center justify-center px-4 py-2 bg-green-600 hover:bg-green-700 disabled:bg-gray-400 text-white font-medium rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition-colors duration-200"
                >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    {isSubmitting ? 'Approving...' : 'Approve Work'}
                </button>
                
                <button
                    onClick={() => setShowRevisionModal(true)}
                    disabled={isSubmitting}
                    className="flex-1 flex items-center justify-center px-4 py-2 bg-orange-600 hover:bg-orange-700 disabled:bg-gray-400 text-white font-medium rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-orange-500 focus:ring-offset-2 transition-colors duration-200"
                >
                    <RotateCcw className="w-4 h-4 mr-2" />
                    Request Revision
                </button>
            </div>

            {error && (
                <div className="mt-4 flex items-center p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-700 rounded-md">
                    <AlertCircle className="w-4 h-4 text-red-600 dark:text-red-400 mr-2" />
                    <span className="text-sm text-red-700 dark:text-red-300">{error}</span>
                </div>
            )}

            {/* Order information */}
            <div className="mt-6 pt-6 border-t border-gray-200 dark:border-gray-700">
                <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">
                    Order Details
                </h4>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                    <div>
                        <span className="text-gray-500 dark:text-gray-400">Order Title:</span>
                        <span className="ml-2 text-gray-900 dark:text-white font-medium">
                            {order.title}
                        </span>
                    </div>
                    <div>
                        <span className="text-gray-500 dark:text-gray-400">Order Number:</span>
                        <span className="ml-2 text-gray-900 dark:text-white font-medium">
                            #{order.order_number}
                        </span>
                    </div>
                    <div>
                        <span className="text-gray-500 dark:text-gray-400">Client Deadline:</span>
                        <span className="ml-2 text-gray-900 dark:text-white">
                            {formatDate(order.deadline)}
                        </span>
                    </div>
                    <div>
                        <span className="text-gray-500 dark:text-gray-400">Work Submitted:</span>
                        <span className="ml-2 text-gray-900 dark:text-white">
                            {formatDate(order.work_submitted_at)}
                        </span>
                    </div>
                    {order.revision_count > 0 && (
                        <div>
                            <span className="text-gray-500 dark:text-gray-400">Revision Count:</span>
                            <span className="ml-2 text-gray-900 dark:text-white">
                                {order.revision_count}
                            </span>
                        </div>
                    )}
                </div>
            </div>

            {/* Revision Modal */}
            {showRevisionModal && (
                <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
                    <div className="bg-white dark:bg-gray-800 rounded-lg p-6 max-w-md w-full mx-4">
                        <div className="flex items-center mb-4">
                            <MessageSquare className="w-5 h-5 text-orange-600 dark:text-orange-400 mr-2" />
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                Request Revision
                            </h3>
                        </div>
                        
                        <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                            Please provide detailed feedback about what needs to be changed or improved in the work.
                        </p>

                        <div className="mb-4">
                            <label htmlFor="revision-notes" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Revision Notes
                            </label>
                            <textarea
                                id="revision-notes"
                                value={revisionNotes}
                                onChange={(e) => setRevisionNotes(e.target.value)}
                                rows={4}
                                className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-orange-500 focus:border-orange-500 dark:bg-gray-700 dark:text-white"
                                placeholder="Describe what changes are needed..."
                                required
                            />
                        </div>

                        <div className="flex justify-end space-x-3">
                            <button
                                onClick={() => {
                                    setShowRevisionModal(false);
                                    setRevisionNotes('');
                                    setError('');
                                }}
                                className="px-4 py-2 text-sm text-gray-600 dark:text-gray-400 bg-gray-100 dark:bg-gray-700 rounded-md hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
                            >
                                Cancel
                            </button>
                            <button
                                onClick={handleRequestRevision}
                                disabled={isSubmitting || !revisionNotes.trim()}
                                className="px-4 py-2 text-sm bg-orange-600 hover:bg-orange-700 disabled:bg-gray-400 text-white font-medium rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-orange-500 focus:ring-offset-2 transition-colors"
                            >
                                {isSubmitting ? 'Submitting...' : 'Submit Revision Request'}
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
};

export default ClientApprovalInterface; 