import React, { useState, useCallback } from "react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
    faFileWord,
    faFileExcel,
    faFilePowerpoint,
    faFilePdf,
    faFileArchive,
    faUpload,
    faXmark,
    faCircleExclamation,
    faInfoCircle,
} from "@fortawesome/free-solid-svg-icons";
import { useDropzone } from "react-dropzone";
import axios from "axios";
import clsx from "clsx";

// File type configurations with icons
const ALLOWED_TYPES = {
    // Microsoft Word
    "application/msword": { ext: ".doc", icon: faFileWord },
    "application/vnd.openxmlformats-officedocument.wordprocessingml.document": {
        ext: ".docx",
        icon: faFileWord,
    },

    // Microsoft Excel
    "application/vnd.ms-excel": { ext: ".xls", icon: faFileExcel },
    "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet": {
        ext: ".xlsx",
        icon: faFileExcel,
    },

    // Microsoft PowerPoint
    "application/vnd.ms-powerpoint": { ext: ".ppt", icon: faFilePowerpoint },
    "application/vnd.openxmlformats-officedocument.presentationml.presentation": {
        ext: ".pptx",
        icon: faFilePowerpoint,
    },

    // PDF
    "application/pdf": { ext: ".pdf", icon: faFilePdf },

    // ZIP
    "application/zip": { ext: ".zip", icon: faFileArchive },
    "application/x-zip-compressed": { ext: ".zip", icon: faFileArchive },
};

// Maximum total file size allowed (40MB)
const MAX_FILE_SIZE = 40 * 1024 * 1024;

// File categories for client uploads
const FILE_CATEGORIES = [
    { value: 'sample', label: 'Sample Work', description: 'Upload a sample of your previous work or similar documents' },
    { value: 'instructions', label: 'Order Instructions', description: 'Upload detailed instructions or requirements for your order' },
];

export default function ClientFileUpload({ orderId, onFileUploaded }) {
    const [selectedFiles, setSelectedFiles] = useState([]);
    const [uploadProgress, setUploadProgress] = useState({});
    const [error, setError] = useState("");
    const [isUploading, setIsUploading] = useState(false);
    const [fileCategory, setFileCategory] = useState('instructions');
    const [fileDescription, setFileDescription] = useState('');

    const formatFileSize = (bytes) => {
        if (bytes === 0) return "0 Bytes";
        const k = 1024;
        const sizes = ["Bytes", "KB", "MB", "GB"];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
    };

    const onDrop = useCallback((acceptedFiles) => {
        setError("");
        const validFiles = acceptedFiles.filter((file) => {
            // Validate file type
            if (!Object.keys(ALLOWED_TYPES).includes(file.type)) {
                setError(`${file.name} is not an allowed file type`);
                return false;
            }
            // Validate total file size against MAX_FILE_SIZE
            if (file.size > MAX_FILE_SIZE) {
                setError(`${file.name} exceeds 40MB limit`);
                return false;
            }
            return true;
        });
        setSelectedFiles((prev) => [...prev, ...validFiles]);
    }, []);

    const { getRootProps, getInputProps, isDragActive } = useDropzone({
        onDrop,
        accept: Object.keys(ALLOWED_TYPES).reduce((acc, curr) => {
            acc[curr] = [ALLOWED_TYPES[curr].ext];
            return acc;
        }, {}),
        maxSize: MAX_FILE_SIZE,
    });

    const removeFile = (index) => {
        setSelectedFiles(prev => prev.filter((_, i) => i !== index));
        setUploadProgress(prev => {
            const newProgress = { ...prev };
            delete newProgress[index];
            return newProgress;
        });
    };

    const uploadFile = async (file, index) => {
        const formData = new FormData();
        formData.append('file', file);
        formData.append('order_id', orderId);
        formData.append('file_category', fileCategory);
        formData.append('description', fileDescription);
        formData.append('uploader_user_type', 'client');

        try {
            const response = await axios.post('/order-files/client-upload', formData, {
                headers: {
                    'Content-Type': 'multipart/form-data',
                },
                onUploadProgress: (progressEvent) => {
                    const percentCompleted = Math.round(
                        (progressEvent.loaded * 100) / progressEvent.total
                    );
                    setUploadProgress(prev => ({
                        ...prev,
                        [index]: percentCompleted
                    }));
                }
            });

            if (response.data.success) {
                // File uploaded successfully
                setUploadProgress(prev => ({
                    ...prev,
                    [index]: 100
                }));
                
                // Call the callback to refresh file list
                if (onFileUploaded) {
                    onFileUploaded();
                }
                
                return true;
            }
        } catch (error) {
            console.error('Upload error:', error);
            console.error('Error response:', error.response);
            console.error('Error response data:', error.response?.data);
            console.error('Validation errors:', error.response?.data?.errors);
            
            // Extract detailed error message
            let errorMessage = 'Upload failed';
            if (error.response?.data?.errors) {
                const errors = error.response.data.errors;
                errorMessage = Object.values(errors).flat().join(', ');
            } else if (error.response?.data?.message) {
                errorMessage = error.response.data.message;
            } else if (error.message) {
                errorMessage = error.message;
            }
            
            setError(errorMessage);
            return false;
        }
    };

    const handleUploadAll = async () => {
        if (selectedFiles.length === 0) {
            setError('Please select files to upload');
            return;
        }

        // Description is now optional - removed the validation check
        setIsUploading(true);
        setError("");

        const uploadPromises = selectedFiles.map((file, index) => uploadFile(file, index));
        const results = await Promise.all(uploadPromises);

        const successCount = results.filter(Boolean).length;
        
        if (successCount === selectedFiles.length) {
            // All files uploaded successfully
            setSelectedFiles([]);
            setFileDescription('');
            setUploadProgress({});
            setError('');
        } else {
            setError(`${selectedFiles.length - successCount} file(s) failed to upload`);
        }

        setIsUploading(false);
    };

    const getFileIcon = (file) => {
        const fileType = ALLOWED_TYPES[file.type];
        return fileType ? fileType.icon : faFileArchive;
    };

    return (
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="mb-6">
                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                    Upload Files
                </h3>
                <p className="text-sm text-gray-600">
                    Upload sample work or order instructions to help writers understand your requirements.
                </p>
            </div>

            {/* File Category Selection */}
            <div className="mb-4">
                <label className="block text-sm font-medium text-gray-700 mb-2">
                    File Type
                </label>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                    {FILE_CATEGORIES.map((category) => (
                        <label
                            key={category.value}
                            className={clsx(
                                "relative flex cursor-pointer rounded-lg border p-4 shadow-sm focus:outline-none",
                                fileCategory === category.value
                                    ? "border-blue-500 ring-2 ring-blue-500"
                                    : "border-gray-300"
                            )}
                        >
                            <input
                                type="radio"
                                name="file_category"
                                value={category.value}
                                checked={fileCategory === category.value}
                                onChange={(e) => setFileCategory(e.target.value)}
                                className="sr-only"
                            />
                            <div className="flex flex-1">
                                <div className="flex flex-col">
                                    <span className="block text-sm font-medium text-gray-900">
                                        {category.label}
                                    </span>
                                    <span className="mt-1 flex items-center text-sm text-gray-500">
                                        {category.description}
                                    </span>
                                </div>
                            </div>
                            {fileCategory === category.value && (
                                <div className="text-blue-500">
                                    <svg className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                        <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                                    </svg>
                                </div>
                            )}
                        </label>
                    ))}
                </div>
            </div>

            {/* File Description */}
            <div className="mb-4">
                <label htmlFor="file_description" className="block text-sm font-medium text-gray-700 mb-2">
                    File Description (Optional)
                </label>
                <textarea
                    id="file_description"
                    rows={3}
                    className="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border-gray-300 rounded-md"
                    placeholder="Describe what you're uploading and any specific requirements (optional)..."
                    value={fileDescription}
                    onChange={(e) => setFileDescription(e.target.value)}
                />
                <p className="mt-1 text-sm text-gray-500">
                    Provide a clear description to help writers understand your requirements. This field is optional.
                </p>
            </div>

            {/* File Upload Area */}
            <div className="mb-4">
                <div
                    {...getRootProps()}
                    className={clsx(
                        "border-2 border-dashed rounded-lg p-6 text-center cursor-pointer transition-colors",
                        isDragActive
                            ? "border-blue-500 bg-blue-50"
                            : "border-gray-300 hover:border-gray-400"
                    )}
                >
                    <input {...getInputProps()} />
                    <FontAwesomeIcon
                        icon={faUpload}
                        className="mx-auto h-12 w-12 text-gray-400 mb-4"
                    />
                    <p className="text-sm text-gray-600">
                        {isDragActive
                            ? "Drop the files here..."
                            : "Drag & drop files here, or click to select files"}
                    </p>
                    <p className="text-xs text-gray-500 mt-2">
                        Supported formats: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP (Max: 40MB)
                    </p>
                </div>
            </div>

            {/* Error Display */}
            {error && (
                <div className="mb-4 p-3 bg-red-50 border border-red-200 rounded-md">
                    <div className="flex">
                        <FontAwesomeIcon
                            icon={faCircleExclamation}
                            className="h-5 w-5 text-red-400 mt-0.5"
                        />
                        <div className="ml-3">
                            <p className="text-sm text-red-800">{error}</p>
                        </div>
                    </div>
                </div>
            )}

            {/* Selected Files */}
            {selectedFiles.length > 0 && (
                <div className="mb-4">
                    <h4 className="text-sm font-medium text-gray-700 mb-2">
                        Selected Files ({selectedFiles.length})
                    </h4>
                    <div className="space-y-2">
                        {selectedFiles.map((file, index) => (
                            <div
                                key={index}
                                className="flex items-center justify-between p-3 bg-gray-50 rounded-md"
                            >
                                <div className="flex items-center space-x-3">
                                    <FontAwesomeIcon
                                        icon={getFileIcon(file)}
                                        className="h-5 w-5 text-gray-500"
                                    />
                                    <div>
                                        <p className="text-sm font-medium text-gray-900">
                                            {file.name}
                                        </p>
                                        <p className="text-xs text-gray-500">
                                            {formatFileSize(file.size)}
                                        </p>
                                    </div>
                                </div>
                                
                                <div className="flex items-center space-x-2">
                                    {/* Upload Progress */}
                                    {uploadProgress[index] !== undefined && (
                                        <div className="flex items-center space-x-2">
                                            <div className="w-16 bg-gray-200 rounded-full h-2">
                                                <div
                                                    className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                                    style={{ width: `${uploadProgress[index]}%` }}
                                                ></div>
                                            </div>
                                            <span className="text-xs text-gray-500">
                                                {uploadProgress[index]}%
                                            </span>
                                        </div>
                                    )}
                                    
                                    <button
                                        onClick={() => removeFile(index)}
                                        className="text-red-500 hover:text-red-700"
                                        disabled={isUploading}
                                    >
                                        <FontAwesomeIcon icon={faXmark} className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            )}

            {/* Upload Button */}
            {selectedFiles.length > 0 && (
                <div className="flex justify-end">
                    <button
                        onClick={handleUploadAll}
                        disabled={isUploading}
                        className={clsx(
                            "inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white",
                            isUploading
                                ? "bg-gray-400 cursor-not-allowed"
                                : "bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                        )}
                    >
                        {isUploading ? (
                            <>
                                <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                </svg>
                                Uploading...
                            </>
                        ) : (
                            <>
                                <FontAwesomeIcon icon={faUpload} className="mr-2 h-4 w-4" />
                                Upload Files
                            </>
                        )}
                    </button>
                </div>
            )}
        </div>
    );
} 