import {
    Mail,
    Phone,
    MessageSquare,
    MapPin,
    Clock,
    Globe,
    Send,
    CheckCircle,
    Star,
    Award,
    Users,
    Shield,
    Zap,
    HeadphonesIcon,
    Calendar,
    FileText,
    GraduationCap,
    BookOpen,
    PenTool,
    Target,
    TrendingUp,
    Heart,
} from "lucide-react";
import { motion } from "framer-motion";
import { Link } from "@inertiajs/react";
import { useState } from "react";

export default function Contact({ fadeIn, staggerContainer }) {
    const [formData, setFormData] = useState({
        name: "",
        email: "",
        phone: "",
        service: "",
        subject: "",
        message: "",
        urgency: "standard",
        academicLevel: "",
        pages: "",
    });
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [activeContactMethod, setActiveContactMethod] = useState(0);

    const handleInputChange = (e) => {
        const { name, value } = e.target;
        setFormData((prev) => ({
            ...prev,
            [name]: value,
        }));
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        setIsSubmitting(true);

        // Track form submission
        if (typeof window.trackContactForm === "function") {
            window.trackContactForm(formData.service, "contact_section");
        }

        // Simulate form submission
        setTimeout(() => {
            setIsSubmitting(false);
            // Handle success/error states here
        }, 2000);
    };

    const handleCTAClick = (ctaText, section) => {
        if (typeof window.trackCTAClick === "function") {
            window.trackCTAClick(ctaText, "contact_section", section);
        }
    };

    const contactMethods = [
        {
            name: "Live Chat",
            icon: <MessageSquare className="w-8 h-8" />,
            title: "Instant Academic Writing Support",
            description:
                "Get immediate help with dissertation writing help, research paper writing service, and custom academic papers",
            availability: "24/7 Available",
            responseTime: "< 30 seconds",
            color: "from-blue-500 to-cyan-500",
            features: [
                "Instant quotes for academic writing services",
                "Direct connection to professional essay writers",
                "Real-time project updates",
                "Immediate order assistance",
            ],
        },
        {
            name: "Phone Support",
            icon: <Phone className="w-8 h-8" />,
            title: "Speak with Academic Writing Experts",
            description:
                "Talk directly with our specialists about professional essay writers and academic writing services",
            availability: "24/7 Available",
            responseTime: "Immediate",
            color: "from-green-500 to-emerald-500",
            features: [
                "Personalized consultation",
                "Complex project discussion",
                "Urgent order placement",
                "Payment assistance",
            ],
        },
        {
            name: "Email Support",
            icon: <Mail className="w-8 h-8" />,
            title: "Detailed Academic Writing Inquiries",
            description:
                "Send detailed questions about dissertation writing help and research paper writing service",
            availability: "24/7 Monitoring",
            responseTime: "< 2 hours",
            color: "from-purple-500 to-indigo-500",
            features: [
                "Detailed project requirements",
                "File attachments support",
                "Comprehensive quotes",
                "Academic consultation",
            ],
        },
    ];

    const supportStats = {
        responseTime: "< 2 minutes",
        satisfactionRate: "98%",
        supportAgents: "50+",
        languagesSupported: "15+",
    };

    const officeLocations = [
        {
            city: "New York",
            address: "123 Academic Avenue, NY 10001",
            phone: "+1 (555) 123-4567",
            email: "ny@academicscribe.com",
            timezone: "EST",
            specialties: ["Dissertation Writing Help", "PhD Support"],
        },
        {
            city: "London",
            address: "456 Scholar Street, London SW1A 1AA",
            phone: "+44 20 7123 4567",
            email: "london@academicscribe.com",
            timezone: "GMT",
            specialties: [
                "Research Paper Writing Service",
                "UK Academic Standards",
            ],
        },
        {
            city: "Sydney",
            address: "789 University Road, Sydney NSW 2000",
            phone: "+61 2 9123 4567",
            email: "sydney@academicscribe.com",
            timezone: "AEST",
            specialties: ["Custom Academic Papers", "Australian Universities"],
        },
    ];

    return (
        <>
            {/* Main Contact Section */}
            <section
                id="contact"
                className="py-20 bg-gradient-to-br from-gray-50 to-blue-50 dark:from-gray-900 dark:to-gray-800"
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Section Header */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white mb-6">
                            Contact Our Academic Writing Services Team
                        </h2>
                        <div className="mt-4 h-1 w-24 bg-gradient-to-r from-blue-600 to-purple-600 mx-auto rounded-full"></div>
                        <p className="mt-6 max-w-4xl mx-auto text-xl text-gray-600 dark:text-gray-300 leading-relaxed">
                            Ready to get started with our{" "}
                            <strong>professional essay writers</strong>? Contact
                            us for <strong>dissertation writing help</strong>,{" "}
                            <strong>research paper writing service</strong>, or{" "}
                            <strong>custom academic papers</strong>. Our{" "}
                            <strong>academic writing services</strong> team is
                            available 24/7 to assist with your academic success.
                        </p>
                    </motion.div>

                    {/* Support Statistics */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl p-8 border border-gray-200 dark:border-gray-700">
                            <h3 className="text-2xl font-bold text-gray-900 dark:text-white text-center mb-8">
                                Academic Writing Services - Support Excellence
                            </h3>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Clock className="w-12 h-12 text-blue-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {supportStats.responseTime}
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Average Response Time
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Star className="w-12 h-12 text-yellow-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {supportStats.satisfactionRate}
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Support Satisfaction
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Users className="w-12 h-12 text-green-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {supportStats.supportAgents}
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Support Specialists
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Globe className="w-12 h-12 text-purple-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {supportStats.languagesSupported}
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Languages Supported
                                    </div>
                                </div>
                            </div>
                        </div>
                    </motion.div>

                    {/* Contact Methods */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white text-center mb-12">
                            Choose Your Preferred Contact Method
                        </h3>

                        {/* Contact Method Tabs */}
                        <div className="flex justify-center mb-8">
                            <div className="flex rounded-xl bg-white dark:bg-gray-800 p-2 shadow-lg border border-gray-200 dark:border-gray-700">
                                {contactMethods.map((method, idx) => (
                                    <button
                                        key={idx}
                                        className={`flex items-center px-6 py-3 rounded-lg font-medium transition-all duration-300 ${
                                            activeContactMethod === idx
                                                ? "bg-gradient-to-r from-blue-600 to-purple-600 text-white shadow-md transform scale-105"
                                                : "text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-700"
                                        }`}
                                        onClick={() =>
                                            setActiveContactMethod(idx)
                                        }
                                    >
                                        {method.icon}
                                        <span className="ml-2">
                                            {method.name}
                                        </span>
                                    </button>
                                ))}
                            </div>
                        </div>

                        {/* Active Contact Method Details */}
                        <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl overflow-hidden border border-gray-200 dark:border-gray-700">
                            <div
                                className={`bg-gradient-to-r ${contactMethods[activeContactMethod].color} p-8 text-white`}
                            >
                                <div className="flex items-center justify-between">
                                    <div>
                                        <h4 className="text-2xl font-bold mb-2">
                                            {
                                                contactMethods[
                                                    activeContactMethod
                                                ].title
                                            }
                                        </h4>
                                        <p className="text-lg opacity-90 mb-4">
                                            {
                                                contactMethods[
                                                    activeContactMethod
                                                ].description
                                            }
                                        </p>
                                        <div className="flex items-center space-x-6">
                                            <div className="flex items-center">
                                                <Clock className="w-5 h-5 mr-2" />
                                                <span>
                                                    {
                                                        contactMethods[
                                                            activeContactMethod
                                                        ].availability
                                                    }
                                                </span>
                                            </div>
                                            <div className="flex items-center">
                                                <Zap className="w-5 h-5 mr-2" />
                                                <span>
                                                    Response:{" "}
                                                    {
                                                        contactMethods[
                                                            activeContactMethod
                                                        ].responseTime
                                                    }
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                    <div className="text-6xl opacity-20">
                                        {
                                            contactMethods[activeContactMethod]
                                                .icon
                                        }
                                    </div>
                                </div>
                            </div>

                            <div className="p-8">
                                <h5 className="text-xl font-bold text-gray-900 dark:text-white mb-4">
                                    What You Can Get Help With:
                                </h5>
                                <div className="grid md:grid-cols-2 gap-4">
                                    {contactMethods[
                                        activeContactMethod
                                    ].features.map((feature, index) => (
                                        <div
                                            key={index}
                                            className="flex items-center"
                                        >
                                            <CheckCircle className="w-5 h-5 text-green-500 mr-3 flex-shrink-0" />
                                            <span className="text-gray-600 dark:text-gray-300">
                                                {feature}
                                            </span>
                                        </div>
                                    ))}
                                </div>

                                <div className="mt-6 flex justify-center">
                                    <button
                                        className={`bg-gradient-to-r ${contactMethods[activeContactMethod].color} text-white px-8 py-3 rounded-lg font-bold hover:shadow-lg transition-all duration-300 transform hover:scale-105`}
                                        onClick={() =>
                                            handleCTAClick(
                                                `Start ${contactMethods[activeContactMethod].name}`,
                                                "contact_method"
                                            )
                                        }
                                    >
                                        Start{" "}
                                        {
                                            contactMethods[activeContactMethod]
                                                .name
                                        }
                                    </button>
                                </div>
                            </div>
                        </div>
                    </motion.div>

                    {/* Main Contact Form and Info Grid */}
                    <div className="grid lg:grid-cols-2 gap-12 mb-16">
                        {/* Contact Form */}
                        <motion.div
                            initial="hidden"
                            whileInView="visible"
                            viewport={{ once: true }}
                            variants={fadeIn}
                            className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl p-8 border border-gray-200 dark:border-gray-700"
                        >
                            <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-6">
                                Get a Free Quote for Academic Writing Services
                            </h3>
                            <p className="text-gray-600 dark:text-gray-300 mb-8">
                                Tell us about your project and get connected
                                with our{" "}
                                <strong>professional essay writers</strong>
                                for personalized{" "}
                                <strong>academic writing services</strong>.
                            </p>

                            <form onSubmit={handleSubmit} className="space-y-6">
                                <div className="grid md:grid-cols-2 gap-6">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Full Name *
                                        </label>
                                        <input
                                            type="text"
                                            name="name"
                                            value={formData.name}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                            placeholder="Your full name"
                                            required
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Email Address *
                                        </label>
                                        <input
                                            type="email"
                                            name="email"
                                            value={formData.email}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                            placeholder="your@email.com"
                                            required
                                        />
                                    </div>
                                </div>

                                <div className="grid md:grid-cols-2 gap-6">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Phone Number
                                        </label>
                                        <input
                                            type="tel"
                                            name="phone"
                                            value={formData.phone}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                            placeholder="+1 (555) 123-4567"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Academic Writing Service *
                                        </label>
                                        <select
                                            name="service"
                                            value={formData.service}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                            required
                                        >
                                            <option value="">
                                                Select a service
                                            </option>
                                            <option value="dissertation-writing-help">
                                                Dissertation Writing Help
                                            </option>
                                            <option value="research-paper-writing-service">
                                                Research Paper Writing Service
                                            </option>
                                            <option value="custom-academic-papers">
                                                Custom Academic Papers
                                            </option>
                                            <option value="essay-writing">
                                                Essay Writing
                                            </option>
                                            <option value="thesis-writing">
                                                Thesis Writing
                                            </option>
                                            <option value="case-study">
                                                Case Study Writing
                                            </option>
                                            <option value="literature-review">
                                                Literature Review
                                            </option>
                                            <option value="other">
                                                Other Academic Writing Services
                                            </option>
                                        </select>
                                    </div>
                                </div>

                                <div className="grid md:grid-cols-3 gap-6">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Academic Level
                                        </label>
                                        <select
                                            name="academicLevel"
                                            value={formData.academicLevel}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                        >
                                            <option value="">
                                                Select level
                                            </option>
                                            <option value="undergraduate">
                                                Undergraduate
                                            </option>
                                            <option value="masters">
                                                Master's
                                            </option>
                                            <option value="phd">
                                                PhD/Doctoral
                                            </option>
                                            <option value="postgraduate">
                                                Postgraduate
                                            </option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Number of Pages
                                        </label>
                                        <input
                                            type="number"
                                            name="pages"
                                            value={formData.pages}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                            placeholder="e.g., 10"
                                            min="1"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Urgency Level
                                        </label>
                                        <select
                                            name="urgency"
                                            value={formData.urgency}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                        >
                                            <option value="standard">
                                                Standard (7+ days)
                                            </option>
                                            <option value="urgent">
                                                Urgent (3-6 days)
                                            </option>
                                            <option value="rush">
                                                Rush (1-2 days)
                                            </option>
                                            <option value="emergency">
                                                Emergency (24 hours)
                                            </option>
                                        </select>
                                    </div>
                                </div>

                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Subject/Topic
                                    </label>
                                    <input
                                        type="text"
                                        name="subject"
                                        value={formData.subject}
                                        onChange={handleInputChange}
                                        className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                        placeholder="Brief description of your topic or subject area"
                                    />
                                </div>

                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Project Details & Requirements
                                    </label>
                                    <textarea
                                        name="message"
                                        value={formData.message}
                                        onChange={handleInputChange}
                                        rows="5"
                                        className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                        placeholder="Please provide detailed information about your academic writing project, including specific requirements, formatting style, number of sources, and any special instructions for our professional essay writers..."
                                    ></textarea>
                                </div>

                                <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4">
                                    <div className="flex items-start">
                                        <Shield className="w-5 h-5 text-blue-600 dark:text-blue-400 mt-1 mr-3 flex-shrink-0" />
                                        <div className="text-sm text-blue-800 dark:text-blue-300">
                                            <strong>Privacy Guarantee:</strong>{" "}
                                            All information is confidential and
                                            secure. Our academic writing
                                            services maintain strict privacy
                                            standards.
                                        </div>
                                    </div>
                                </div>

                                <button
                                    type="submit"
                                    disabled={isSubmitting}
                                    className="w-full bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 disabled:from-gray-400 disabled:to-gray-500 text-white py-4 px-6 rounded-lg font-bold transition-all duration-300 transform hover:scale-105 disabled:transform-none disabled:cursor-not-allowed flex items-center justify-center"
                                >
                                    {isSubmitting ? (
                                        <>
                                            <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-2"></div>
                                            Sending Request...
                                        </>
                                    ) : (
                                        <>
                                            <Send className="w-5 h-5 mr-2" />
                                            Get Free Quote & Connect with
                                            Professional Writers
                                        </>
                                    )}
                                </button>
                            </form>
                        </motion.div>

                        {/* Contact Information */}
                        <motion.div
                            initial="hidden"
                            whileInView="visible"
                            viewport={{ once: true }}
                            variants={fadeIn}
                            className="space-y-8"
                        >
                            {/* Direct Contact Methods */}
                            <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl p-8 border border-gray-200 dark:border-gray-700">
                                <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-6">
                                    Direct Contact for Academic Writing Services
                                </h3>

                                <div className="space-y-6">
                                    <div className="flex items-start">
                                        <div className="bg-blue-100 dark:bg-blue-900/30 rounded-lg p-3 mr-4">
                                            <Phone className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                                        </div>
                                        <div>
                                            <h4 className="font-bold text-gray-900 dark:text-white mb-1">
                                                Call Our Academic Writing
                                                Experts
                                            </h4>
                                            <p className="text-gray-600 dark:text-gray-300 mb-2">
                                                Speak directly with specialists
                                                about dissertation writing help
                                            </p>
                                            <div className="space-y-1">
                                                <p className="text-blue-600 dark:text-blue-400 font-medium">
                                                    +1 (555) 123-4567
                                                    (US/Canada)
                                                </p>
                                                <p className="text-blue-600 dark:text-blue-400 font-medium">
                                                    +44 20 7123 4567 (UK)
                                                </p>
                                                <p className="text-blue-600 dark:text-blue-400 font-medium">
                                                    +61 2 9123 4567 (Australia)
                                                </p>
                                            </div>
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                                                Available 24/7 for urgent
                                                academic writing services
                                            </p>
                                        </div>
                                    </div>

                                    <div className="flex items-start">
                                        <div className="bg-green-100 dark:bg-green-900/30 rounded-lg p-3 mr-4">
                                            <Mail className="w-6 h-6 text-green-600 dark:text-green-400" />
                                        </div>
                                        <div>
                                            <h4 className="font-bold text-gray-900 dark:text-white mb-1">
                                                Email Our Professional Essay
                                                Writers
                                            </h4>
                                            <p className="text-gray-600 dark:text-gray-300 mb-2">
                                                Detailed inquiries about
                                                research paper writing service
                                            </p>
                                            <div className="space-y-1">
                                                <p className="text-green-600 dark:text-green-400 font-medium">
                                                    support@academicscribe.com
                                                </p>
                                                <p className="text-green-600 dark:text-green-400 font-medium">
                                                    orders@academicscribe.com
                                                </p>
                                                <p className="text-green-600 dark:text-green-400 font-medium">
                                                    writers@academicscribe.com
                                                </p>
                                            </div>
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                                                Response within 2 hours for all
                                                academic writing services
                                            </p>
                                        </div>
                                    </div>

                                    <div className="flex items-start">
                                        <div className="bg-purple-100 dark:bg-purple-900/30 rounded-lg p-3 mr-4">
                                            <MessageSquare className="w-6 h-6 text-purple-600 dark:text-purple-400" />
                                        </div>
                                        <div>
                                            <h4 className="font-bold text-gray-900 dark:text-white mb-1">
                                                Live Chat with Academic
                                                Specialists
                                            </h4>
                                            <p className="text-gray-600 dark:text-gray-300 mb-2">
                                                Instant help with custom
                                                academic papers and quotes
                                            </p>
                                            <button className="bg-purple-600 hover:bg-purple-700 text-white px-4 py-2 rounded-lg font-medium transition-colors">
                                                Start Live Chat
                                            </button>
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                                                Average response time: 30
                                                seconds
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Service Guarantees */}
                            <div className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-2xl p-8 text-white">
                                <h3 className="text-2xl font-bold mb-6">
                                    Why Choose Our Academic Writing Services?
                                </h3>
                                <div className="grid md:grid-cols-2 gap-6">
                                    {[
                                        {
                                            icon: (
                                                <CheckCircle className="w-6 h-6" />
                                            ),
                                            title: "100% Original Work",
                                            description:
                                                "All custom academic papers are written from scratch by professional essay writers",
                                        },
                                        {
                                            icon: (
                                                <Shield className="w-6 h-6" />
                                            ),
                                            title: "Complete Confidentiality",
                                            description:
                                                "Your privacy is guaranteed with our secure academic writing services",
                                        },
                                        {
                                            icon: <Award className="w-6 h-6" />,
                                            title: "PhD-Level Writers",
                                            description:
                                                "Expert dissertation writing help from qualified academic professionals",
                                        },
                                        {
                                            icon: <Clock className="w-6 h-6" />,
                                            title: "24/7 Support",
                                            description:
                                                "Round-the-clock assistance for all research paper writing service needs",
                                        },
                                    ].map((guarantee, index) => (
                                        <div
                                            key={index}
                                            className="flex items-start"
                                        >
                                            <div className="bg-white/20 rounded-lg p-2 mr-3 flex-shrink-0">
                                                {guarantee.icon}
                                            </div>
                                            <div>
                                                <h4 className="font-bold mb-1">
                                                    {guarantee.title}
                                                </h4>
                                                <p className="text-blue-100 text-sm">
                                                    {guarantee.description}
                                                </p>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </motion.div>
                    </div>

                    {/* Global Office Locations */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white text-center mb-12">
                            Global Academic Writing Services Offices
                        </h3>
                        <div className="grid md:grid-cols-3 gap-8">
                            {officeLocations.map((office, index) => (
                                <div
                                    key={index}
                                    className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 border border-gray-200 dark:border-gray-700"
                                >
                                    <div className="flex items-center justify-between mb-4">
                                        <h4 className="text-xl font-bold text-gray-900 dark:text-white">
                                            {office.city}
                                        </h4>
                                        <div className="bg-blue-100 dark:bg-blue-900/30 rounded-lg p-2">
                                            <MapPin className="w-5 h-5 text-blue-600 dark:text-blue-400" />
                                        </div>
                                    </div>

                                    <div className="space-y-3 mb-6">
                                        <div className="flex items-start">
                                            <MapPin className="w-4 h-4 text-gray-400 mt-1 mr-2 flex-shrink-0" />
                                            <span className="text-gray-600 dark:text-gray-300 text-sm">
                                                {office.address}
                                            </span>
                                        </div>
                                        <div className="flex items-center">
                                            <Phone className="w-4 h-4 text-gray-400 mr-2 flex-shrink-0" />
                                            <span className="text-gray-600 dark:text-gray-300 text-sm">
                                                {office.phone}
                                            </span>
                                        </div>
                                        <div className="flex items-center">
                                            <Mail className="w-4 h-4 text-gray-400 mr-2 flex-shrink-0" />
                                            <span className="text-gray-600 dark:text-gray-300 text-sm">
                                                {office.email}
                                            </span>
                                        </div>
                                        <div className="flex items-center">
                                            <Clock className="w-4 h-4 text-gray-400 mr-2 flex-shrink-0" />
                                            <span className="text-gray-600 dark:text-gray-300 text-sm">
                                                Timezone: {office.timezone}
                                            </span>
                                        </div>
                                    </div>

                                    <div>
                                        <h5 className="font-semibold text-gray-900 dark:text-white mb-2">
                                            Specializations:
                                        </h5>
                                        <div className="flex flex-wrap gap-2">
                                            {office.specialties.map(
                                                (specialty, specIndex) => (
                                                    <span
                                                        key={specIndex}
                                                        className="bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300 px-2 py-1 rounded-full text-xs font-medium"
                                                    >
                                                        {specialty}
                                                    </span>
                                                )
                                            )}
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </motion.div>

                    {/* FAQ Quick Access */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl p-8 border border-gray-200 dark:border-gray-700">
                            <h3 className="text-2xl font-bold text-gray-900 dark:text-white text-center mb-8">
                                Quick Answers About Academic Writing Services
                            </h3>
                            <div className="grid md:grid-cols-2 gap-6">
                                {[
                                    {
                                        question:
                                            "How quickly can I get dissertation writing help?",
                                        answer: "Our professional essay writers can start immediately. Rush orders completed in 24-48 hours.",
                                        icon: (
                                            <Clock className="w-5 h-5 text-blue-600" />
                                        ),
                                    },
                                    {
                                        question:
                                            "Are your research paper writing services confidential?",
                                        answer: "100% confidential. We never share client information or custom academic papers with third parties.",
                                        icon: (
                                            <Shield className="w-5 h-5 text-green-600" />
                                        ),
                                    },
                                    {
                                        question:
                                            "What qualifications do your writers have?",
                                        answer: "All professional essay writers hold PhD degrees in their respective fields with proven academic experience.",
                                        icon: (
                                            <GraduationCap className="w-5 h-5 text-purple-600" />
                                        ),
                                    },
                                    {
                                        question:
                                            "Do you offer revisions for academic writing services?",
                                        answer: "Yes, unlimited free revisions within 14 days to ensure your complete satisfaction with custom academic papers.",
                                        icon: (
                                            <Award className="w-5 h-5 text-orange-600" />
                                        ),
                                    },
                                ].map((faq, index) => (
                                    <div
                                        key={index}
                                        className="border border-gray-200 dark:border-gray-700 rounded-lg p-4"
                                    >
                                        <div className="flex items-start mb-2">
                                            {faq.icon}
                                            <h4 className="font-semibold text-gray-900 dark:text-white ml-2">
                                                {faq.question}
                                            </h4>
                                        </div>
                                        <p className="text-gray-600 dark:text-gray-300 text-sm ml-7">
                                            {faq.answer}
                                        </p>
                                    </div>
                                ))}
                            </div>
                            <div className="text-center mt-6">
                                <Link
                                    href="/faq"
                                    className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 font-medium"
                                    onClick={() =>
                                        handleCTAClick(
                                            "View All FAQs",
                                            "faq_link"
                                        )
                                    }
                                >
                                    View All Frequently Asked Questions →
                                </Link>
                            </div>
                        </div>
                    </motion.div>

                    {/* Emergency Contact Section */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <div className="bg-gradient-to-r from-red-500 to-pink-600 rounded-2xl p-8 text-white">
                            <div className="text-center">
                                <div className="bg-white/20 rounded-full p-4 w-16 h-16 flex items-center justify-center mx-auto mb-4">
                                    <Zap className="w-8 h-8" />
                                </div>
                                <h3 className="text-3xl font-bold mb-4">
                                    Need Urgent Academic Writing Help?
                                </h3>
                                <p className="text-xl mb-6 opacity-90">
                                    Emergency dissertation writing help and rush
                                    research paper writing service available
                                    24/7
                                </p>
                                <div className="flex flex-col sm:flex-row justify-center space-y-4 sm:space-y-0 sm:space-x-6">
                                    <button
                                        className="bg-white text-red-600 px-8 py-3 rounded-lg font-bold hover:bg-red-50 transition-colors"
                                        onClick={() =>
                                            handleCTAClick(
                                                "Emergency Call",
                                                "emergency_contact"
                                            )
                                        }
                                    >
                                        📞 Call Now: +1 (555) 911-HELP
                                    </button>
                                    <button
                                        className="border-2 border-white text-white hover:bg-white hover:text-red-600 px-8 py-3 rounded-lg font-bold transition-colors"
                                        onClick={() =>
                                            handleCTAClick(
                                                "Emergency Chat",
                                                "emergency_contact"
                                            )
                                        }
                                    >
                                        💬 Emergency Live Chat
                                    </button>
                                </div>
                                <p className="text-sm mt-4 opacity-75">
                                    Professional essay writers available for
                                    same-day custom academic papers
                                </p>
                            </div>
                        </div>
                    </motion.div>

                    {/* Social Media & Follow Section */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center"
                    >
                        <div className="bg-white dark:bg-gray-800 rounded-2xl p-8 shadow-xl border border-gray-200 dark:border-gray-700">
                            <h3 className="text-2xl font-bold text-gray-900 dark:text-white mb-6">
                                Stay Connected with Our Academic Writing
                                Community
                            </h3>
                            <p className="text-gray-600 dark:text-gray-300 mb-8">
                                Follow us for academic writing tips, success
                                stories, and updates about our
                                <strong>
                                    {" "}
                                    professional essay writers
                                </strong> and{" "}
                                <strong>academic writing services</strong>
                            </p>

                            <div className="flex justify-center space-x-6 mb-8">
                                {[
                                    {
                                        name: "Facebook",
                                        icon: (
                                            <svg
                                                className="w-6 h-6"
                                                fill="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z" />
                                            </svg>
                                        ),
                                        color: "text-blue-600 hover:text-blue-800",
                                        followers: "25K+",
                                    },
                                    {
                                        name: "Twitter",
                                        icon: (
                                            <svg
                                                className="w-6 h-6"
                                                fill="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path d="M23.953 4.57a10 10 0 01-2.825.775 4.958 4.958 0 002.163-2.723c-.951.555-2.005.959-3.127 1.184a4.92 4.92 0 00-8.384 4.482C7.69 8.095 4.067 6.13 1.64 3.162a4.822 4.822 0 00-.666 2.475c0 1.71.87 3.213 2.188 4.096a4.904 4.904 0 01-2.228-.616v.06a4.923 4.923 0 003.946 4.827 4.996 4.996 0 01-2.212.085 4.936 4.936 0 004.604 3.417 9.867 9.867 0 01-6.102 2.105c-.39 0-.779-.023-1.17-.067a13.995 13.995 0 007.557 2.209c9.053 0 13.998-7.496 13.998-13.985 0-.21 0-.42-.015-.63A9.935 9.935 0 0024 4.59z" />
                                            </svg>
                                        ),
                                        color: "text-blue-400 hover:text-blue-600",
                                        followers: "18K+",
                                    },
                                    {
                                        name: "LinkedIn",
                                        icon: (
                                            <svg
                                                className="w-6 h-6"
                                                fill="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433c-1.144 0-2.063-.926-2.063-2.065 0-1.138.92-2.063 2.063-2.063 1.14 0 2.064.925 2.064 2.063 0 1.139-.925 2.065-2.064 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z" />
                                            </svg>
                                        ),
                                        color: "text-blue-700 hover:text-blue-900",
                                        followers: "12K+",
                                    },
                                    {
                                        name: "Instagram",
                                        icon: (
                                            <svg
                                                className="w-6 h-6"
                                                fill="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path d="M12.017 0C5.396 0 .029 5.367.029 11.987c0 6.618 5.367 11.986 11.988 11.986s11.987-5.368 11.987-11.986C24.014 5.367 18.635.001 12.017.001zM8.449 16.988c-1.297 0-2.448-.49-3.323-1.297C4.198 14.926 3.708 13.775 3.708 12.478s.49-2.448 1.418-3.323c.875-.807 2.026-1.297 3.323-1.297s2.448.49 3.323 1.297c.928.875 1.418 2.026 1.418 3.323s-.49 2.448-1.418 3.213c-.875.807-2.026 1.297-3.323 1.297zm7.83-9.608c-.807 0-1.418-.612-1.418-1.418s.612-1.418 1.418-1.418 1.418.612 1.418 1.418-.611 1.418-1.418 1.418zm3.323 9.608c-1.297 0-2.448-.49-3.323-1.297-.928-.765-1.418-1.916-1.418-3.213s.49-2.448 1.418-3.323c.875-.807 2.026-1.297 3.323-1.297s2.448.49 3.323 1.297c.928.875 1.418 2.026 1.418 3.323s-.49 2.448-1.418 3.213c-.875.807-2.026 1.297-3.323 1.297z" />
                                            </svg>
                                        ),
                                        color: "text-pink-600 hover:text-pink-800",
                                        followers: "8K+",
                                    },
                                    {
                                        name: "YouTube",
                                        icon: (
                                            <svg
                                                className="w-6 h-6"
                                                fill="currentColor"
                                                viewBox="0 0 24 24"
                                            >
                                                <path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z" />
                                            </svg>
                                        ),
                                        color: "text-red-600 hover:text-red-800",
                                        followers: "15K+",
                                    },
                                ].map((social, index) => (
                                    <div key={index} className="text-center">
                                        <Link
                                            href={`/${social.name.toLowerCase()}`}
                                            className={`${social.color} transition-colors duration-200 flex flex-col items-center`}
                                            onClick={() =>
                                                handleCTAClick(
                                                    `Follow ${social.name}`,
                                                    "social_media"
                                                )
                                            }
                                        >
                                            <div className="bg-gray-100 dark:bg-gray-700 rounded-full p-3 mb-2 hover:scale-110 transition-transform duration-200">
                                                {social.icon}
                                            </div>
                                            <span className="text-sm font-medium text-gray-900 dark:text-white">
                                                {social.name}
                                            </span>
                                            <span className="text-xs text-gray-500 dark:text-gray-400">
                                                {social.followers}
                                            </span>
                                        </Link>
                                    </div>
                                ))}
                            </div>

                            <div className="bg-gradient-to-r from-blue-50 to-purple-50 dark:from-blue-900/20 dark:to-purple-900/20 rounded-lg p-6">
                                <h4 className="text-lg font-bold text-gray-900 dark:text-white mb-3">
                                    Join Our Academic Writing Community
                                </h4>
                                <p className="text-gray-600 dark:text-gray-300 mb-4">
                                    Get exclusive tips, writing guides, and
                                    special offers for academic writing services
                                </p>
                                <div className="flex flex-col sm:flex-row gap-3 max-w-md mx-auto">
                                    <input
                                        type="email"
                                        placeholder="Enter your email"
                                        className="flex-1 px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                    />
                                    <button
                                        className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium transition-colors"
                                        onClick={() =>
                                            handleCTAClick(
                                                "Subscribe Newsletter",
                                                "newsletter"
                                            )
                                        }
                                    >
                                        Subscribe
                                    </button>
                                </div>
                            </div>
                        </div>
                    </motion.div>
                </div>
            </section>

            {/* Additional Contact Resources */}
            <section className="py-16 bg-white dark:bg-gray-800">
                <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center mb-12"
                    >
                        <h3 className="text-3xl font-bold text-gray-900 dark:text-white mb-6">
                            Additional Resources for Academic Writing Services
                        </h3>
                        <p className="text-lg text-gray-600 dark:text-gray-300">
                            Explore more ways to connect with our professional
                            essay writers and academic writing services
                        </p>
                    </motion.div>

                    <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
                        {[
                            {
                                title: "Writing Samples",
                                description:
                                    "View examples of dissertation writing help and research paper writing service",
                                icon: <FileText className="w-8 h-8" />,
                                link: "/samples",
                                color: "from-blue-500 to-cyan-500",
                                cta: "View Samples",
                            },
                            {
                                title: "Writer Profiles",
                                description:
                                    "Meet our professional essay writers and their academic qualifications",
                                icon: <Users className="w-8 h-8" />,
                                link: "/writers",
                                color: "from-green-500 to-emerald-500",
                                cta: "Meet Writers",
                            },
                            {
                                title: "Pricing Calculator",
                                description:
                                    "Get instant quotes for custom academic papers and academic writing services",
                                icon: <Target className="w-8 h-8" />,
                                link: "/pricing",
                                color: "from-purple-500 to-indigo-500",
                                cta: "Calculate Price",
                            },
                            {
                                title: "Success Stories",
                                description:
                                    "Read testimonials about our dissertation writing help and research services",
                                icon: <TrendingUp className="w-8 h-8" />,
                                link: "/testimonials",
                                color: "from-orange-500 to-red-500",
                                cta: "Read Stories",
                            },
                        ].map((resource, index) => (
                            <motion.div
                                key={index}
                                variants={fadeIn}
                                className="bg-gray-50 dark:bg-gray-700 rounded-xl p-6 hover:shadow-lg transition-all duration-300 transform hover:-translate-y-1"
                            >
                                <div
                                    className={`bg-gradient-to-r ${resource.color} p-4 rounded-lg mb-4 text-white`}
                                >
                                    {resource.icon}
                                </div>
                                <h4 className="text-lg font-bold text-gray-900 dark:text-white mb-2">
                                    {resource.title}
                                </h4>
                                <p className="text-gray-600 dark:text-gray-300 mb-4 text-sm">
                                    {resource.description}
                                </p>
                                <Link
                                    href={resource.link}
                                    className={`inline-flex items-center text-sm font-medium bg-gradient-to-r ${resource.color} text-white px-4 py-2 rounded-lg hover:shadow-md transition-all duration-200`}
                                    onClick={() =>
                                        handleCTAClick(
                                            resource.cta,
                                            "resource_link"
                                        )
                                    }
                                >
                                    {resource.cta}
                                    <svg
                                        className="w-4 h-4 ml-1"
                                        fill="none"
                                        stroke="currentColor"
                                        viewBox="0 0 24 24"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M9 5l7 7-7 7"
                                        />
                                    </svg>
                                </Link>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Final CTA Section */}
            <section className="py-16 bg-gradient-to-r from-blue-600 to-purple-600">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                    >
                        <h3 className="text-3xl md:text-4xl font-bold text-white mb-6">
                            Ready to Start Your Academic Success Journey?
                        </h3>
                        <p className="text-xl text-blue-100 mb-8 leading-relaxed">
                            Connect with our{" "}
                            <strong>professional essay writers</strong> today
                            for expert
                            <strong> dissertation writing help</strong>,{" "}
                            <strong>research paper writing service</strong>, and{" "}
                            <strong>custom academic papers</strong>. Our{" "}
                            <strong>academic writing services</strong>
                            are designed to help you achieve excellence.
                        </p>

                        <div className="flex flex-col sm:flex-row justify-center space-y-4 sm:space-y-0 sm:space-x-6 mb-8">
                            <Link
                                href="/order"
                                className="bg-white text-blue-600 hover:bg-blue-50 px-8 py-4 rounded-lg font-bold transition-all duration-300 transform hover:scale-105 shadow-lg"
                                onClick={() =>
                                    handleCTAClick(
                                        "Start Your Order",
                                        "final_cta"
                                    )
                                }
                            >
                                🚀 Start Your Order Now
                            </Link>
                            <button
                                className="border-2 border-white text-white hover:bg-white hover:text-blue-600 px-8 py-4 rounded-lg font-bold transition-all duration-300 transform hover:scale-105"
                                onClick={() =>
                                    handleCTAClick(
                                        "Get Free Consultation",
                                        "final_cta"
                                    )
                                }
                            >
                                💬 Get Free Consultation
                            </button>
                        </div>

                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm text-blue-100">
                            <div className="flex items-center justify-center">
                                <CheckCircle className="w-4 h-4 mr-2" />
                                <span>24/7 Support</span>
                            </div>
                            <div className="flex items-center justify-center">
                                <CheckCircle className="w-4 h-4 mr-2" />
                                <span>Money-Back Guarantee</span>
                            </div>
                            <div className="flex items-center justify-center">
                                <CheckCircle className="w-4 h-4 mr-2" />
                                <span>100% Confidential</span>
                            </div>
                            <div className="flex items-center justify-center">
                                <CheckCircle className="w-4 h-4 mr-2" />
                                <span>Plagiarism-Free</span>
                            </div>
                        </div>
                    </motion.div>
                </div>
            </section>
        </>
    );
}
