import React from "react";
import {
    LineChart,
    Line,
    BarChart,
    Bar,
    PieChart,
    Pie,
    AreaChart,
    Area,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    Legend,
    ResponsiveContainer,
    Cell,
} from "recharts";

// Original DashboardCharts component - keep this functionality
const DashboardCharts = ({ chartData, clientsOrders }) => {
    const COLORS = ["#0088FE", "#00C49F", "#FFBB28", "#FF8042"];

    return (
        <div className="w-full">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {/* Line Chart */}
                <div className="bg-gradient-to-b from-blue-100 to-white dark:bg-none dark:bg-gray-800 p-4 rounded-lg shadow-md">
                    <h3 className="text-lg font-medium text-gray-700 dark:text-gray-300 mb-4">
                        Monthly Orders & Revenue
                    </h3>
                    <div className="h-64">
                        <ResponsiveContainer width="100%" height="100%">
                            <LineChart
                                data={chartData}
                                margin={{
                                    top: 5,
                                    right: 30,
                                    left: 20,
                                    bottom: 5,
                                }}
                            >
                                <CartesianGrid
                                    strokeDasharray="3 3"
                                    stroke="#374151"
                                />
                                <XAxis dataKey="name" stroke="#9CA3AF" />
                                <YAxis stroke="#9CA3AF" />
                                <Tooltip
                                    contentStyle={{
                                        backgroundColor: "#1F2937",
                                        color: "#F9FAFB",
                                        borderColor: "#374151",
                                    }}
                                />
                                <Legend />
                                <Line
                                    type="monotone"
                                    dataKey="orders"
                                    stroke="#3B82F6"
                                    activeDot={{ r: 8 }}
                                />
                                <Line
                                    type="monotone"
                                    dataKey="revenue"
                                    stroke="#10B981"
                                />
                            </LineChart>
                        </ResponsiveContainer>
                    </div>
                </div>

                {/* Bar Chart */}
                <div className="bg-gradient-to-b from-blue-100 to-white dark:bg-none dark:bg-gray-800 p-4 rounded-lg shadow-md">
                    <h3 className="text-lg font-medium text-gray-700 dark:text-gray-300 mb-4">
                        Client Order Count
                    </h3>
                    <div className="h-64">
                        <ResponsiveContainer width="100%" height="100%">
                            <BarChart
                                data={clientsOrders}
                                margin={{
                                    top: 5,
                                    right: 30,
                                    left: 20,
                                    bottom: 5,
                                }}
                            >
                                <CartesianGrid
                                    strokeDasharray="3 3"
                                    stroke="#374151"
                                />
                                <XAxis dataKey="name" stroke="#9CA3AF" />
                                <YAxis stroke="#9CA3AF" />
                                <Tooltip
                                    contentStyle={{
                                        backgroundColor: "#1F2937",
                                        color: "#F9FAFB",
                                        borderColor: "#374151",
                                    }}
                                />
                                <Legend />
                                <Bar dataKey="order_count" fill="#8884d8" />
                            </BarChart>
                        </ResponsiveContainer>
                    </div>
                </div>

                {/* Area Chart */}
                <div className="bg-gradient-to-b from-blue-100 to-white dark:bg-none dark:bg-gray-800 p-4 rounded-lg shadow-md">
                    <h3 className="text-lg font-medium text-gray-700 dark:text-gray-300 mb-4">
                        Growth Trends
                    </h3>
                    <div className="h-64">
                        <ResponsiveContainer width="100%" height="100%">
                            <AreaChart
                                data={chartData}
                                margin={{
                                    top: 10,
                                    right: 30,
                                    left: 0,
                                    bottom: 0,
                                }}
                            >
                                <CartesianGrid
                                    strokeDasharray="3 3"
                                    stroke="#374151"
                                />
                                <XAxis dataKey="name" stroke="#9CA3AF" />
                                <YAxis stroke="#9CA3AF" />
                                <Tooltip
                                    contentStyle={{
                                        backgroundColor: "#1F2937",
                                        color: "#F9FAFB",
                                        borderColor: "#374151",
                                    }}
                                />
                                <Area
                                    type="monotone"
                                    dataKey="revenue"
                                    stroke="#8884d8"
                                    fill="#8884d8"
                                    fillOpacity={0.3}
                                />
                                <Area
                                    type="monotone"
                                    dataKey="orders"
                                    stroke="#82ca9d"
                                    fill="#82ca9d"
                                    fillOpacity={0.3}
                                />
                            </AreaChart>
                        </ResponsiveContainer>
                    </div>
                </div>

                {/* Pie Chart */}
                <div className="bg-gradient-to-b from-blue-100 to-white dark:bg-none dark:bg-gray-800 p-4 rounded-lg shadow-md">
                    <h3 className="text-lg font-medium text-gray-700 dark:text-gray-300 mb-4">
                        Revenue Distribution
                    </h3>
                    <div className="h-64">
                        <ResponsiveContainer width="100%" height="100%">
                            <PieChart>
                                <Pie
                                    data={chartData}
                                    cx="50%"
                                    cy="50%"
                                    labelLine={false}
                                    outerRadius={80}
                                    fill="#8884d8"
                                    dataKey="revenue"
                                    label={({ name, percent }) =>
                                        `${name}: ${
                                            typeof percent === "number"
                                                ? (percent * 100).toFixed(0)
                                                : 0
                                        }%`
                                    }
                                >
                                    {chartData.map((entry, index) => (
                                        <Cell
                                            key={`cell-${index}`}
                                            fill={COLORS[index % COLORS.length]}
                                        />
                                    ))}
                                </Pie>
                                <Tooltip
                                    contentStyle={{
                                        backgroundColor: "#1F2937",
                                        color: "#F9FAFB",
                                        borderColor: "#374151",
                                    }}
                                    formatter={(value, name) => {
                                        // Check if value is a number before calling toFixed
                                        const formattedValue =
                                            typeof value === "number"
                                                ? value.toFixed(2)
                                                : value;
                                        return [formattedValue, name];
                                    }}
                                />
                            </PieChart>
                        </ResponsiveContainer>
                    </div>
                </div>
            </div>
        </div>
    );
};

// New subcomponents to be exported
const MonthlyRevenueChart = ({ data }) => {
    return (
        <ResponsiveContainer width="100%" height="100%">
            <LineChart
                data={data}
                margin={{
                    top: 5,
                    right: 30,
                    left: 20,
                    bottom: 5,
                }}
            >
                <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                <XAxis dataKey="name" stroke="#9CA3AF" />
                <YAxis yAxisId="left" stroke="#10B981" />
                <YAxis yAxisId="right" orientation="right" stroke="#3B82F6" />
                <Tooltip
                    contentStyle={{
                        backgroundColor: "#1F2937",
                        color: "#F9FAFB",
                        borderColor: "#374151",
                    }}
                />
                <Legend />
                <Line
                    yAxisId="left"
                    type="monotone"
                    dataKey="revenue"
                    name="Revenue"
                    stroke="#10B981"
                    strokeWidth={2}
                    dot={{ r: 4 }}
                    activeDot={{ r: 8 }}
                />
                <Line
                    yAxisId="right"
                    type="monotone"
                    dataKey="orders"
                    name="Orders"
                    stroke="#3B82F6"
                    strokeWidth={2}
                    dot={{ r: 4 }}
                />
                {/* Only show refund line if the data includes refund_amount */}
                {data && data[0] && data[0].refund_amount !== undefined && (
                    <Line
                        yAxisId="left"
                        type="monotone"
                        dataKey="refund_amount"
                        name="Refunds"
                        stroke="#EF4444"
                        strokeWidth={2}
                        strokeDasharray="5 5"
                        dot={{ r: 4 }}
                    />
                )}
            </LineChart>
        </ResponsiveContainer>
    );
};

const RefundTrendsChart = ({ data }) => {
    // Filter out months with no refunds for better visualization
    const refundData = data.map((item) => ({
        name: item.name,
        refund_amount: item.refund_amount || 0,
        refund_count: item.refund_count || 0,
    }));

    return (
        <ResponsiveContainer width="100%" height="100%">
            <AreaChart
                data={refundData}
                margin={{
                    top: 10,
                    right: 30,
                    left: 0,
                    bottom: 0,
                }}
            >
                <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                <XAxis dataKey="name" stroke="#9CA3AF" />
                <YAxis yAxisId="left" stroke="#EF4444" />
                <YAxis yAxisId="right" orientation="right" stroke="#F59E0B" />
                <Tooltip
                    contentStyle={{
                        backgroundColor: "#1F2937",
                        color: "#F9FAFB",
                        borderColor: "#374151",
                    }}
                />
                <Legend />
                <Area
                    yAxisId="left"
                    type="monotone"
                    dataKey="refund_amount"
                    name="Refund Amount"
                    stroke="#EF4444"
                    fill="#EF4444"
                    fillOpacity={0.3}
                />
                <Area
                    yAxisId="right"
                    type="monotone"
                    dataKey="refund_count"
                    name="Refund Count"
                    stroke="#F59E0B"
                    fill="#F59E0B"
                    fillOpacity={0.3}
                />
            </AreaChart>
        </ResponsiveContainer>
    );
};

const RevenueComparisonChart = ({
    data,
    grossRevenue,
    netRevenue,
    refundedAmount,
}) => {
    // Process data to ensure we have gross_revenue and net_revenue
    const processedData = data.map((item) => ({
        name: item.name,
        gross_revenue: item.gross_revenue || item.revenue || 0,
        refunded: item.refund_amount || 0,
        net_revenue:
            item.net_revenue || item.revenue - (item.refund_amount || 0) || 0,
    }));

    return (
        <ResponsiveContainer width="100%" height="100%">
            <BarChart
                data={processedData}
                margin={{
                    top: 5,
                    right: 30,
                    left: 20,
                    bottom: 5,
                }}
            >
                <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                <XAxis dataKey="name" stroke="#9CA3AF" />
                <YAxis stroke="#9CA3AF" />
                <Tooltip
                    contentStyle={{
                        backgroundColor: "#1F2937",
                        color: "#F9FAFB",
                        borderColor: "#374151",
                    }}
                    formatter={(value, name) => {
                        // Check if value is a number before calling toFixed
                        const formattedValue =
                            typeof value === "number"
                                ? value.toFixed(2)
                                : value || "0";
                        return [`${formattedValue}`, name];
                    }}
                />
                <Legend />
                <Bar
                    dataKey="gross_revenue"
                    name="Gross Revenue"
                    fill="#10B981"
                />
                <Bar dataKey="refunded" name="Refunded" fill="#EF4444" />
                <Bar dataKey="net_revenue" name="Net Revenue" fill="#3B82F6" />
            </BarChart>
        </ResponsiveContainer>
    );
};

const ClientsOrdersChart = ({ data }) => {
    return (
        <ResponsiveContainer width="100%" height="100%">
            <BarChart
                data={data}
                margin={{
                    top: 5,
                    right: 30,
                    left: 20,
                    bottom: 5,
                }}
            >
                <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                <XAxis dataKey="name" stroke="#9CA3AF" />
                <YAxis stroke="#9CA3AF" />
                <Tooltip
                    contentStyle={{
                        backgroundColor: "#1F2937",
                        color: "#F9FAFB",
                        borderColor: "#374151",
                    }}
                />
                <Legend />
                <Bar dataKey="order_count" name="Order Count" fill="#8884d8" />
            </BarChart>
        </ResponsiveContainer>
    );
};

// Attach subcomponents to the main component
DashboardCharts.MonthlyRevenueChart = MonthlyRevenueChart;
DashboardCharts.RefundTrendsChart = RefundTrendsChart;
DashboardCharts.RevenueComparisonChart = RevenueComparisonChart;
DashboardCharts.ClientsOrdersChart = ClientsOrdersChart;

export default DashboardCharts;
