import { useState, useEffect } from "react";

export default function DeadlineCountdown({ deadline }) {
    const [timeDisplay, setTimeDisplay] = useState("");
    const [isExpired, setIsExpired] = useState(false);

    useEffect(() => {
        const calculateTime = () => {
            const now = new Date();
            const deadlineDate = new Date(deadline);
            const diff = deadlineDate - now;

            if (diff > 0) {
                // Future deadline - show countdown
                const days = Math.floor(diff / (1000 * 60 * 60 * 24));
                const hours = Math.floor(
                    (diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)
                );
                const minutes = Math.floor(
                    (diff % (1000 * 60 * 60)) / (1000 * 60)
                );
                const seconds = Math.floor((diff % (1000 * 60)) / 1000);

                setIsExpired(false);
                setTimeDisplay(`${days}d ${hours}h ${minutes}m ${seconds}s`);
            } else {
                // Past deadline - show time elapsed
                const elapsed = Math.abs(diff);
                const minutes = Math.floor(elapsed / (1000 * 60));
                const hours = Math.floor(elapsed / (1000 * 60 * 60));
                const days = Math.floor(elapsed / (1000 * 60 * 60 * 24));
                const months = Math.floor(days / 30);
                const years = Math.floor(days / 365);

                setIsExpired(true);

                if (years > 0) {
                    setTimeDisplay(`${years} year${years > 1 ? "s" : ""} ago`);
                } else if (months > 0) {
                    setTimeDisplay(
                        `${months} month${months > 1 ? "s" : ""} ago`
                    );
                } else if (days > 0) {
                    setTimeDisplay(`${days} day${days > 1 ? "s" : ""} ago`);
                } else if (hours > 0) {
                    setTimeDisplay(`${hours} hour${hours > 1 ? "s" : ""} ago`);
                } else {
                    setTimeDisplay(
                        `${minutes} minute${minutes > 1 ? "s" : ""} ago`
                    );
                }
            }
        };

        calculateTime();
        const timer = setInterval(calculateTime, 1000);
        return () => clearInterval(timer);
    }, [deadline]);

    return (
        <div className="text-center rounded-lg p-4 bg-gray-100 dark:bg-gray-800 shadow-md">
            <h3
                className={`text-xl font-bold mb-2 ${
                    isExpired
                        ? "text-red-600 dark:text-red-400"
                        : "text-gray-900 dark:text-gray-100"
                }`}
            >
                {isExpired ? "Expired" : "Time Left"}
            </h3>
            <span
                className={`font-mono text-lg ${
                    isExpired
                        ? "text-red-500 dark:text-red-300"
                        : "text-gray-700 dark:text-gray-300"
                }`}
            >
                {timeDisplay}
            </span>
        </div>
    );
}
