import { Transition } from "@headlessui/react";
import { useContext, createContext, useState } from "react";

const DropdownMenuContext = createContext();

const DropdownMenu = ({ children }) => {
    const [open, setOpen] = useState(false);

    const toggleOpen = () => {
        setOpen((previousState) => !previousState);
    };

    return (
        <DropdownMenuContext.Provider value={{ open, setOpen, toggleOpen }}>
            <div className="relative">{children}</div>
        </DropdownMenuContext.Provider>
    );
};

// Rename Trigger to DropdownMenuTrigger
const DropdownMenuTrigger = ({ children }) => {
    const { open, setOpen, toggleOpen } = useContext(DropdownMenuContext);

    return (
        <>
            <div onClick={toggleOpen}>{children}</div>

            {open && (
                <div
                    className="fixed inset-0 z-40"
                    onClick={() => setOpen(false)}
                ></div>
            )}
        </>
    );
};

// Rename Content to DropdownMenuContent
const DropdownMenuContent = ({
    align = "right",
    width = "48",
    contentClasses = "py-1 bg-white dark:bg-gray-800 dark:text-white",
    children,
}) => {
    const { open, setOpen } = useContext(DropdownMenuContext);

    let alignmentClasses = "origin-top";

    if (align === "left") {
        alignmentClasses = "ltr:origin-top-left rtl:origin-top-right start-0";
    } else if (align === "right") {
        alignmentClasses = "ltr:origin-top-right rtl:origin-top-left end-0";
    }

    let widthClasses = "";

    if (width === "48") {
        widthClasses = "w-48";
    }

    return (
        <>
            <Transition
                show={open}
                enter="transition ease-out duration-200"
                enterFrom="opacity-0 scale-95"
                enterTo="opacity-100 scale-100"
                leave="transition ease-in duration-75"
                leaveFrom="opacity-100 scale-100"
                leaveTo="opacity-0 scale-95"
            >
                <div
                    className={`absolute z-50 mt-2 rounded-md shadow-lg ${alignmentClasses} ${widthClasses}`}
                    onClick={() => setOpen(false)}
                >
                    <div
                        className={
                            `rounded-md ring-1 ring-black ring-opacity-5 dark:ring-gray-700 transition-all ` +
                            contentClasses
                        }
                    >
                        {children}
                    </div>
                </div>
            </Transition>
        </>
    );
};

// Rename Item to DropdownMenuItem
const DropdownMenuItem = ({ className = "", children, ...props }) => {
    return (
        <div
            {...props}
            className={
                "block w-full px-4 py-2 text-left text-sm leading-5 transition duration-150 ease-in-out " +
                "text-gray-700 hover:bg-gray-100 focus:bg-gray-100 dark:text-gray-200 dark:hover:bg-gray-700 dark:focus:bg-gray-700 focus:outline-none " +
                className
            }
        >
            {children}
        </div>
    );
};

// Add the missing DropdownMenuSeparator component
const DropdownMenuSeparator = ({ className = "", ...props }) => {
    return (
        <div
            className={`h-0 my-1 border-t border-gray-200 dark:border-gray-700 ${className}`}
            {...props}
        />
    );
};

// Add the older naming as well for backward compatibility
DropdownMenu.Trigger = DropdownMenuTrigger;
DropdownMenu.Content = DropdownMenuContent;
DropdownMenu.Item = DropdownMenuItem;
DropdownMenu.Separator = DropdownMenuSeparator;

// Export the components with both naming patterns
export {
    DropdownMenu,
    DropdownMenuTrigger,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
};
