import { motion } from "framer-motion";
import { useState } from "react";
import { Link } from "@inertiajs/react";
import {
    ChevronDown,
    Search,
    HelpCircle,
    CheckCircle,
    Star,
    Clock,
    Shield,
    Award,
    Users,
    BookOpen,
    GraduationCap,
    MessageSquare,
    Phone,
    Mail,
    Globe,
    FileText,
    DollarSign,
    Calendar,
    Zap,
    Lock,
    RefreshCw,
} from "lucide-react";

export default function Faqs({ fadeIn, staggerContainer }) {
    const [activeFaq, setActiveFaq] = useState(null);
    const [activeCategory, setActiveCategory] = useState(0);
    const [searchTerm, setSearchTerm] = useState("");

    const handleFaqClick = (faqIndex, question) => {
        setActiveFaq(activeFaq === faqIndex ? null : faqIndex);
        if (typeof window.trackFAQInteraction === "function") {
            window.trackFAQInteraction(
                question,
                activeFaq === faqIndex ? "close" : "open"
            );
        }
    };

    const handleCategoryClick = (categoryIndex, categoryName) => {
        setActiveCategory(categoryIndex);
        setActiveFaq(null);
        if (typeof window.trackFAQCategory === "function") {
            window.trackFAQCategory(categoryName, "faq_section");
        }
    };

    const handleCTAClick = (ctaText, section) => {
        if (typeof window.trackCTAClick === "function") {
            window.trackCTAClick(ctaText, "faq_section", section);
        }
    };

    const faqCategories = [
        {
            name: "General Services",
            icon: <HelpCircle className="w-5 h-5" />,
            description: "Common questions about our academic writing services",
            faqs: [
                {
                    question:
                        "What types of academic writing services do you offer?",
                    answer: "We provide comprehensive academic writing services including dissertation writing help, research paper writing service, custom academic papers, essays, case studies, literature reviews, and thesis writing. As a trusted academic writing company, our professional essay writers specialize in all academic levels from undergraduate to PhD, covering over 35 academic disciplines including STEM, humanities, business, social sciences, and health sciences. We are one of the leading websites that write assignments and essays for students online.",
                    keywords: [
                        "academic writing services",
                        "academic writing company",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                        "websites that write assignments and essays for students online",
                        "online assignment writing services for college students",
                        "professional essay writing websites for university students",
                    ],
                    category: "services",
                },
                {
                    question:
                        "How do you ensure the quality of your academic writing services?",
                    answer: "Our quality assurance process involves multiple stages: First, we match your project with professional essay writers who have relevant expertise in your field. Then, every custom academic paper goes through rigorous editing, proofreading, and plagiarism checking. Our dissertation writing help includes methodology review, and our research paper writing service ensures proper citation and formatting. Finally, all work is reviewed by our quality control team before delivery.",
                    keywords: [
                        "academic writing services",
                        "professional essay writers",
                        "custom academic papers",
                        "dissertation writing help",
                        "research paper writing service",
                    ],
                    category: "quality",
                },
                {
                    question:
                        "What qualifications do your professional essay writers have?",
                    answer: "Our team consists of 180+ professional essay writers with advanced degrees. 67% hold PhD degrees from accredited universities, while 33% have Master's degrees in their respective fields. All writers providing dissertation writing help have doctoral-level experience, and those handling research paper writing service have published academic work. Our custom academic papers are written by experts with an average of 8+ years in academic writing services.",
                    keywords: [
                        "professional essay writers",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "academic writing services",
                    ],
                    category: "writers",
                },
                {
                    question:
                        "How do you match writers to academic writing projects?",
                    answer: "We use a sophisticated matching system that considers your academic level, subject area, deadline, and specific requirements. For dissertation writing help, we assign PhD-level writers with expertise in your research area. Research paper writing service projects are matched with writers who have published in similar fields. All custom academic papers are assigned to professional essay writers with relevant academic backgrounds and proven track records in academic writing services.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "matching",
                },
            ],
        },
        {
            name: "Quality & Plagiarism",
            icon: <Shield className="w-5 h-5" />,
            description:
                "Questions about originality, quality control, and academic integrity",
            faqs: [
                {
                    question:
                        "How do you guarantee original, plagiarism-free academic writing?",
                    answer: "We maintain strict academic integrity standards across all our academic writing services. Every custom academic paper is written from scratch by our professional essay writers. Our dissertation writing help includes original research and analysis, while our research paper writing service ensures proper citation of all sources. We use advanced plagiarism detection software and provide free plagiarism reports with every order. All work is guaranteed 100% original.",
                    keywords: [
                        "academic writing services",
                        "custom academic papers",
                        "professional essay writers",
                        "dissertation writing help",
                        "research paper writing service",
                    ],
                    category: "originality",
                },
                {
                    question:
                        "What is your revision policy for academic writing services?",
                    answer: "We offer unlimited free revisions within 14 days of delivery for all our academic writing services. Whether you need adjustments to dissertation writing help, research paper writing service, or custom academic papers, our professional essay writers will make necessary changes at no extra cost. Revisions are processed within 24-48 hours, and we ensure all your requirements are met to your complete satisfaction.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "revisions",
                },
                {
                    question:
                        "How do you handle citation styles and formatting?",
                    answer: "Our professional essay writers are experts in all major citation styles including APA, MLA, Chicago, Harvard, and IEEE. Whether you need dissertation writing help with complex academic formatting or research paper writing service with specific journal requirements, we ensure perfect compliance. All custom academic papers include proper in-text citations, reference lists, and formatting according to your institution's guidelines. Our academic writing services cover all formatting requirements.",
                    keywords: [
                        "professional essay writers",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "academic writing services",
                    ],
                    category: "formatting",
                },
                {
                    question:
                        "What quality control measures do you have in place?",
                    answer: "Our academic writing services include a comprehensive quality control process: 1) Expert writer assignment based on qualifications, 2) Research and writing by professional essay writers, 3) Internal review and editing, 4) Plagiarism checking, 5) Formatting verification, 6) Final quality assessment. For dissertation writing help and research paper writing service, we include additional methodology and literature review checks. All custom academic papers undergo this rigorous process.",
                    keywords: [
                        "academic writing services",
                        "professional essay writers",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                    ],
                    category: "quality_control",
                },
            ],
        },
        {
            name: "Pricing & Payment",
            icon: <DollarSign className="w-5 h-5" />,
            description:
                "Information about pricing, payment methods, and refund policies",
            faqs: [
                {
                    question:
                        "How is pricing determined for your academic writing services?",
                    answer: "Our pricing for academic writing services is based on several factors: academic level (undergraduate, master's, PhD), deadline urgency, page count, and complexity. Dissertation writing help typically costs more due to extensive research requirements, while standard custom academic papers have competitive rates. Research paper writing service pricing varies by subject complexity. Our professional essay writers' expertise is reflected in fair, transparent pricing with no hidden fees.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "custom academic papers",
                        "research paper writing service",
                        "professional essay writers",
                    ],
                    category: "pricing",
                },
                {
                    question: "What payment methods do you accept?",
                    answer: "We accept all major payment methods for our academic writing services including credit cards (Visa, MasterCard, American Express), PayPal, and bank transfers. All payments are processed through secure, encrypted systems. Whether you're ordering dissertation writing help, research paper writing service, or custom academic papers, your financial information is completely protected. We also offer flexible payment plans for larger projects.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                    ],
                    category: "payment_methods",
                },
                {
                    question:
                        "Do you offer discounts for academic writing services?",
                    answer: "Yes, we provide various discounts for our academic writing services. First-time customers receive welcome discounts, returning clients get loyalty bonuses, and bulk orders qualify for volume discounts. Students ordering dissertation writing help or multiple custom academic papers can benefit from package deals. Our research paper writing service includes seasonal promotions. Professional essay writers' premium services are available at competitive rates with regular discount opportunities.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "custom academic papers",
                        "research paper writing service",
                        "professional essay writers",
                    ],
                    category: "discounts",
                },
                {
                    question: "What is your refund policy?",
                    answer: "We offer a comprehensive money-back guarantee for all academic writing services. If you're not satisfied with dissertation writing help, research paper writing service, or custom academic papers, you can request a full refund within 14 days. Our professional essay writers strive for 100% satisfaction, but if we fail to meet your requirements despite unlimited revisions, we'll provide a complete refund. Partial refunds are available for late deliveries.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "refunds",
                },
            ],
        },
        {
            name: "Deadlines & Delivery",
            icon: <Clock className="w-5 h-5" />,
            description:
                "Questions about turnaround times, urgent orders, and delivery",
            faqs: [
                {
                    question:
                        "What are your typical turnaround times for academic writing services?",
                    answer: "Our academic writing services offer flexible deadlines to meet your needs. Standard custom academic papers: 3-14 days, Research paper writing service: 5-21 days depending on length and complexity, Dissertation writing help: 2-8 weeks based on chapters and requirements. Our professional essay writers can handle urgent orders with 24-48 hour delivery for shorter assignments. Rush orders are available for most academic writing services with premium pricing.",
                    keywords: [
                        "academic writing services",
                        "custom academic papers",
                        "research paper writing service",
                        "dissertation writing help",
                        "professional essay writers",
                    ],
                    category: "turnaround",
                },
                {
                    question:
                        "Can you handle urgent academic writing requests?",
                    answer: "Yes, our academic writing services include urgent delivery options. Our professional essay writers can complete custom academic papers within 24-48 hours for essays and short assignments. Research paper writing service offers 3-5 day rush delivery, while dissertation writing help can be expedited with dedicated writer teams. We maintain quality standards even for urgent academic writing services, though rush orders include premium pricing.",
                    keywords: [
                        "academic writing services",
                        "professional essay writers",
                        "custom academic papers",
                        "research paper writing service",
                        "dissertation writing help",
                    ],
                    category: "urgent",
                },
                {
                    question: "How do you ensure on-time delivery?",
                    answer: "Our academic writing services guarantee on-time delivery through careful project management. Professional essay writers receive clear deadlines with buffer time built in. For dissertation writing help and research paper writing service, we use milestone-based delivery systems. Custom academic papers are tracked throughout the writing process. If any delays occur, we notify you immediately and offer compensation. Our 99.2% on-time delivery rate reflects our commitment to punctuality.",
                    keywords: [
                        "academic writing services",
                        "professional essay writers",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                    ],
                    category: "delivery",
                },
                {
                    question: "What happens if my order is delivered late?",
                    answer: "Late delivery is rare in our academic writing services (less than 1% of orders), but when it occurs, we take full responsibility. You'll receive automatic compensation, partial refunds, or complete refunds depending on the delay length. Our professional essay writers and project managers work to prevent delays in dissertation writing help, research paper writing service, and custom academic papers. We also offer expedited completion at no extra charge for delayed orders.",
                    keywords: [
                        "academic writing services",
                        "professional essay writers",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                    ],
                    category: "late_delivery",
                },
            ],
        },
        {
            name: "Communication & Support",
            icon: <MessageSquare className="w-5 h-5" />,
            description:
                "Questions about customer support, writer communication, and assistance",
            faqs: [
                {
                    question: "How can I communicate with my assigned writer?",
                    answer: "Our academic writing services include direct communication with your professional essay writers through our secure messaging system. Whether you're receiving dissertation writing help, research paper writing service, or custom academic papers, you can message your writer 24/7. Share additional requirements, ask questions, or request updates. All communication is confidential and helps ensure your academic writing services meet your exact specifications.",
                    keywords: [
                        "academic writing services",
                        "professional essay writers",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                    ],
                    category: "writer_communication",
                },
                {
                    question: "What customer support do you provide?",
                    answer: "We offer 24/7 customer support for all academic writing services. Our support team assists with ordering dissertation writing help, tracking research paper writing service progress, and resolving any issues with custom academic papers. Contact us via live chat, email, or phone. Our professional essay writers are supported by dedicated project managers who ensure smooth communication and timely delivery of all academic writing services.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "customer_support",
                },
                {
                    question: "Can I request progress updates on my order?",
                    answer: "Absolutely! Our academic writing services include regular progress updates. For dissertation writing help, you'll receive chapter-by-chapter updates. Research paper writing service includes milestone reports, and custom academic papers come with draft previews. Professional essay writers provide status updates, and you can request additional updates anytime. Our transparent process ensures you're always informed about your academic writing services progress.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "progress_updates",
                },
                {
                    question: "What if I need to make changes to my order?",
                    answer: "Our academic writing services allow order modifications within reasonable timeframes. Contact our support team to discuss changes to dissertation writing help, research paper writing service, or custom academic papers. Minor adjustments are usually free, while major changes may affect pricing and deadlines. Professional essay writers can accommodate most modifications, and we'll always inform you of any implications before proceeding with changes to your academic writing services.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "order_changes",
                },
            ],
        },
        {
            name: "Confidentiality & Security",
            icon: <Lock className="w-5 h-5" />,
            description:
                "Questions about privacy, data security, and confidentiality",
            faqs: [
                {
                    question: "How do you protect my personal information?",
                    answer: "We maintain strict confidentiality for all academic writing services. Your personal information is protected by advanced encryption and secure servers. Professional essay writers never see your personal details beyond what's necessary for dissertation writing help, research paper writing service, or custom academic papers. We never share client information with third parties. Your privacy is guaranteed throughout all academic writing services interactions.",
                    keywords: [
                        "academic writing services",
                        "professional essay writers",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                    ],
                    category: "privacy",
                },
                {
                    question:
                        "Is using your academic writing services confidential?",
                    answer: "Yes, complete confidentiality is guaranteed for all our academic writing services. We never disclose that you've used dissertation writing help, research paper writing service, or custom academic papers. Professional essay writers sign strict non-disclosure agreements. Your academic institution will never know you've used our academic writing services. We maintain anonymity throughout the entire process and beyond completion.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "confidentiality",
                },
                {
                    question: "How secure are your payment systems?",
                    answer: "Our academic writing services use bank-level security for all transactions. SSL encryption protects your payment information when ordering dissertation writing help, research paper writing service, or custom academic papers. We're PCI DSS compliant and never store credit card details. Professional essay writers never have access to your financial information. All payments for academic writing services are processed through secure, verified payment gateways.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "payment_security",
                },
                {
                    question: "What data do you collect and why?",
                    answer: "We collect minimal data necessary to provide academic writing services effectively. This includes contact information, academic requirements, and project specifications for dissertation writing help, research paper writing service, and custom academic papers. Professional essay writers receive only assignment-related information. We don't collect unnecessary personal data, and all information is used solely to deliver quality academic writing services and customer support.",
                    keywords: [
                        "academic writing services",
                        "dissertation writing help",
                        "research paper writing service",
                        "custom academic papers",
                        "professional essay writers",
                    ],
                    category: "data_collection",
                },
            ],
        },
    ];

    const allFaqs = faqCategories.flatMap((category) =>
        category.faqs.map((faq) => ({ ...faq, categoryName: category.name }))
    );

    const filteredFaqs = searchTerm
        ? allFaqs.filter(
              (faq) =>
                  faq.question
                      .toLowerCase()
                      .includes(searchTerm.toLowerCase()) ||
                  faq.answer.toLowerCase().includes(searchTerm.toLowerCase()) ||
                  faq.keywords.some((keyword) =>
                      keyword.toLowerCase().includes(searchTerm.toLowerCase())
                  )
          )
        : faqCategories[activeCategory].faqs;

    const faqStats = {
        totalQuestions: allFaqs.length,
        categoriesCount: faqCategories.length,
        avgResponseTime: "< 2 minutes",
        satisfactionRate: "98%",
    };

    return (
        <>
            {/* Main FAQ Section */}
            <section
                id="faq"
                className="py-20 bg-gradient-to-br from-gray-50 to-blue-50 dark:from-gray-900 dark:to-gray-800 w-full"
            >
                <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Section Header */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl md:text-5xl font-bold text-gray-900 dark:text-white mb-6">
                            Frequently Asked Questions About Our Academic
                            Writing Services
                        </h2>
                        <div className="mt-4 h-1 w-24 bg-gradient-to-r from-blue-600 to-purple-600 mx-auto rounded-full"></div>
                        <p className="mt-6 max-w-4xl mx-auto text-xl text-gray-600 dark:text-gray-300 leading-relaxed">
                            Get answers to common questions about our{" "}
                            <strong>professional essay writers</strong>,
                            <strong> dissertation writing help</strong>,{" "}
                            <strong>research paper writing service</strong>, and{" "}
                            <strong>custom academic papers</strong>. As one of the leading{" "}
                            <strong>online assignment writing services for college students</strong> and{" "}
                            <strong>professional essay writing websites for university students</strong>, our
                            comprehensive{" "}
                            <strong>academic writing services</strong>
                            are designed to support your academic success.
                        </p>
                    </motion.div>

                    {/* FAQ Statistics */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-16"
                    >
                        <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl p-8 border border-gray-200 dark:border-gray-700">
                            <h3 className="text-2xl font-bold text-gray-900 dark:text-white text-center mb-8">
                                Academic Writing Services - Support Overview
                            </h3>
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <HelpCircle className="w-12 h-12 text-blue-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {faqStats.totalQuestions}+
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Questions Answered
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <BookOpen className="w-12 h-12 text-green-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {faqStats.categoriesCount}
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Topic Categories
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Clock className="w-12 h-12 text-purple-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {faqStats.avgResponseTime}
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Response Time
                                    </div>
                                </div>
                                <div className="text-center">
                                    <div className="flex justify-center mb-3">
                                        <Star className="w-12 h-12 text-yellow-500" />
                                    </div>
                                    <div className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                                        {faqStats.satisfactionRate}
                                    </div>
                                    <div className="text-sm text-gray-600 dark:text-gray-300">
                                        Help Satisfaction
                                    </div>
                                </div>
                            </div>
                        </div>
                    </motion.div>

                    {/* Search Bar */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="mb-12"
                    >
                        <div className="max-w-2xl mx-auto">
                            <div className="relative">
                                <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                                <input
                                    type="text"
                                    placeholder="Search FAQs about academic writing services, dissertation help, research papers..."
                                    className="w-full pl-12 pr-4 py-4 border border-gray-300 dark:border-gray-600 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-800 dark:text-white text-lg"
                                    value={searchTerm}
                                    onChange={(e) =>
                                        setSearchTerm(e.target.value)
                                    }
                                />
                            </div>
                            {searchTerm && (
                                <div className="mt-4 text-center">
                                    <span className="text-gray-600 dark:text-gray-300">
                                        Found {filteredFaqs.length} results for
                                        "{searchTerm}"
                                    </span>
                                    <button
                                        onClick={() => setSearchTerm("")}
                                        className="ml-4 text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300"
                                    >
                                        Clear search
                                    </button>
                                </div>
                            )}
                        </div>
                    </motion.div>

                    {/* Category Tabs */}
                    {!searchTerm && (
                        <motion.div
                            initial="hidden"
                            whileInView="visible"
                            viewport={{ once: true }}
                            variants={fadeIn}
                            className="mb-12"
                        >
                            <div className="flex flex-wrap justify-center gap-4">
                                {faqCategories.map((category, idx) => (
                                    <button
                                        key={idx}
                                        className={`flex items-center px-6 py-3 rounded-xl font-medium transition-all duration-300 ${
                                            activeCategory === idx
                                                ? "bg-gradient-to-r from-blue-600 to-purple-600 text-white shadow-lg transform scale-105"
                                                : "bg-white dark:bg-gray-800 text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:shadow-md border border-gray-200 dark:border-gray-700"
                                        }`}
                                        onClick={() =>
                                            handleCategoryClick(
                                                idx,
                                                category.name
                                            )
                                        }
                                    >
                                        {category.icon}
                                        <span className="ml-2">
                                            {category.name}
                                        </span>
                                        <span className="ml-2 text-sm opacity-75">
                                            ({category.faqs.length})
                                        </span>
                                    </button>
                                ))}
                            </div>

                            {/* Category Description */}
                            <div className="text-center mt-6">
                                <p className="text-gray-600 dark:text-gray-300 max-w-2xl mx-auto">
                                    {faqCategories[activeCategory].description}
                                </p>
                            </div>
                        </motion.div>
                    )}

                    {/* FAQ Items */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={staggerContainer}
                        className="space-y-4 mb-16"
                    >
                        {filteredFaqs.map((faq, index) => (
                            <motion.div
                                key={index}
                                variants={fadeIn}
                                className="bg-white dark:bg-gray-800 rounded-xl shadow-lg hover:shadow-xl transition-all duration-300 border border-gray-200 dark:border-gray-700 overflow-hidden"
                            >
                                <button
                                    className="flex justify-between items-center w-full p-6 text-left hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors duration-200"
                                    onClick={() =>
                                        handleFaqClick(index, faq.question)
                                    }
                                >
                                    <div className="flex-1">
                                        <h3 className="font-semibold text-lg text-gray-900 dark:text-white pr-4">
                                            {faq.question}
                                        </h3>
                                        {searchTerm && (
                                            <div className="mt-2">
                                                <span className="text-sm text-blue-600 dark:text-blue-400 bg-blue-50 dark:bg-blue-900/30 px-2 py-1 rounded">
                                                    {faq.categoryName}
                                                </span>
                                            </div>
                                        )}
                                    </div>
                                    <ChevronDown
                                        className={`w-6 h-6 text-gray-500 dark:text-gray-400 transition-transform duration-300 flex-shrink-0 ${
                                            activeFaq === index
                                                ? "transform rotate-180"
                                                : ""
                                        }`}
                                    />
                                </button>

                                {activeFaq === index && (
                                    <motion.div
                                        initial={{ opacity: 0, height: 0 }}
                                        animate={{ opacity: 1, height: "auto" }}
                                        exit={{ opacity: 0, height: 0 }}
                                        transition={{ duration: 0.3 }}
                                        className="px-6 pb-6"
                                    >
                                        <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
                                            <div className="prose prose-blue dark:prose-invert max-w-none">
                                                <p className="text-gray-600 dark:text-gray-300 leading-relaxed">
                                                    {faq.answer}
                                                </p>
                                            </div>

                                            {/* Keywords Tags */}
                                            <div className="mt-4 flex flex-wrap gap-2">
                                                {faq.keywords.map(
                                                    (keyword, keyIndex) => (
                                                        <span
                                                            key={keyIndex}
                                                            className="text-xs bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300 px-2 py-1 rounded-full"
                                                        >
                                                            {keyword}
                                                        </span>
                                                    )
                                                )}
                                            </div>
                                        </div>
                                    </motion.div>
                                )}
                            </motion.div>
                        ))}
                    </motion.div>

                    {/* Still Have Questions Section */}
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center"
                    >
                        <div className="bg-white dark:bg-gray-800 rounded-2xl p-8 md:p-12 shadow-xl border border-gray-200 dark:border-gray-700">
                            <h3 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-6">
                                Still Have Questions About Our Academic Writing
                                Services?
                            </h3>
                            <p className="text-xl text-gray-600 dark:text-gray-300 mb-8 max-w-3xl mx-auto leading-relaxed">
                                Can't find what you're looking for? Our{" "}
                                <strong>professional essay writers</strong> and
                                support team are here to help with any questions
                                about <strong>dissertation writing help</strong>
                                ,
                                <strong> research paper writing service</strong>
                                , or <strong>custom academic papers</strong>.
                            </p>

                            <div className="flex flex-col sm:flex-row justify-center space-y-4 sm:space-y-0 sm:space-x-6 mb-8">
                                <Link
                                    href="/contact"
                                    className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white px-8 py-4 rounded-lg font-bold transition-all duration-300 transform hover:scale-105 shadow-lg"
                                    onClick={() =>
                                        handleCTAClick(
                                            "Contact Support",
                                            "main_cta"
                                        )
                                    }
                                >
                                    Contact Our Support Team
                                </Link>
                                <Link
                                    href="/place-order"
                                    className="border-2 border-blue-600 text-blue-600 dark:text-blue-400 hover:bg-blue-600 hover:text-white dark:hover:bg-blue-600 dark:hover:text-white px-8 py-4 rounded-lg font-bold transition-all duration-300 transform hover:scale-105"
                                    onClick={() =>
                                        handleCTAClick(
                                            "Get Started",
                                            "order_cta"
                                        )
                                    }
                                >
                                    Get Started Today
                                </Link>
                            </div>

                            {/* Contact Methods */}
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-6 text-sm text-gray-600 dark:text-gray-300">
                                <div className="flex items-center justify-center">
                                    <MessageSquare className="w-4 h-4 mr-2 text-blue-500" />
                                    <span>Live Chat: Available 24/7</span>
                                </div>
                                <div className="flex items-center justify-center">
                                    <Phone className="w-4 h-4 mr-2 text-green-500" />
                                    <span>Phone: +1 (555) 123-4567</span>
                                </div>
                                <div className="flex items-center justify-center">
                                    <Mail className="w-4 h-4 mr-2 text-purple-500" />
                                    <span>
                                        Email: support@academicscribe.com
                                    </span>
                                </div>
                            </div>
                        </div>
                    </motion.div>
                </div>
            </section>
        </>
    );
}
