import { useForm, usePage } from "@inertiajs/react";
import { Download, Trash, User, Clock } from "lucide-react";

export default function FileManagement({ files, isAdmin }) {
    const { auth } = usePage().props;
    const { delete: destroy } = useForm();

    // Function to truncate text with ellipsis
    const truncateFileName = (fileName, maxLength = 30) => {
        if (fileName.length <= maxLength) return fileName;
        return fileName.substring(0, maxLength) + "...";
    };

    // Function to format date
    const formatUploadDate = (dateString) => {
        if (!dateString) return "Unknown";
        const date = new Date(dateString);
        return date.toLocaleString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const handleDownload = async (fileId) => {
        try {
            // Add credentials and headers for proper authentication
            const response = await window.fetch(
                route("files.download", fileId),
                {
                    method: "GET",
                    credentials: "same-origin",
                    headers: {
                        "X-Requested-With": "XMLHttpRequest",
                        Accept: "application/octet-stream",
                    },
                }
            );

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            // Get the filename from the response headers
            const contentDisposition = response.headers.get(
                "content-disposition"
            );
            let filename;
            if (contentDisposition) {
                const matches = /filename[^;=\n]*=((['"]).*?\2|[^;\n]*)/.exec(
                    contentDisposition
                );
                if (matches != null && matches[1]) {
                    filename = matches[1].replace(/['"]/g, "");
                }
            }

            // If no filename in headers, use a default one
            filename = filename || "download";

            // Create blob and trigger download
            const blob = await response.blob();
            const downloadUrl = window.URL.createObjectURL(blob);
            const a = document.createElement("a");
            a.style.display = "none";
            a.href = downloadUrl;
            a.download = decodeURIComponent(filename);
            document.body.appendChild(a);
            a.click();

            // Cleanup
            window.URL.revokeObjectURL(downloadUrl);
            document.body.removeChild(a);
        } catch (error) {
            console.error("Download failed:", error);
            // You might want to add a toast notification here
            alert("Failed to download file. Please try again.");
        }
    };

    const handleDelete = (fileId) => {
        if (confirm("Are you sure you want to delete this file?")) {
            destroy(route("files.destroy", fileId), {
                preserveScroll: true,
            });
        }
    };

    return (
        <div className="space-y-4">
            {files && files.length > 0 ? (
                files.map((file) => (
                    <div
                        key={file.id}
                        className="flex flex-col sm:flex-row sm:items-center p-4 border dark:border-gray-600 rounded-lg gap-3 bg-white dark:bg-gray-800 shadow-sm"
                    >
                        <div className="flex-1 min-w-0 overflow-hidden">
                            <p
                                className="font-medium text-gray-900 dark:text-gray-100 text-lg mb-2"
                                title={file.file_name} // Show full name on hover
                            >
                                {truncateFileName(file.file_name, 25)}
                            </p>
                            
                            {/* Upload Information Section */}
                            <div className="space-y-2">
                                {/* Uploader Information */}
                                <div className="flex items-center gap-2 text-sm">
                                    <User className="h-4 w-4 text-gray-500 dark:text-gray-400" />
                                    <span className="text-gray-600 dark:text-gray-300 font-medium">
                                        Uploaded by: 
                                    </span>
                                    <span className="text-gray-900 dark:text-gray-100 font-semibold">
                                        {file.uploader_name || 'Unknown User'}
                                    </span>
                                    <span className="text-gray-500 dark:text-gray-400">
                                        ({file.uploader_user_type || 'Unknown Type'})
                                    </span>
                                </div>
                                
                                {/* Upload Time Information */}
                                <div className="flex items-center gap-2 text-sm">
                                    <Clock className="h-4 w-4 text-gray-500 dark:text-gray-400" />
                                    <span className="text-gray-600 dark:text-gray-300 font-medium">
                                        Uploaded on: 
                                    </span>
                                    <span className="text-gray-900 dark:text-gray-100 font-semibold">
                                        {formatUploadDate(file.date_uploaded)}
                                    </span>
                                </div>
                                
                                {/* File Size Information */}
                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                    Size: {(file.file_size / 1024 / 1024).toFixed(2)} MB
                                </div>
                            </div>
                        </div>
                        
                        <div className="flex flex-wrap gap-2 mt-2 sm:mt-0">
                            <button
                                onClick={() => handleDownload(file.id)}
                                className="flex items-center px-3 py-2 bg-indigo-500 text-white rounded-md hover:bg-indigo-600 transition-colors"
                                aria-label="Download file"
                            >
                                <Download className="h-4 w-4 mr-2" />
                                <span className="sm:hidden">Download</span>
                            </button>
                            {(file.uploader_id === auth.user.id || isAdmin) && (
                                <button
                                    onClick={() => handleDelete(file.id)}
                                    className="flex items-center px-3 py-2 bg-red-500 text-white rounded-md hover:bg-red-600 transition-colors"
                                    aria-label="Delete file"
                                >
                                    <Trash className="h-4 w-4 mr-2" />
                                    Delete
                                </button>
                            )}
                        </div>
                    </div>
                ))
            ) : (
                <p className="text-center text-gray-500 dark:text-gray-400">
                    No files attached to this order
                </p>
            )}
        </div>
    );
}
