import React from 'react';

export default function ContactSupport({ title, description, contacts, notice }) {
    if (!contacts || contacts.length === 0) return null;

    const icons = {
        phone: (
            <svg className="w-8 h-8 text-green-600" fill="currentColor" viewBox="0 0 20 20">
                <path d="M2 3a1 1 0 011-1h2.153a1 1 0 01.986.836l.74 4.435a1 1 0 01-.54 1.06l-1.548.773a11.037 11.037 0 006.105 6.105l.774-1.548a1 1 0 011.059-.54l4.435.74a1 1 0 01.836.986V17a1 1 0 01-1 1h-2C7.82 18 2 12.18 2 5V3z" />
            </svg>
        ),
        email: (
            <svg className="w-8 h-8 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                <path d="M2.003 5.884L10 9.882l7.997-3.998A2 2 0 0016 4H4a2 2 0 00-1.997 1.884z" />
                <path d="M18 8.118l-8 4-8-4V14a2 2 0 002 2h12a2 2 0 002-2V8.118z" />
            </svg>
        ),
        chat: (
            <svg className="w-8 h-8 text-purple-600" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M18 10c0 3.866-3.582 7-8 7a8.841 8.841 0 01-4.083-.98L2 17l1.338-3.123C2.493 12.767 2 11.434 2 10c0-3.866 3.582-7 8-7s8 3.134 8 7zM7 9H5v2h2V9zm8 0h-2v2h2V9zM9 9h2v2H9V9z" clipRule="evenodd" />
            </svg>
        ),
    };

    const colorSchemes = {
        0: { bg: 'bg-green-100', icon: 'phone', color: 'text-green-600 hover:text-green-800' },
        1: { bg: 'bg-blue-100', icon: 'email', color: 'text-blue-600 hover:text-blue-800' },
        2: { bg: 'bg-purple-100', icon: 'chat', color: 'text-purple-600 hover:text-purple-800' },
    };

    return (
        <section className="py-16 bg-gray-50">
            <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                {title && (
                    <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
                        {title}
                    </h2>
                )}
                {description && (
                    <div 
                        className="text-xl text-gray-600 mb-8 content-area"
                        dangerouslySetInnerHTML={{ __html: description }}
                    />
                )}

                <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-8">
                    {contacts.map((contact, index) => {
                        const colors = colorSchemes[index % 3];
                        return (
                            <div key={index} className="text-center">
                                <div className={`${colors.bg} rounded-full w-16 h-16 flex items-center justify-center mx-auto mb-4`}>
                                    {icons[colors.icon]}
                                </div>
                                <h3 className="text-lg font-semibold text-gray-900 mb-2">
                                    {contact.title}
                                </h3>
                                <div 
                                    className="text-gray-600 mb-2 content-area"
                                    dangerouslySetInnerHTML={{ __html: contact.description }}
                                />
                                {contact.link && contact.value && (
                                    contact.type === 'button' ? (
                                        <button className={`${colors.color} font-semibold`}>
                                            {contact.value}
                                        </button>
                                    ) : (
                                        <a
                                            href={contact.link}
                                            className={`${colors.color} font-semibold`}
                                        >
                                            {contact.value}
                                        </a>
                                    )
                                )}
                            </div>
                        );
                    })}
                </div>

                {notice && (
                    <div className="bg-green-50 rounded-lg p-6">
                        <h3 className="text-lg font-semibold text-gray-900 mb-2">
                            {notice.icon} {notice.title}
                        </h3>
                        <div 
                            className="text-gray-600 content-area"
                            dangerouslySetInnerHTML={{ __html: notice.description }}
                        />
                    </div>
                )}
            </div>
        </section>
    );
}





