import React from 'react';
import { TrendingUp, TrendingDown } from 'lucide-react';
import PropTypes from 'prop-types';

/**
 * KPICard Component
 * Clean, modern KPI card with large corner icon and trend indicator
 */
const KPICard = ({ 
    label, 
    value, 
    trend,
    icon: Icon, 
    colorScheme = 'blue'
}) => {
    // Color schemes
    const colorSchemes = {
        green: {
            bg: 'bg-gradient-to-br from-green-500 to-green-600',
            iconColor: 'text-green-100',
            trendPositive: 'text-green-100',
            trendNegative: 'text-red-100',
        },
        purple: {
            bg: 'bg-gradient-to-br from-purple-500 to-purple-600',
            iconColor: 'text-purple-100',
            trendPositive: 'text-purple-100',
            trendNegative: 'text-red-100',
        },
        blue: {
            bg: 'bg-gradient-to-br from-blue-500 to-blue-600',
            iconColor: 'text-blue-100',
            trendPositive: 'text-blue-100',
            trendNegative: 'text-red-100',
        },
        orange: {
            bg: 'bg-gradient-to-br from-orange-500 to-red-600',
            iconColor: 'text-orange-100',
            trendPositive: 'text-orange-100',
            trendNegative: 'text-red-200',
        },
        cyan: {
            bg: 'bg-gradient-to-br from-cyan-500 to-cyan-600',
            iconColor: 'text-cyan-100',
            trendPositive: 'text-cyan-100',
            trendNegative: 'text-red-100',
        },
        red: {
            bg: 'bg-gradient-to-br from-red-500 to-red-600',
            iconColor: 'text-red-100',
            trendPositive: 'text-green-200',
            trendNegative: 'text-red-100',
        },
    };

    const scheme = colorSchemes[colorScheme] || colorSchemes.blue;
    const hasTrend = trend !== undefined && trend !== null;
    const isPositive = hasTrend && trend.value > 0;

    return (
        <div className={`
            ${scheme.bg}
            rounded-xl shadow-lg p-6 text-white
            transform transition-all duration-300
            hover:shadow-xl hover:-translate-y-1
            relative overflow-hidden
        `}>
            {/* Large Background Icon */}
            <div className="absolute top-4 right-4">
                {Icon && <Icon className={`h-16 w-16 ${scheme.iconColor} opacity-20`} />}
            </div>

            {/* Content */}
            <div className="relative z-10">
                {/* Label */}
                <div className="text-sm font-medium uppercase tracking-wide opacity-90 mb-2">
                    {label}
                </div>

                {/* Value */}
                <div className="text-3xl font-bold mb-3">
                    {value}
                </div>

                {/* Trend Indicator */}
                {hasTrend && (
                    <div className="flex items-center gap-1">
                        {isPositive ? (
                            <TrendingUp className={`h-4 w-4 ${scheme.trendPositive}`} />
                        ) : (
                            <TrendingDown className={`h-4 w-4 ${scheme.trendNegative}`} />
                        )}
                        <span className={`text-sm font-medium ${isPositive ? scheme.trendPositive : scheme.trendNegative}`}>
                            {isPositive ? '+' : ''}{trend.value}%
                        </span>
                        {trend.label && (
                            <span className="text-xs opacity-75 ml-1">
                                {trend.label}
                            </span>
                        )}
                    </div>
                )}
            </div>
        </div>
    );
};

KPICard.propTypes = {
    label: PropTypes.string.isRequired,
    value: PropTypes.oneOfType([PropTypes.string, PropTypes.number]).isRequired,
    trend: PropTypes.shape({
        value: PropTypes.number,
        label: PropTypes.string,
    }),
    icon: PropTypes.elementType,
    colorScheme: PropTypes.oneOf(['green', 'purple', 'blue', 'orange', 'cyan', 'red']),
};

export default KPICard;




