import React, { useState } from "react";
import { Archive, CheckCircle, Square, CheckSquare, X } from "lucide-react";
import axios from "axios";

export default function BulkActions({ 
    conversations, 
    isAdmin, 
    onBulkComplete,
    selectedConversations,
    setSelectedConversations 
}) {
    const [isProcessing, setIsProcessing] = useState(false);

    // Don't show bulk actions for non-admins or if no conversations
    if (!isAdmin || !conversations || conversations.length === 0) {
        return null;
    }

    const toggleSelectAll = () => {
        if (selectedConversations.length === conversations.length) {
            setSelectedConversations([]);
        } else {
            setSelectedConversations(conversations.map(c => c.id));
        }
    };

    const toggleSelectConversation = (conversationId) => {
        setSelectedConversations(prev => 
            prev.includes(conversationId)
                ? prev.filter(id => id !== conversationId)
                : [...prev, conversationId]
        );
    };

    const handleBulkAction = async (action) => {
        if (selectedConversations.length === 0) {
            alert('Please select conversations first');
            return;
        }

        const actionName = action === 'archive' ? 'archive' : 'resolve';
        const confirmed = confirm(`Are you sure you want to ${actionName} ${selectedConversations.length} conversation(s)?`);
        
        if (!confirmed) return;

        setIsProcessing(true);
        
        try {
            const endpoint = action === 'archive' 
                ? '/messaging/conversations/bulk-archive'
                : '/messaging/conversations/bulk-resolve';
                
            const response = await axios.post(endpoint, {
                conversation_ids: selectedConversations
            });

            alert(response.data.message);
            setSelectedConversations([]);
            onBulkComplete();
            
        } catch (error) {
            console.error(`Error performing bulk ${actionName}:`, error);
            alert(`Failed to ${actionName} conversations`);
        } finally {
            setIsProcessing(false);
        }
    };

    const clearSelection = () => {
        setSelectedConversations([]);
    };

    const allSelected = selectedConversations.length === conversations.length;
    const someSelected = selectedConversations.length > 0;

    return (
        <div className="border-b border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-750">
            <div className="p-3">
                <div className="flex items-center justify-between">
                    {/* Selection Controls */}
                    <div className="flex items-center space-x-3">
                        <button
                            onClick={toggleSelectAll}
                            className="flex items-center space-x-2 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white"
                        >
                            {allSelected ? (
                                <CheckSquare className="w-4 h-4" />
                            ) : (
                                <Square className="w-4 h-4" />
                            )}
                            <span>
                                {allSelected ? 'Deselect All' : 'Select All'}
                            </span>
                        </button>

                        {someSelected && (
                            <span className="text-sm text-gray-600 dark:text-gray-300">
                                {selectedConversations.length} selected
                            </span>
                        )}
                    </div>

                    {/* Bulk Actions */}
                    {someSelected && (
                        <div className="flex items-center space-x-2">
                            <button
                                onClick={() => handleBulkAction('resolve')}
                                disabled={isProcessing}
                                className="flex items-center space-x-1 px-3 py-1 bg-green-100 dark:bg-green-900 text-green-700 dark:text-green-300 rounded-md hover:bg-green-200 dark:hover:bg-green-800 disabled:opacity-50 text-sm"
                            >
                                <CheckCircle className="w-3 h-3" />
                                <span>Resolve</span>
                            </button>

                            <button
                                onClick={() => handleBulkAction('archive')}
                                disabled={isProcessing}
                                className="flex items-center space-x-1 px-3 py-1 bg-gray-100 dark:bg-gray-900 text-gray-700 dark:text-gray-300 rounded-md hover:bg-gray-200 dark:hover:bg-gray-800 disabled:opacity-50 text-sm"
                            >
                                <Archive className="w-3 h-3" />
                                <span>Archive</span>
                            </button>

                            <button
                                onClick={clearSelection}
                                disabled={isProcessing}
                                className="flex items-center space-x-1 px-2 py-1 text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300 text-sm"
                                title="Clear Selection"
                            >
                                <X className="w-3 h-3" />
                            </button>
                        </div>
                    )}
                </div>
            </div>

            {/* Selection Helper */}
            {conversations.map(conversation => (
                <ConversationCheckbox
                    key={conversation.id}
                    conversation={conversation}
                    isSelected={selectedConversations.includes(conversation.id)}
                    onToggle={() => toggleSelectConversation(conversation.id)}
                />
            ))}
        </div>
    );
}

function ConversationCheckbox({ conversation, isSelected, onToggle }) {
    return (
        <div className="absolute left-2 top-1/2 transform -translate-y-1/2 z-10">
            <button
                onClick={(e) => {
                    e.stopPropagation();
                    onToggle();
                }}
                className="p-1 hover:bg-gray-200 dark:hover:bg-gray-600 rounded transition-colors"
            >
                {isSelected ? (
                    <CheckSquare className="w-4 h-4 text-cyan-600" />
                ) : (
                    <Square className="w-4 h-4 text-gray-400" />
                )}
            </button>
        </div>
    );
}