import React, { useState } from "react";
import { Search, Calendar, Filter, Users, CheckCircle, AlertTriangle, Archive, MessageSquare } from "lucide-react";

export default function ConversationFilters({ 
    currentFilter, 
    onFilterChange, 
    onSearchChange, 
    searchTerm, 
    isAdmin,
    conversationStats 
}) {
    const [showDateFilter, setShowDateFilter] = useState(false);
    const [dateRange, setDateRange] = useState({ from: '', to: '' });

    const filterOptions = [
        { 
            key: 'active', 
            label: 'Active', 
            icon: MessageSquare, 
            color: 'text-green-600', 
            count: conversationStats?.active || 0,
            description: 'Ongoing conversations'
        },
        { 
            key: 'resolved', 
            label: 'Resolved', 
            icon: CheckCircle, 
            color: 'text-gray-500', 
            count: conversationStats?.resolved || 0,
            description: 'Recently resolved conversations'
        },
        { 
            key: 'escalated', 
            label: 'Escalated', 
            icon: AlertTriangle, 
            color: 'text-red-500', 
            count: conversationStats?.escalated || 0,
            description: 'Priority conversations',
            adminOnly: true
        },
        { 
            key: 'archived', 
            label: 'Archived', 
            icon: Archive, 
            color: 'text-gray-400', 
            count: conversationStats?.archived || 0,
            description: 'Archived conversations',
            adminOnly: true
        },
        { 
            key: 'all', 
            label: 'All', 
            icon: Users, 
            color: 'text-blue-600', 
            count: conversationStats?.total || 0,
            description: 'All conversations',
            adminOnly: true
        }
    ];

    const visibleFilters = filterOptions.filter(filter => !filter.adminOnly || isAdmin);

    const handleDateRangeChange = (field, value) => {
        const newRange = { ...dateRange, [field]: value };
        setDateRange(newRange);
        
        // Apply filter if both dates are set
        if (newRange.from && newRange.to) {
            onFilterChange(currentFilter, newRange);
        }
    };

    const clearDateFilter = () => {
        setDateRange({ from: '', to: '' });
        onFilterChange(currentFilter, null);
    };

    return (
        <div className="border-b border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
            {/* Search Bar */}
            <div className="p-4">
                <div className="relative">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                    <input
                        type="text"
                        placeholder="Search conversations by name or email..."
                        value={searchTerm}
                        onChange={(e) => onSearchChange(e.target.value)}
                        className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg 
                                 bg-white dark:bg-gray-700 text-gray-900 dark:text-white
                                 focus:ring-2 focus:ring-cyan-500 focus:border-transparent
                                 placeholder-gray-500 dark:placeholder-gray-400"
                    />
                </div>
            </div>

            {/* Filter Tabs */}
            <div className="px-4">
                <div className="flex space-x-1 overflow-x-auto">
                    {visibleFilters.map((filter) => {
                        const Icon = filter.icon;
                        const isActive = currentFilter === filter.key;
                        
                        return (
                            <button
                                key={filter.key}
                                onClick={() => onFilterChange(filter.key)}
                                className={`
                                    flex items-center space-x-2 px-4 py-2 rounded-lg text-sm font-medium
                                    transition-colors whitespace-nowrap
                                    ${isActive 
                                        ? 'bg-cyan-100 dark:bg-cyan-900 text-cyan-700 dark:text-cyan-300 border border-cyan-200 dark:border-cyan-700' 
                                        : 'text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700'
                                    }
                                `}
                                title={filter.description}
                            >
                                <Icon className={`w-4 h-4 ${isActive ? 'text-cyan-600 dark:text-cyan-400' : filter.color}`} />
                                <span>{filter.label}</span>
                                {filter.count > 0 && (
                                    <span className={`
                                        px-2 py-1 rounded-full text-xs font-medium
                                        ${isActive 
                                            ? 'bg-cyan-200 dark:bg-cyan-800 text-cyan-800 dark:text-cyan-200' 
                                            : 'bg-gray-200 dark:bg-gray-600 text-gray-700 dark:text-gray-300'
                                        }
                                    `}>
                                        {filter.count}
                                    </span>
                                )}
                            </button>
                        );
                    })}
                </div>
            </div>

            {/* Advanced Filters (Admin Only) */}
            {isAdmin && (
                <div className="px-4 py-3 border-t border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between">
                        <button
                            onClick={() => setShowDateFilter(!showDateFilter)}
                            className="flex items-center space-x-2 text-sm text-gray-600 dark:text-gray-300 
                                     hover:text-gray-900 dark:hover:text-white transition-colors"
                        >
                            <Calendar className="w-4 h-4" />
                            <span>Date Range</span>
                            <Filter className={`w-3 h-3 transition-transform ${showDateFilter ? 'rotate-180' : ''}`} />
                        </button>

                        {(dateRange.from || dateRange.to) && (
                            <button
                                onClick={clearDateFilter}
                                className="text-xs text-cyan-600 dark:text-cyan-400 hover:text-cyan-700 dark:hover:text-cyan-300"
                            >
                                Clear
                            </button>
                        )}
                    </div>

                    {showDateFilter && (
                        <div className="mt-3 grid grid-cols-2 gap-3">
                            <div>
                                <label className="block text-xs font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    From Date
                                </label>
                                <input
                                    type="date"
                                    value={dateRange.from}
                                    onChange={(e) => handleDateRangeChange('from', e.target.value)}
                                    className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 
                                             rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white
                                             focus:ring-2 focus:ring-cyan-500 focus:border-transparent"
                                />
                            </div>
                            <div>
                                <label className="block text-xs font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    To Date
                                </label>
                                <input
                                    type="date"
                                    value={dateRange.to}
                                    onChange={(e) => handleDateRangeChange('to', e.target.value)}
                                    className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 
                                             rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white
                                             focus:ring-2 focus:ring-cyan-500 focus:border-transparent"
                                />
                            </div>
                        </div>
                    )}
                </div>
            )}
        </div>
    );
}